'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _dump = require('./_dump');

var _dump2 = _interopRequireDefault(_dump);

var _lodash = require('lodash');

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _fs = require('fs');

var _restore_kibi_preprocess = require('./_restore_kibi_preprocess');

var _restore_kibi_preprocess2 = _interopRequireDefault(_restore_kibi_preprocess);

var _requestPromise = require('request-promise');

var _requestPromise2 = _interopRequireDefault(_requestPromise);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * RestoreKibi takes a dump file created by the backup command and adds its objects to the kibi index.
 */
class RestoreKibi {

  /**
   * @param config the investigate.yml config
   * @param backupDir the folder with the backuped indices
   */
  constructor(config, backupDir) {
    this._config = config;
    this._backupDir = backupDir;
    this._dump = new _dump2.default(config, backupDir);

    this.isAccessControPluginEnabled = (0, _lodash.get)(this._config, 'investigate_access_control.enabled');
    this.isACLEnabled = (0, _lodash.get)(this._config, 'investigate_access_control.acl.enabled');
    this.kibiIndex = (0, _lodash.get)(this._config, 'kibana.index', '.siren');
    if (this.isAccessControPluginEnabled && this.isACLEnabled) {
      this.aclIndex = (0, _lodash.get)(this._config, 'investigate_access_control.acl.index', '.sirenaccess');
    }

    this.DEFAULT_ES_URL = 'http://localhost:9220';
    this.SIREN_DEFAULT_ES_USER = 'sirenserver';
    this.SIREN_DEFAULT_ES_USER_PASSWORD = 'password';
  }

  // Returns true if the passed index does exist
  async _doesIndexExist(indexName) {
    const esUrl = (0, _lodash.get)(this._config, 'elasticsearch.url', this.DEFAULT_ES_URL);

    const options = {
      json: true,
      uri: esUrl + '/' + indexName
    };

    if ((0, _lodash.has)(this._config, 'elasticsearch.username') && (0, _lodash.has)(this._config, 'elasticsearch.password')) {
      const [protocol, ...rest] = esUrl.split('//');

      if (protocol === 'https:') {
        if (process.env.NODE_TLS_REJECT_UNAUTHORIZED) {
          this._originalTlsRejectUnauthorizedValue = process.env.NODE_TLS_REJECT_UNAUTHORIZED;
        }
        process.env.NODE_TLS_REJECT_UNAUTHORIZED = 0;
      }

      const username = (0, _lodash.get)(this._config, 'elasticsearch.username', this.SIREN_DEFAULT_ES_USER);
      const password = (0, _lodash.get)(this._config, 'elasticsearch.password', this.SIREN_DEFAULT_ES_USER_PASSWORD);

      options.uri = `${protocol}//${username}:${password}@${rest.join('//')}/${indexName}`;
    }

    return _requestPromise2.default.get(options).then(index => {
      return true;
    }).catch(err => {
      if (err && err.error && err.error.error && err.error.error.type === 'index_not_found_exception') {
        return false;
      } else {
        process.stderr.write(`${err}\n`);
        return false;
      }
    }).finally(() => {
      if (this._originalTlsRejectUnauthorizedValue) {
        process.env.NODE_TLS_REJECT_UNAUTHORIZED = this._originalTlsRejectUnauthorizedValue;
      } else {
        delete process.NODE_TLS_REJECT_UNAUTHORIZED;
      }
    });
  }

  async doesKibiIndexExists() {
    return await this._doesIndexExist(this.kibiIndex);
  }

  async doesAclIndexExists() {
    if (this.aclIndex) {
      return await this._doesIndexExist(this.aclIndex);
    } else {
      return _bluebird2.default.resolve(false);
    }
  }

  async getIndicesInfo() {
    const kibiIndexExists = await this.doesKibiIndexExists();
    const aclIndexExists = await this.doesAclIndexExists();

    const indicesInfo = {
      kibi: kibiIndexExists,
      acl: aclIndexExists,
      indexNames: []
    };

    if (kibiIndexExists) {
      indicesInfo.indexNames.push(this.kibiIndex);
    }
    if (aclIndexExists) {
      indicesInfo.indexNames.push(this.aclIndex);
    }

    return indicesInfo;
  }

  async restore(deleteIndicesBeforeRestoring, indexExist = { kibi: true, acl: true }) {
    let dirExists;
    try {
      await _bluebird2.default.fromNode(cb => (0, _fs.access)(this._backupDir, cb));
      dirExists = true;
    } catch (err) {
      dirExists = false;
    }
    if (!dirExists) {
      throw new Error(`Backup folder [${this._backupDir}] does not exist`);
    }

    const restoreKibiPreprocess = new _restore_kibi_preprocess2.default(this._config, this._backupDir);
    const upgraded = restoreKibiPreprocess.preprocess();

    if (deleteIndicesBeforeRestoring && indexExist.kibi) {
      await this._dump.deleteIndex(this.kibiIndex);
      process.stdout.write('\nIndex ' + this.kibiIndex + ' deleted.\n\n');
    }
    await this._dump.fromFileToElasticsearch(this.kibiIndex, 'mapping', upgraded.mapping);
    await this._dump.fromFileToElasticsearch(this.kibiIndex, 'data', upgraded.data);
    process.stdout.write('\nIndex ' + this.kibiIndex + ' properly restored.\n\n');

    if (this.isAccessControPluginEnabled && this.isACLEnabled) {
      const aclIndex = (0, _lodash.get)(this._config, 'investigate_access_control.acl.index', '.sirenaccess');
      if (deleteIndicesBeforeRestoring && indexExist.acl) {
        await this._dump.deleteIndex(aclIndex);
        process.stdout.write('\nIndex ' + aclIndex + ' deleted.\n\n');
      }
      await this._dump.fromFileToElasticsearch(aclIndex, 'mapping');
      await this._dump.fromFileToElasticsearch(aclIndex, 'data');
      process.stdout.write('\nIndex ' + aclIndex + ' properly restored.\n\n');
    }
  }
}
exports.default = RestoreKibi;
module.exports = exports['default'];
