'use strict';

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

// kibi: kibi imports


var _lodash = require('lodash');

var _utils = require('../../utils');

var _boom = require('boom');

var _health_check = require('./lib/health_check');

var _health_check2 = _interopRequireDefault(_health_check);

var _create_data_cluster = require('./lib/create_data_cluster');

var _create_admin_cluster = require('./lib/create_admin_cluster');

var _client_logger = require('./lib/client_logger');

var _create_clusters = require('./lib/create_clusters');

var _filter_headers = require('./lib/filter_headers');

var _filter_headers2 = _interopRequireDefault(_filter_headers);

var _create_kibana_proxy = require('./lib/create_kibana_proxy');

var _create_kibana_proxy2 = _interopRequireDefault(_create_kibana_proxy);

var _create_kibi_proxy = require('./lib/create_kibi_proxy');

var _create_kibi_proxy2 = _interopRequireDefault(_create_kibi_proxy);

var _custom_clusters = require('./lib/custom_clusters');

var _util = require('./lib/util');

var _util2 = _interopRequireDefault(_util);

var _dbfilter = require('./lib/dbfilter');

var _dbfilter2 = _interopRequireDefault(_dbfilter);

var _siren_join = require('./lib/siren_join');

var _siren_join2 = _interopRequireDefault(_siren_join);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// kibi: end

const DEFAULT_REQUEST_HEADERS = ['authorization'];

module.exports = function (kibana) {
  return new kibana.Plugin({
    require: ['clientside_compression', 'kibana'],
    config(Joi) {
      const { array, boolean, number, object, string, ref } = Joi;

      const sslSchema = object({
        verificationMode: string().valid('none', 'certificate', 'full').default('full'),
        certificateAuthorities: array().single().items(string()),
        certificate: string(),
        key: string(),
        keyPassphrase: string()
      }).default();

      // kibi: extracted common schema part to reuse it inside "clusters" option
      const elasticsearchOptions = {
        url: string().uri({ scheme: ['http', 'https'] }).default('http://localhost:9200'),
        preserveHost: boolean().default(true),
        username: string(),
        password: string(),
        shardTimeout: number().default(0),
        requestTimeout: number().default(300000),
        requestHeadersWhitelist: array().items().single().default(DEFAULT_REQUEST_HEADERS),
        customHeaders: object().default({}),
        pingTimeout: number().default(ref('requestTimeout')),
        startupTimeout: number().default(5000),
        logQueries: boolean().default(false),
        ssl: sslSchema,
        apiVersion: Joi.string().default('5.x'),
        healthCheck: object({
          delay: number().default(2500),
          // On avarage when there is a latency on single elasticsearch call the whole healtCheck
          // on fast local box can take
          // latency   total
          // .5s      4 .. 8s
          // 1s       7 .. 14s
          // 3s      27 .. 40s
          // 5s      45 .. 55s
          // to be safe the default value of healthCheck.timeout is set to 2 min
          // user can lower this value - this will cause es status to change to red when
          // system can not perform full helthcheck faster than then timeout value
          timeout: number().default(120000)
        }).default(),
        tribe: object({
          url: string().uri({ scheme: ['http', 'https'] }),
          preserveHost: boolean().default(true),
          username: string(),
          password: string(),
          shardTimeout: number().default(0),
          requestTimeout: number().default(30000),
          requestHeadersWhitelist: array().items().single().default(DEFAULT_REQUEST_HEADERS),
          customHeaders: object().default({}),
          pingTimeout: number().default(ref('requestTimeout')),
          startupTimeout: number().default(5000),
          logQueries: boolean().default(false),
          ssl: sslSchema,
          apiVersion: Joi.string().default('5.x')
        }).default()
      };

      return object(_extends({
        enabled: boolean().default(true),
        // kibi: added extra properties
        siren: Joi.object({
          clusters: object().pattern(/[a-zA-Z0-9]/, elasticsearchOptions),
          connector: Joi.object({
            admin: Joi.object({
              cluster: Joi.string().alphanum()
            })
          }),
          alert: Joi.object({
            admin: Joi.object({
              cluster: Joi.string().alphanum()
            })
          }),
          security: Joi.object({
            admin: Joi.object({
              cluster: Joi.string().alphanum()
            })
          })
        })
      }, elasticsearchOptions)).default();
    },

    deprecations({ rename }) {
      const sslVerify = basePath => {
        const getKey = path => {
          return (0, _lodash.compact)([basePath, path]).join('.');
        };

        return (settings, log) => {
          const sslSettings = (0, _lodash.get)(settings, getKey('ssl'));

          if (!(0, _lodash.has)(sslSettings, 'verify')) {
            return;
          }

          const verificationMode = (0, _lodash.get)(sslSettings, 'verify') ? 'full' : 'none';
          (0, _lodash.set)(sslSettings, 'verificationMode', verificationMode);
          (0, _utils.unset)(sslSettings, 'verify');

          log(`Config key "${getKey('ssl.verify')}" is deprecated. It has been replaced with "${getKey('ssl.verificationMode')}"`);
        };
      };

      return [rename('ssl.ca', 'ssl.certificateAuthorities'), rename('ssl.cert', 'ssl.certificate'), sslVerify(), rename('tribe.ssl.ca', 'tribe.ssl.certificateAuthorities'), rename('tribe.ssl.cert', 'tribe.ssl.certificate'), sslVerify('tribe')];
    },

    uiExports: {
      injectDefaultVars(server, options) {
        return {
          esRequestTimeout: options.requestTimeout,
          esShardTimeout: options.shardTimeout,
          esApiVersion: options.apiVersion,
          esDataIsTribe: (0, _lodash.get)(options, 'tribe.url') ? true : false
        };
      }
    },

    init(server) {
      const kibanaIndex = server.config().get('kibana.index');
      const clusters = (0, _create_clusters.createClusters)(server);

      server.expose('getCluster', clusters.get);
      server.expose('createCluster', clusters.create);

      server.expose('filterHeaders', _filter_headers2.default);
      server.expose('ElasticsearchClientLogging', (0, _client_logger.clientLogger)(server));

      (0, _create_data_cluster.createDataCluster)(server);
      (0, _create_admin_cluster.createAdminCluster)(server);
      (0, _custom_clusters.createCustomClusters)(server); // kibi: create extra clusters based on investigate.yml elasticsearch.siren.clusters option

      // kibi: expose Investigate utility methods
      server.expose('getQueriesAsPromise', _util2.default.getQueriesAsPromise);
      server.expose('dbfilter', _dbfilter2.default);
      server.expose('sirenJoinSequence', (0, _siren_join2.default)(server).sequence);
      server.expose('sirenJoinSet', (0, _siren_join2.default)(server).set);
      // kibi: end

      (0, _create_kibi_proxy2.default)(server, 'GET', '/{paths*}');
      (0, _create_kibi_proxy2.default)(server, 'POST', '/_mget');

      (0, _create_kibi_proxy2.default)(server, 'PUT', '/_siren/connector/{type}/{id}'); // kibi:
      (0, _create_kibi_proxy2.default)(server, 'GET', '/_siren/connector/{type}/{id}'); // kibi:
      (0, _create_kibi_proxy2.default)(server, 'POST', '/_siren/connector/{type}/{id}/{action}'); // kibi:
      (0, _create_kibi_proxy2.default)(server, 'DELETE', '/_siren/connector/{type}/{id}'); // kibi:

      (0, _create_kibi_proxy2.default)(server, 'POST', '/_siren/job/{id}/_cancel'); // kibi: job cancellation API

      (0, _create_kibi_proxy2.default)(server, 'PUT', '/_siren/license'); // kibi: Insert new license
      (0, _create_kibi_proxy2.default)(server, 'DELETE', '/_siren/license'); // kibi: Delete existing license
      (0, _create_kibi_proxy2.default)(server, 'POST', '/{index}/_search');
      (0, _create_kibi_proxy2.default)(server, 'POST', '/{index}/{type}/_search'); // siren: for some searches we specify the index type
      (0, _create_kibi_proxy2.default)(server, 'POST', '/_msearch');
      (0, _create_kibi_proxy2.default)(server, 'POST', '/_search/scroll');
      (0, _create_kibana_proxy2.default)(server, 'POST', '/_ingest/pipeline/_simulate');

      function noBulkCheck({ path }, reply) {
        if (/\/_bulk/.test(path)) {
          return reply({
            error: 'You can not send _bulk requests to this interface.'
          }).code(400).takeover();
        }
        return reply.continue();
      }

      function noDirectIndex({ path }, reply) {
        const requestPath = (0, _lodash.trimRight)((0, _lodash.trim)(path), '/');
        const matchPath = (0, _create_kibana_proxy.createPath)('/elasticsearch', kibanaIndex);

        if (requestPath === matchPath) {
          return reply((0, _boom.methodNotAllowed)('You cannot modify the primary kibana index through this interface.'));
        }

        reply.continue();
      }

      // These routes are actually used to deal with things such as managing
      // index patterns and advanced settings, but since hapi treats route
      // wildcards as zero-or-more, the routes also match the kibana index
      // itself. The client-side kibana code does not deal with creating nor
      // destroying the kibana index, so we limit that ability here.
      (0, _create_kibi_proxy2.default)(server, ['PUT', 'POST', 'DELETE'], `/${kibanaIndex}/{paths*}`, {
        pre: [noDirectIndex, noBulkCheck]
      });
      // Set up the health check service and start it.
      const mappings = kibana.uiExports.mappings.getCombined();
      const { start, waitUntilReady } = (0, _health_check2.default)(this, server, { mappings });
      server.expose('waitUntilReady', waitUntilReady);
      start();
    }
  });
};
