'use strict';

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _elasticsearch = require('elasticsearch');

var _elasticsearch2 = _interopRequireDefault(_elasticsearch);

var _create_kibana_index = require('./create_kibana_index');

var _create_kibana_index2 = _interopRequireDefault(_create_kibana_index);

var _kibana_version = require('./kibana_version');

var _kibana_version2 = _interopRequireDefault(_kibana_version);

var _ensure_es_version = require('./ensure_es_version');

var _ensure_not_tribe = require('./ensure_not_tribe');

var _ensure_allow_explicit_index = require('./ensure_allow_explicit_index');

var _determine_enabled_scripting_langs = require('./determine_enabled_scripting_langs');

var _ensure_types_exist = require('./ensure_types_exist');

var _server = require('../../../siren_core_plugins/federate_resolver/server');

var _utils = require('../../../utils');

var _kibi_version = require('./kibi_version');

var _kibi_version2 = _interopRequireDefault(_kibi_version);

var _get_elasticsearch_version = require('./get_elasticsearch_version');

var _wait_for_plugin_list = require('./wait_for_plugin_list');

var _wait_for_plugin_list2 = _interopRequireDefault(_wait_for_plugin_list);

var _custom_clusters = require('./custom_clusters');

var _util = require('util');

var _util2 = _interopRequireDefault(_util);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// kibi: end

//added by kibi to know the list of installed plugins

// kibi: added imports
const NoConnections = _elasticsearch2.default.errors.NoConnections;

const format = _util2.default.format;

const NO_INDEX = 'no_index';
const INITIALIZING = 'initializing';
const READY = 'ready';

module.exports = function (plugin, server, { mappings }) {
  const config = server.config();
  const callAdminAsKibanaUser = server.plugins.elasticsearch.getCluster('admin').callWithInternalUser;
  const callDataAsKibanaUser = server.plugins.elasticsearch.getCluster('data').callWithInternalUser;
  const REQUEST_DELAY = config.get('elasticsearch.healthCheck.delay');
  const healthCheckTimeout = config.get('elasticsearch.healthCheck.timeout');

  (0, _server.logCompatibilityMatrix)(_utils.pkg.siren.compatibility, server);
  plugin.status.yellow('Waiting for Elasticsearch');
  function waitForPong(callWithInternalUser, url) {
    return callWithInternalUser('ping').catch(function (err) {
      if (!(err instanceof NoConnections)) throw err;
      plugin.status.red(format('Unable to connect to Elasticsearch at %s.', url));

      return _bluebird2.default.delay(REQUEST_DELAY).then(waitForPong.bind(null, callWithInternalUser, url));
    });
  }

  // just figure out the current "health" of the es setup
  function getHealth() {
    return callAdminAsKibanaUser('cluster.health', {
      timeout: '5s', // tells es to not sit around and wait forever
      index: config.get('kibana.index'),
      ignore: [408]
    }).then(function (resp) {
      // if "timed_out" === true then elasticsearch could not
      // find any idices matching our filter within 5 seconds
      if (!resp || resp.timed_out) {
        return NO_INDEX;
      }

      // If status === "red" that means that index(es) were found
      // but the shards are not ready for queries
      if (resp.status === 'red') {
        return INITIALIZING;
      }

      return READY;
    });
  }

  function waitUntilReady() {
    return getHealth().then(function (health) {
      if (health !== READY) {
        return _bluebird2.default.delay(REQUEST_DELAY).then(waitUntilReady);
      }
    });
  }

  function waitForShards() {
    return getHealth().then(function (health) {
      if (health === NO_INDEX) {
        plugin.status.yellow('No existing Kibana index found');
        return (0, _create_kibana_index2.default)(server, mappings);
      }

      if (health === INITIALIZING) {
        plugin.status.red('Elasticsearch is still initializing the Siren Investigate index.'); // kibi: changed kibana to Siren Investigate
        return _bluebird2.default.delay(REQUEST_DELAY).then(waitForShards);
      }
    });
  }

  function waitForEsVersion(clusterName) {
    return (0, _ensure_es_version.ensureEsVersion)(server, _kibi_version2.default.get(), clusterName).catch(err => {
      plugin.status.red(err);
      return _bluebird2.default.delay(REQUEST_DELAY).then(() => {
        waitForEsVersion(clusterName);
      });
    });
  }

  function setGreenStatus() {
    return plugin.status.green('Siren Investigate index ready');
  }

  function pingCustomCluster(clusterNameProperty) {
    // remember "has" will return true if it is defined in the schema !!!  So use get
    if (config.get(clusterNameProperty)) {
      const clusterName = config.get(clusterNameProperty);
      const clustersConfig = config.get(_custom_clusters.CLUSTERS_PROPERTY);
      if (!clustersConfig || !clustersConfig[clusterName]) {
        return new Error((0, _custom_clusters.getConfigMismatchErrorMessage)(clusterName, clusterNameProperty));
      }
      const clusterConfig = clustersConfig[clusterName];
      let url = clusterConfig.url;
      if (clusterConfig.tribe && clusterConfig.tribe.url) {
        url = clusterConfig.tribe.url;
      }
      const callClusterAsKibanaUser = server.plugins.elasticsearch.getCluster(clusterName).callWithInternalUser;
      return waitForPong(callClusterAsKibanaUser, url).then(() => (0, _ensure_es_version.ensureEsVersion)(server, _kibi_version2.default.get(), clusterName));
    }
  };

  function check() {
    const results = {};
    // NOTE: it must be bluebird promise as the native one does not support timeout yet
    const healthCheck = new _bluebird2.default(function (resolve, reject) {
      waitForPong(callAdminAsKibanaUser, config.get('elasticsearch.url')).then(() => waitForEsVersion('admin')).then(() => (0, _ensure_not_tribe.ensureNotTribe)(callAdminAsKibanaUser)).then(() => (0, _ensure_allow_explicit_index.ensureAllowExplicitIndex)(callAdminAsKibanaUser, config)).then(waitForShards).then(_lodash2.default.partial(_wait_for_plugin_list2.default, plugin, server)) // kibi: added by kibi to know the list of installed plugins
      // kibi: mappings are handled by savedObjectAPI
      // and we take take care about migrating config ourselves
      // .then(() => ensureTypesExist({
      //   callCluster: callAdminAsKibanaUser,
      //   log: (...args) => server.log(...args),
      //   indexName: config.get('kibana.index'),
      //   types: Object.keys(mappings).map(name => ({ name, mapping: mappings[name] }))
      // }))
      // .then(_.partial(migrateConfig, server))
      .then(() => (0, _get_elasticsearch_version.getElasticsearchVersion)(server)).then(async version => {
        if (version.major < 6) {
          results.enabledScriptingLangs = await (0, _determine_enabled_scripting_langs.determineEnabledScriptingLangs)(callDataAsKibanaUser);
        } else {
          // kibi: on 6 you can use only these
          results.enabledScriptingLangs = ['painless', 'expression'];
        }
      }).then(() => {
        const tribeUrl = config.get('elasticsearch.tribe.url');
        if (tribeUrl) {
          return waitForPong(callDataAsKibanaUser, tribeUrl).then(() => (0, _ensure_es_version.ensureEsVersion)(server, _kibi_version2.default.get(), 'admin'));
        }
      }).then(pingCustomCluster(_custom_clusters.CONNECTOR_CLUSTER_PROPERTY)).then(pingCustomCluster(_custom_clusters.SECURITY_CLUSTER_PROPERTY)).then(pingCustomCluster(_custom_clusters.ALERT_CLUSTER_PROPERTY)).then(resolve).catch(reject);
    }).timeout(healthCheckTimeout);

    return healthCheck.then(() => server.expose('latestHealthCheckResults', results)).then(setGreenStatus).catch(err => plugin.status.red(err));
  }

  let timeoutId = null;

  function scheduleCheck(ms) {
    if (timeoutId) return;

    const myId = setTimeout(function () {
      check().finally(function () {
        if (timeoutId === myId) startorRestartChecking();
      });
    }, ms);

    timeoutId = myId;
  }

  function startorRestartChecking() {
    scheduleCheck(stopChecking() ? REQUEST_DELAY : 1);
  }

  function stopChecking() {
    if (!timeoutId) return false;
    clearTimeout(timeoutId);
    timeoutId = null;
    return true;
  }

  server.ext('onPreStop', (request, reply) => {
    stopChecking();
    reply();
  });

  return {
    waitUntilReady: waitUntilReady,
    run: check,
    start: startorRestartChecking,
    stop: stopChecking,
    isRunning: function () {
      return !!timeoutId;
    }
  };
};
