import _ from 'lodash';
import angular from 'angular';
import { uiModules } from 'ui/modules';
import uiRoutes from 'ui/routes';
import chrome from 'ui/chrome';

import 'plugins/kibana/dashboard/grid';
import 'plugins/kibana/dashboard/panel/panel';
import 'ui/kibi/components/dashboards360/use_visualization_for_count_on_dashboard';

// kibi: SavedObjectAuthorizationError is added by kibi
import { SavedObjectNotFound, SavedObjectAuthorizationError } from 'ui/errors';
import { getDashboardTitle, getDashBoardTitleMaxLength, isDashboardUnsaved, getUnsavedChangesWarningMessage } from './dashboard_strings';
import { DashboardViewMode } from './dashboard_view_mode';
import { TopNavIds } from './top_nav/top_nav_ids';
import { ConfirmationButtonTypes } from 'ui/modals/confirm_modal';
import dashboardTemplate from 'plugins/kibana/dashboard/dashboard.html';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { DocTitleProvider } from 'ui/doc_title';
import { getTopNavConfig } from './top_nav/get_top_nav_config';
import { DashboardConstants, createDashboardEditUrl } from './dashboard_constants';
import { VisualizeConstants } from 'plugins/kibana/visualize/visualize_constants';
import { UtilsBrushEventProvider } from 'ui/utils/brush_event';
import { FilterBarClickHandlerProvider } from 'ui/filter_bar/filter_bar_click_handler';
// kibi: need to call private on dashboard_state
import { DashboardStateProvider } from './dashboard_state';
import { notify } from 'ui/notify';
import './panel/get_object_loaders_for_dashboard';
import { documentationLinks } from 'ui/documentation_links/documentation_links';
import { showCloneModal } from './top_nav/show_clone_modal';

// kibi: imports
import { getDefaultQuery } from 'ui/parse_query';
import 'ui/kibi/directives/kibi_human_readable_number';
import { HashedItemStoreSingleton } from 'ui/state_management/state_storage';
import { SpinnerStatus } from 'ui/kibi/spinners/spinner_status';
import { replaceKibiStateInLastURLHelperFactory } from 'ui/kibi/helpers/replace_kibi_state_in_last_url_helper';
import { savedVisualizationProvider } from 'plugins/kibana/visualize/saved_visualizations/saved_visualization_register';
import { SirenExportDashboardHelperProvider } from 'ui/kibi/helpers/siren_export_dashboard_helper';
import 'ui/kibi/components/dashboards360/directives/dash360_explanation';
import {
  findMainCoatNode,
  getCoatVisualizations,
  getParsedDecoratedCoat,
  shouldTimefilterBeEnabled,
  findMainCoatNodeSearchId,
  checkTreeItems,
  isThereAssignedVisBasedOnMainSearch,
  setCountForMainSearch
} from 'ui/kibi/components/dashboards360/coat_tree';
import { DashboardDatamodelType } from 'plugins/kibana/dashboard/saved_dashboard/dashboard_datamodel_type';
import { JoinAlgorithmType } from 'ui/kibi/components/dashboards360/lib/join_algorithm_type';
import { SetEntityAndIndexpatternOnDashProvider } from './set_entity_and_indexpattern_on_dash_provider';
// kibi: end
const app = uiModules.get('app/dashboard', [
  'elasticsearch',
  'ngRoute',
  'kibana/courier',
  'kibana/config',
  'kibana/notify',
  'kibana/typeahead',
]);

uiRoutes
  .when(DashboardConstants.CREATE_NEW_DASHBOARD_URL, {
    template: dashboardTemplate,
    resolve: {
      dash: function (savedDashboards, courier) {
        return savedDashboards.get()
          .catch(courier.redirectWhenMissing({
            'dashboard': DashboardConstants.LANDING_PAGE_PATH
          }));
      }
    }
  })
  .when(createDashboardEditUrl(':id'), {
    template: dashboardTemplate,
    resolve: {
      dash: function (savedDashboards, coatDashboardMap, Private,
        $route, courier, kbnUrl, AppState, ontologyModel, useVisualizationForCountOnDashboard, savedSearches, timefilter) {
        const id = $route.current.params.id;
        return savedDashboards.get(id)
          .then(function (dash) {

            dash.siren = {
              entities: null, // TODO: rename with coat prefix
              relations: null,
              searches: null,

              // the below properties have to be reloaded
              visualizations: null, // when adding removing panels
              coat: null, // when saving dashboard
              coatMainSearchId: null, // when saving dashboard
              coatMainNodeId: null, // when saving dashboard
              appState: new AppState()
            };
            useVisualizationForCountOnDashboard.set(dash.id, false);

            return coatDashboardMap.updateCoatDashboardMapForDashboard(dash)
              .then(() => {
              // for current dashboard expose getPanels method
              // to be able to grab panels dynamically even if dashboard is in edit mode
                // kibi decide if timefilter should be shown
                timefilter.enabled = shouldTimefilterBeEnabled(
                  _.get(dash, 'siren.coat.items') ? dash.siren.coat.items : []
                );
                // kibi: end
                dash.siren.getPanels = function () {
                  return this.appState.panels;
                };
                const savedSearchId = dash.getMainSavedSearchId();
                if (savedSearchId) {

                  if (dash.siren) {
                    useVisualizationForCountOnDashboard.set(
                      dash.id,
                      isThereAssignedVisBasedOnMainSearch(dash.siren.coat.items)
                    );
                  }
                  const setEntityAndIndexPatternOnDash = Private(SetEntityAndIndexpatternOnDashProvider);
                  return setEntityAndIndexPatternOnDash(dash, savedSearchId);
                } else {
                  return dash;
                }
              });
          })
          .catch((error) => {
          // Preserve BWC of v5.3.0 links for new, unsaved dashboards.
          // See https://github.com/elastic/kibana/issues/10951 for more context.
            if (error instanceof SavedObjectNotFound && id === 'create') {
            // Note "new AppState" is neccessary so the state in the url is preserved through the redirect.
              kbnUrl.redirect(DashboardConstants.CREATE_NEW_DASHBOARD_URL, {}, new AppState());
              notify.error(
                'The url "dashboard/create" is deprecated and will be removed in 6.0. Please update your bookmarks.');
            } else if (error instanceof SavedObjectAuthorizationError) {
              notify.error('Dashboard ' + id + '. ' + error.message);
              throw error;
            } else {
              throw error;
            }
          })
          .catch(courier.redirectWhenMissing({
            'dashboard' : DashboardConstants.LANDING_PAGE_PATH
          }));
      }
    }
  });

app.directive('dashboardApp', function (createNotifier, $injector) {
  const courier = $injector.get('courier');
  const AppState = $injector.get('AppState');
  const timefilter = $injector.get('timefilter');
  const quickRanges = $injector.get('quickRanges');
  const kbnUrl = $injector.get('kbnUrl');
  const confirmModal = $injector.get('confirmModal');
  const Private = $injector.get('Private');

  const brushEvent = Private(UtilsBrushEventProvider);
  const filterBarClickHandler = Private(FilterBarClickHandlerProvider);
  // kibi: added to keep correct kibiState if timefilter was applied from discover page
  const replaceKibiStateInLastURL = Private(replaceKibiStateInLastURLHelperFactory);
  // kibi: end

  return {
    restrict: 'E',
    controllerAs: 'dashboardApp',
    controller: function ($scope, $rootScope, $route, $routeParams, $location, getAppState, $compile,
      // kibi: added dashboardGroups, kibiState, config, $window, chrome, $timeout
      ontologyModel, dashboardGroups, kibiState, config, $window, chrome, $timeout, $document,
      savedVisualizations, savedSearches, coatDashboardMap, useVisualizationForCountOnDashboard) {
      const setEntityAndIndexPatternOnDash = Private(SetEntityAndIndexpatternOnDashProvider);
      const filterBar = Private(FilterBarQueryFilterProvider);
      const docTitle = Private(DocTitleProvider);
      const exportDashboardHelper = Private(SirenExportDashboardHelperProvider);
      let notify = createNotifier({
        location: 'Dashboard'
      });
      $scope.queryDocLinks = documentationLinks.query;

      const dash = $scope.dash = $route.current.locals.dash;
      let originalCoat = JSON.parse($scope.dash.coatJSON);

      if (dash.id) {
        docTitle.change(dash.title);
      }

      // kibi: expose the SpinnerStatus class
      $scope.SpinnerStatus = SpinnerStatus;

      // kibi: adds information about the group membership and stats
      function getMetadata(data) {
        const noEventDataValidDash = !data && dash.id;
        // if there was an event make sure to execute this function only if the event was about this dashboard
        const yesEventDataValidDash = data && dash.id && dash.id === data.id;

        if (noEventDataValidDash || yesEventDataValidDash) {
          delete dash.group;
          dash.group = dashboardGroups.getGroup(dash.id);
          if (dash.group) {
            const found = _.find(dash.group.dashboards, d => d.id === dash.id);
            dash.count = found.count;
            dash.isPruned = found.isPruned;

            // Note:
            // Update the count for main node on the tree if there is no vis from which the count could be taken
            if (
              yesEventDataValidDash &&
              dash.siren.coat.datamodelType === DashboardDatamodelType.MULTIPLE_SEARCHES &&
              !isThereAssignedVisBasedOnMainSearch(dash.siren.coat.items)
            ) {
              setCountForMainSearch(dash.siren.coat.items, found.count);
            }
          }
        }
      }

      $scope.$watch(() => {
        let data;
        if (useVisualizationForCountOnDashboard.get(dash.id) && dash.siren && dash.siren.coat && dash.siren.coat.items) {
          const node = findMainCoatNode(dash.siren.coat.items);
          if (node && node.d.entity.data !== undefined) {
            data = _.cloneDeep(node.d.entity.data);
          }
        }
        return data;
      }, data => {
        if (data !== undefined) {
          if (data.error) {
            dashboardGroups.updateDashboardMetaData(dash.id, data);
          } else {
            kibiState.compareStateToSavedDashState(dash)
              .then(dirtyDiff => {
                const meta = Object.assign(data, dirtyDiff);
                dashboardGroups.updateDashboardMetaData(dash.id, meta);
              });
          }
        };
      }, true);

      const getMetaWrapper = (data) => getMetadata(data);
      dashboardGroups.on('groupsMetadataUpdated', getMetaWrapper);
      dashboardGroups.on('dashboardMetadataUpdated', getMetaWrapper);

      getMetadata();
      // kibi: end

      // kibi: get the DashboardState class
      const DashboardState = Private(DashboardStateProvider);
      const dashboardState = new DashboardState(dash, AppState, kibiState);

      // kibi: check filters or query changed or not.
      $scope.filtersQueryTitleOrSavedSearchChanged = false;


      const appStateSaveWithChangeHandler = function (diff) {
        const checkQuery = _.indexOf(diff, 'query');
        const checkFilters = _.indexOf(diff, 'filters');
        if (checkQuery !== -1 || checkFilters !== -1 || dashboardState.getTimeChanged(timefilter)) {
          $scope.filtersQueryTitleOrSavedSearchChanged = true;
        }
      };
      let appStateSaveWithChangeHandlerOff;
      $scope.$watch(getAppState, function (appState) {
        if (appState) {
          if (appStateSaveWithChangeHandlerOff) {
            // kibi: remove the old handler before attaching the new one
            appStateSaveWithChangeHandlerOff();
          }
          appState.on('save_with_changes', appStateSaveWithChangeHandler);
          appStateSaveWithChangeHandlerOff = function () {
            appState.off('save_with_changes', appStateSaveWithChangeHandler);
          };
        }
      });

      $scope.$watch(function () {
        return dashboardState.getTitle();
      }, (newValue, oldValue) => {
        if (newValue !== oldValue) {
          $scope.filtersQueryTitleOrSavedSearchChanged = true;
        }
      });

      $scope.$watch(function () {
        return findMainCoatNodeSearchId(dashboardState.savedDashboard.coatJSON);
      }, (newValue, oldValue) => {
        if (newValue !== oldValue) {
          $scope.filtersQueryTitleOrSavedSearchChanged = true;
        }
      });
      // kibi: end


      // kibi: time from the kibi state.
      const dashboardTime = kibiState._getDashboardProperty(dash.id, kibiState._properties.time);
      if (dashboardTime) {
        // this allows to set a time (not save it with a dashboard), switch between dashboards, and
        // still retain the time set until the app is reloaded
        timefilter.time.mode = dashboardTime.m;
        timefilter.time.from = dashboardTime.f;
        timefilter.time.to = dashboardTime.t;
        if (dash.refreshInterval) {
          timefilter.refreshInterval = dash.refreshInterval;
        }
      } else if (dashboardState.getIsTimeSavedWithDashboard()) {
        dashboardState.syncTimefilterWithDashboard(timefilter, quickRanges);
      } else {
        // kibi: set default time
        const { mode, from, to } = config.get('timepicker:timeDefaults');
        timefilter.time.mode = mode;
        timefilter.time.to = to;
        timefilter.time.from = from;
        // kibi: end
      }

      const updateState = () => {
        // Following the "best practice" of always have a '.' in your ng-models –
        // https://github.com/angular/angular.js/wiki/Understanding-Scopes
        $scope.shouldTimeFilterEnabled = shouldTimefilterBeEnabled(
          _.get($scope.dash, 'siren.coat.items') ? $scope.dash.siren.coat.items : []
        );

        const defaultQuery = getDefaultQuery();
        const searchQueryPresent = !_.isEqual(dashboardState.getQuery(), defaultQuery);
        const filterPresent = filterBar.getFilters().length !== 0 || $scope.orFiltersPresent;
        const searchBarVisible = searchQueryPresent || !dashboardState.getHideSearchBar();
        const filterBarVisible = filterPresent || ($scope.showFilterButton && searchBarVisible);

        $scope.model = {
          hideBorders: dashboardState.getHideBorders(), // kibi: toggle borders around panels
          showTimePicker: $scope.shouldTimeFilterEnabled || dashboardState.getShowTimePicker(),
          hideSearchBar: dashboardState.getHideSearchBar(),
          searchBarVisible,
          filterBarVisible,
          query: dashboardState.getQuery(),
          darkTheme: dashboardState.getDarkTheme(),
          timeRestore: dashboardState.getTimeRestore(),
          title: dashboardState.getTitle(),
          description: dashboardState.getDescription(),
        };
        $scope.panels = dashboardState.getPanels();
        $scope.filterButtonIsDisabled = filterBar.getFilters().length !== 0;
      };

      $scope.showFilterButton = false;
      $scope.setShowFilterButton = (value) => {
        $scope.showFilterButton = value;
        updateState();
      };

      $scope.orFiltersPresent = false;
      const orFiltersPresentHandler = $rootScope.$on('filterbar:orFiltersPresent', (event, orFiltersPresent) => {
        $scope.orFiltersPresent = orFiltersPresent;
        updateState();
      });

      $scope.coatOptions = {
        DashboardDatamodelType,
        datamodelType: dash.getDashboardDataModelType(),
        mainSearchId: dash.getMainSavedSearchId(),
        joinAlgorithmType: dash.getJoinAlgorithmType()
      };

      function setMainSearchId($scope, dash) {
        if (!$scope.coatOptions.mainSearchId && dash.siren.searches.length > 0) {
          const firstSearchId = _.sortBy(dash.siren.searches, 'title')[0].id;
          $scope.coatOptions.mainSearchId = firstSearchId;
          $scope.dash.setMainSavedSearchId(firstSearchId);
        }
      }

      $scope.$watch('coatOptions.datamodelType', (datamodelType, oldDataModelType) => {
        if (datamodelType && datamodelType !== oldDataModelType) {
          dash.setDashboardDataModelType(datamodelType);
          if (datamodelType === DashboardDatamodelType.MULTIPLE_SEARCHES) {
            $scope.coatOptions.joinAlgorithmType = JoinAlgorithmType.LEFT_FILTER_JOIN;
            setMainSearchId($scope, dash);
          } else if (datamodelType === DashboardDatamodelType.SINGLE_SEARCH) {
            setMainSearchId($scope, dash);
          } else {
            $scope.coatOptions.joinAlgorithmType = undefined;
            $scope.coatOptions.mainSearchId = undefined;
            $scope.dash.setMainSavedSearchId(undefined);
          }
        }
      });

      $scope.$watch('coatOptions.mainSearchId', (mainSearchId, oldMainSearchId) => {
        if (mainSearchId && mainSearchId !== oldMainSearchId) {
          if ($scope.coatOptions.datamodelType === DashboardDatamodelType.SINGLE_SEARCH) {
            $scope.dash.setMainSavedSearchId(mainSearchId);
            setEntityAndIndexPatternOnDash($scope.dash, mainSearchId);
          }
        }
      });

      $scope.$watch('coatOptions.joinAlgorithmType', (joinAlgorithmType, oldJoinAlgorithmType) => {
        if (joinAlgorithmType && joinAlgorithmType !== oldJoinAlgorithmType) {
          dash.setJoinAlgorithmType(joinAlgorithmType);
        }
      });

      // Part of the exposed plugin API - do not remove without careful consideration.
      this.appStatus = {
        dirty: !dash.id
      };
      dashboardState.stateMonitor.onChange(status => {
        this.appStatus.dirty = status.dirty || !dash.id;
        updateState();
      });

      dashboardState.applyFilters(dashboardState.getQuery(), filterBar.getFilters());
      let pendingVisCount = _.size(dashboardState.getPanels());

      dash.searchSource.highlightAll(true);
      dash.searchSource.version(true);
      courier.setRootSearchSource(dash.searchSource);

      updateState();

      $scope.refresh = (...args) => {
        $rootScope.$broadcast('fetch');
        courier.fetch(...args);
        // kibi: added to keep correct kibiState if timefilter was applied from discover page
        replaceKibiStateInLastURL('kibana:discover');
        // kibi: end
      };
      $scope.timefilter = timefilter;
      $scope.expandedPanel = null;
      // kibi: set 'showSavedSearchExplanation' to false
      $scope.showSavedSearchExplanation = false;
      $scope.dashboardViewMode = dashboardState.getViewMode();

      $scope.landingPageUrl = () => `#${DashboardConstants.LANDING_PAGE_PATH}`;
      $scope.listingPageUrl = () => `#${DashboardConstants.LISTING_PAGE_PATH}`; // kibi: expose listing page path
      $scope.getBrushEvent = () => brushEvent(dashboardState.getAppState());
      $scope.getFilterBarClickHandler = () => filterBarClickHandler(dashboardState.getAppState());
      $scope.hasExpandedPanel = () => $scope.expandedPanel !== null;
      // kibi: pass max length of dashboard title to getDashboardTitle()
      $scope.getDashTitle = () => getDashboardTitle(
        dashboardState.getTitle(),
        dashboardState.getViewMode(),
        dashboardState.getIsDirty(timefilter),
        getDashBoardTitleMaxLength());
      $scope.isDashUnsaved = () => isDashboardUnsaved(
        dashboardState.getViewMode(),
        dashboardState.getIsDirty(timefilter)
      );
      $scope.newDashboard = () => { kbnUrl.change(DashboardConstants.CREATE_NEW_DASHBOARD_URL, {}); };
      $scope.saveState = () => dashboardState.saveState();
      $scope.getShouldShowEditHelp = () => !dashboardState.getPanels().length && dashboardState.getIsEditMode();
      $scope.getShouldShowViewHelp = () => !dashboardState.getPanels().length && dashboardState.getIsViewMode();

      $scope.toggleExpandPanel = (panelIndex) => {
        if ($scope.expandedPanel && $scope.expandedPanel.panelIndex === panelIndex) {
          $scope.expandedPanel = null;
          dashboardState.setExpandedPanelIndex(null);
        } else {
          $scope.expandedPanel =
            dashboardState.getPanels().find((panel) => panel.panelIndex === panelIndex);
          dashboardState.setExpandedPanelIndex(panelIndex);
        }
      };


      if (dashboardState.getExpandedPanelIndex()) {
        const expandedPanelIndex = dashboardState.getExpandedPanelIndex();
        const panel = dashboardState.getPanels().find((panel) => panel.panelIndex === expandedPanelIndex);
        if (panel && !$scope.expandedPanel) {
          $scope.toggleExpandPanel(expandedPanelIndex);
        } else {
          // seems panel does no longer exist - null the remove the state
          dashboardState.setExpandedPanelIndex(null);
        }
      }

      $scope.filterResults = function () {
        dashboardState.applyFilters($scope.model.query, filterBar.getFilters());
        $scope.refresh();
      };

      // kibi: refresh things related to coat
      // refresh visualizations
      const refreshVisualizationsForCoat = function () {
        return getCoatVisualizations(savedVisualizations, savedSearches, dashboardState.getPanels())
          .then(visualizations => {
            if ($scope.dash && $scope.dash.siren) {
              $scope.dash.siren.visualizations = visualizations;

              useVisualizationForCountOnDashboard.set(
                dash.id,
                isThereAssignedVisBasedOnMainSearch(dash.siren.coat.items)
              );
            }
          });
      };
      refreshVisualizationsForCoat();

      // refresh coat
      $scope.$watch('dash.coatJSON', (coatJSON, oldCoatJSON) => {
        if (coatJSON !== oldCoatJSON)  {
          dashboardState.savedDashboard.coatJSON = coatJSON;

          coatDashboardMap.updateCoatDashboardMapForDashboard($scope.dash)
            .then(() => dash.getMainSavedSearchId())
            .then(mainSearchId => {
              if (mainSearchId) {
                return setEntityAndIndexPatternOnDash($scope.dash, mainSearchId);
              }
            })
            .then(() => $scope.refresh())
            .then(() => refreshVisualizationsForCoat())
            .then(() => dashboardGroups.updateMetadataOfDashboardIds([$scope.dash.id]))
            .then(() => {
              timefilter.enabled = shouldTimefilterBeEnabled(
                _.get($scope.dash, 'siren.coat.items') ? $scope.dash.siren.coat.items : []
              );
              $scope.model.showTimePicker = timefilter.enabled;
            });
        }
      });
      // kibi: end

      // called by the saved-object-finder when a user clicks a vis
      $scope.addVis = function (hit, showToast = true) {
        pendingVisCount++;
        dashboardState.addNewPanel(hit.id, 'visualization');
        refreshVisualizationsForCoat();
        if (showToast) {
          notify.info(`Visualization successfully added to your dashboard`);
        }
      };

      $scope.addSearch = function (hit) {
        pendingVisCount++;
        dashboardState.addNewPanel(hit.id, 'search');
        refreshVisualizationsForCoat();
        notify.info(`Search successfully added to your dashboard`);
      };

      /**
       * Creates a child ui state for the panel. It's passed the ui state to use, but needs to
       * be generated from the parent (why, I don't know yet).
       * @param path {String} - the unique path for this ui state.
       * @param uiState {Object} - the uiState for the child.
       * @returns {Object}
       */
      $scope.createChildUiState = function createChildUiState(path, uiState) {
        return dashboardState.uiState.createChild(path, uiState, true);
      };

      $scope.onPanelRemoved = (panelIndex) => {
        dashboardState.removePanel(panelIndex);
        if (dashboardState.getExpandedPanelIndex()) {
          if (dashboardState.getExpandedPanelIndex() === panelIndex) {
            dashboardState.setExpandedPanelIndex(null);
          }
        }

        refreshVisualizationsForCoat();
      };

      $scope.toggleSavedSearchExplanation = () => $scope.showSavedSearchExplanation = !$scope.showSavedSearchExplanation;
      $scope.jumpToDiscover = () => $window.location.href = chrome.getNavLinkById('kibana:discover').url;

      $scope.$watch('model.darkTheme', () => {
        dashboardState.setDarkTheme($scope.model.darkTheme);
        updateTheme();
      });

      // kibi: watch 'model.hideBorders'
      $scope.$watch('model.hideBorders', () => dashboardState.setHideBorders($scope.model.hideBorders));
      $scope.$watch('model.hideSearchBar', () => {
        dashboardState.setHideSearchBar($scope.model.hideSearchBar);
      });
      $scope.$watch('model.showTimePicker', () => {
        dashboardState.setShowTimePicker($scope.model.showTimePicker);
        timefilter.enabled = $scope.model.showTimePicker;
      });
      $scope.$watch('model.description', () => dashboardState.setDescription($scope.model.description));
      $scope.$watch('model.title', () => dashboardState.setTitle($scope.model.title));
      $scope.$watch('model.timeRestore', () => dashboardState.setTimeRestore($scope.model.timeRestore));
      $scope.indexPatterns = [];

      if ($scope.dash.indexPattern) {
        $scope.indexPatterns = [$scope.dash.indexPattern];
      } else {
        $scope.registerPanelIndexPattern = (panelIndex, pattern) => {
          dashboardState.registerPanelIndexPatternMap(panelIndex, pattern);
          $scope.indexPatterns = dashboardState.getPanelIndexPatterns();
        };

        $scope.onPanelRemoved = (panelIndex) => {
          dashboardState.removePanel(panelIndex);
          refreshVisualizationsForCoat();
          $scope.indexPatterns = dashboardState.getPanelIndexPatterns();
        };
      }

      $scope.$listen(timefilter, 'fetch', () => {
        // kibi: the listener below is needed to react when the global time is changed by the user
        // either directly in time widget or by clicking on histogram chart etc
        const { mode, from, to } = timefilter.time;
        if (dash.id && !kibiState._isDefaultTime(mode, from, to)) {
          kibiState._saveTimeForDashboardId(dash.id, mode, from, to);
          kibiState.save();
        }
        // kibi: end
        $scope.refresh();
      });

      function updateViewMode(newMode) {
        $scope.topNavMenu = getTopNavConfig(newMode, navActions); // eslint-disable-line no-use-before-define
        dashboardState.switchViewMode(newMode);
        $scope.dashboardViewMode = newMode;
      }

      const revertChangesAndExitEditMode = () => {
        const currentCoat = JSON.parse($scope.dash.coatJSON);
        if (!_.isEqual(currentCoat, originalCoat)) {
          $scope.dash.coatJSON = JSON.stringify(originalCoat);
          updateViewMode(DashboardViewMode.VIEW);
        }

        if (dashboardState.getIsDirty(timefilter)) {
          dashboardState.resetState();

          // kibi: Ensure dashboard counts update on exiting edit mode
          if (dash.id) {
            dashboardGroups.selectDashboard(dash.id);
          } else {
            kbnUrl.change(DashboardConstants.CREATE_NEW_DASHBOARD_URL);
          }
          // kibi: end

          // This is only necessary for new dashboards, which will default to Edit mode.
          updateViewMode(DashboardViewMode.VIEW);

          // kibi: switch back to old state
          filterBar.removeAll();
          return filterBar.addFilters($scope.currentState.currentAppFilters)
            .then(filterBar.addFilters($scope.currentState.currentGlobalFilters, true))
            .then(() => {
              // 'edit' mode can be an initial mode when we did not save/close dashboard but switched to another app
              // $scope.currentState can be empty if 'edit' is initial mode and we can loose state
              if (!_.isEmpty($scope.currentState)) {
                dashboardState.applyFilters($scope.currentState.currentQuery, filterBar.getAppFilters());
                timefilter.time = $scope.currentState.currentTime;
              }
            });
        }
        // kibi: end
      };

      // kibi: added by kibi
      $scope.currentState = {};
      // kibi: end
      const onChangeViewMode = (newMode) => {
        $scope.exportIsCurrentKey = false;
        const isPageRefresh = newMode === dashboardState.getViewMode();
        const isLeavingEditMode = !isPageRefresh && newMode === DashboardViewMode.VIEW;
        const currentCoat = JSON.parse($scope.dash.coatJSON);
        const willLoseChanges = isLeavingEditMode && (dashboardState.getIsDirty(timefilter) ||
         !_.isEqual(currentCoat, originalCoat));

        if (!willLoseChanges) {
          // kibi: if user switch to edit mode store these information
          if (newMode === 'edit') {
            $scope.currentState.currentQuery = dashboardState.getQuery();
            $scope.currentState.currentAppFilters = filterBar.getAppFilters();
            $scope.currentState.currentGlobalFilters = filterBar.getGlobalFilters();
            $scope.currentState.currentTime = angular.copy(timefilter.time);
          }
          // kibi: end
          updateViewMode(newMode);
          return;
        }

        confirmModal(
          getUnsavedChangesWarningMessage(dashboardState.getChangedFilterTypes(timefilter)),
          {
            onConfirm: revertChangesAndExitEditMode,
            onCancel: _.noop,
            confirmButtonText: 'Yes, lose changes',
            cancelButtonText: 'No, keep working',
            defaultFocusedButton: ConfirmationButtonTypes.CANCEL
          }
        );
      };

      // kibi: allows to change the view mode and select the current menu option (issue #3368)
      $scope.$on('kibi:dashboardviewmode:change', (event, newMode, topNavKey) => {
        updateViewMode(newMode);
        if (topNavKey) {
          $timeout(() => {
            $scope.kbnTopNav.setCurrent(topNavKey);
          });
        }
      });
      // kibi: end

      $scope.save = function () {
        //returns a unique id for a visualization in a specific dashboard
        const getUniqueVisId = function (dashId, visTitle) {
          return dashId.replace(/:/g, '') + '-' + visTitle.replace(/\s+/g, '');
        };
        const origId = dashboardState.savedDashboard.id;
        return dashboardState.saveDashboard(angular.toJson, timefilter, $scope.model.dontSaveFiltersQuery).then(function (id) {
          $scope.kbnTopNav.close('save');
          originalCoat = JSON.parse($scope.dash.coatJSON);
          if (id) {
            return Promise.all($scope.panels.map(panel => savedVisualizations.get(panel.id)))
              .then((visualizationsOnDashboard) => {
                for (let i = 0; i < visualizationsOnDashboard.length; i++) {
                  const savedVis = visualizationsOnDashboard[i];
                  if (savedVis && savedVis.uiStateJSON) {
                    const uiStateJsonVis = JSON.parse(savedVis.uiStateJSON);
                    if (uiStateJsonVis && uiStateJsonVis.siren) {
                      const newVisUiDashId = getUniqueVisId(id, savedVis.title);
                      if (uiStateJsonVis.siren[newVisUiDashId]) {
                        return Promise.resolve();
                      }
                      const visUiDashId = getUniqueVisId(origId, savedVis.title);
                      if (uiStateJsonVis.siren[visUiDashId]) {
                        uiStateJsonVis.siren[newVisUiDashId] = uiStateJsonVis.siren[visUiDashId];
                        savedVis.uiStateJSON = JSON.stringify(uiStateJsonVis);
                        return savedVis.save().then(function (visId) {
                          if (visId) {
                            return Promise.resolve(id);
                          } else {
                            return Promise.reject('Not able to save dashboard state on visualization');
                          }
                        });
                      }
                    }
                  }
                }
                return Promise.resolve();
              })
              .then(() => {
                notify.info(`Saved Dashboard as "${dash.title}"`);
                // kibi: if filters or query not changed, don't compute counts
                if ($scope.filtersQueryTitleOrSavedSearchChanged) {
                  $rootScope.$emit('kibi:dashboard:changed', id); // kibi: allow helpers to react to dashboard changes
                  $scope.filtersQueryTitleOrSavedSearchChanged = false;
                }
                // kibi: end
                if (dash.id !== $routeParams.id) {
                  kbnUrl.change(createDashboardEditUrl(dash.id));
                } else {
                  docTitle.change(dash.lastSavedTitle);
                  updateViewMode(DashboardViewMode.VIEW);
                }
                return id;
              });
          }
          return id;
        }).catch(notify.error);
      };

      // kibi: if user select save as a new dashboard option, set dontSaveFiltersQuery option to false
      $scope.$watch('opts.dashboard.copyOnSave', function (saveAsNew) {
        if (saveAsNew) {
          $scope.model.dontSaveFiltersQuery = false;
        }
      });
      // kibi: end

      const navActions = {};
      navActions[TopNavIds.EXIT_EDIT_MODE] = () => onChangeViewMode(DashboardViewMode.VIEW);
      navActions[TopNavIds.ENTER_EDIT_MODE] = () => onChangeViewMode(DashboardViewMode.EDIT);
      navActions[TopNavIds.CLONE] = () => {
        const currentTitle = $scope.model.title;
        const onClone = (newTitle) => {
          dashboardState.savedDashboard.copyOnSave = true;
          dashboardState.setTitle(newTitle);
          return $scope.save().then(id => {
            // If the save wasn't successful, put the original title back.
            if (!id) {
              $scope.model.title = currentTitle;
              // There is a watch on $scope.model.title that *should* call this automatically but
              // angular is failing to trigger it, so do so manually here.
              dashboardState.setTitle(currentTitle);
            }
            return id;
          });
        };

        showCloneModal(onClone, currentTitle, $rootScope, $compile);
      };
      updateViewMode(dashboardState.getViewMode());

      // kibi: update root source on kibiState reset
      $scope.$listen(kibiState, 'reset_app_state_query', function (query) {
        if (query) {
          $scope.model.query = query;
        }

        $scope.filterResults();
      });
      // kibi: end

      // update root source when filters update
      $scope.$listen(filterBar, 'update', function () {
        dashboardState.applyFilters($scope.model.query, filterBar.getFilters());
      });

      // update data when filters fire fetch event
      $scope.$listen(filterBar, 'fetch', () => {
        updateState();
        $scope.refresh();
      });

      $scope.$on('$destroy', function onDashboardDestroy() {
        // if dashboard count uses vis and user switch dashboard before request response,
        // assign dashboard count to latest one then set false in useVisualizationForCountOnDashboard
        if (useVisualizationForCountOnDashboard.get(dash.id) && dash.count === SpinnerStatus.FETCHING) {
          const node = findMainCoatNode(dash.siren.coat.items);
          if (node.d.entity && node.d.entity.data && node.d.entity.data.hits && node.d.entity.data.hits.total) {
            dash.count = node.d.entity.data.hits.total;
          }
        }
        useVisualizationForCountOnDashboard.set(dash.id, false);

        dashboardGroups.off('groupsMetadataUpdated', getMetaWrapper);
        dashboardGroups.off('dashboardMetadataUpdated', getMetaWrapper);
        if (appStateSaveWithChangeHandlerOff) {
          appStateSaveWithChangeHandlerOff();
        }
        if (orFiltersPresentHandler) {
          orFiltersPresentHandler();
        }
        dashboardState.destroy();
        // Remove dark theme to keep it from affecting the appearance of other apps.
        setLightTheme();
        notify = null; // this allow to completely garbage collect the object from memory
      });

      function updateTheme() {
        dashboardState.getDarkTheme() ? setDarkTheme() : setLightTheme();
      }

      function setDarkTheme() {
        chrome.removeApplicationClass(['theme-light']);
        chrome.addApplicationClass('theme-dark');
      }

      function setLightTheme() {
        chrome.removeApplicationClass(['theme-dark']);
        chrome.addApplicationClass('theme-light');
      }

      $scope.$on('ready:vis', function () {
        if (pendingVisCount > 0) pendingVisCount--;
        if (pendingVisCount === 0) {
          dashboardState.saveState();
          $scope.refresh();
        }
      });

      if ($route.current.params && $route.current.params[DashboardConstants.NEW_VISUALIZATION_ID_PARAM]) {
        // Hide the toast message since they will already see a notification from saving the visualization,
        // and one is sufficient (especially given how the screen jumps down a bit for each unique notification).
        const showToast = false;
        $scope.addVis({ id: $route.current.params[DashboardConstants.NEW_VISUALIZATION_ID_PARAM] }, showToast);

        kbnUrl.removeParam(DashboardConstants.ADD_VISUALIZATION_TO_DASHBOARD_MODE_PARAM);
        kbnUrl.removeParam(DashboardConstants.NEW_VISUALIZATION_ID_PARAM);
      }

      const addNewVis = function addNewVis() {
        kbnUrl.change(
          `${VisualizeConstants.WIZARD_STEP_1_PAGE_PATH}?${DashboardConstants.ADD_VISUALIZATION_TO_DASHBOARD_MODE_PARAM}`);
      };

      $scope.opts = {
        displayName: dash.getDisplayName(),
        dashboard: dash,
        save: $scope.save,
        addVis: $scope.addVis,
        addNewVis,
        addSearch: $scope.addSearch,
        timefilter: $scope.timefilter
      };

      // kibi: export dashboard as pdf or open to print
      $scope.exportDashboard = (download) => {
        $scope.exportIsCurrentKey = false;
        $scope.kbnTopNav.close('export');
        $scope.dashboardExportInProgress = true;
        const elem = document.getElementById('export-dashboard-progress-bar-foreground');
        let width = 1;

        const id = setInterval(frame, 10);
        function frame() {
          if (width >= 90) {
            clearInterval(id);
          } else {
            width++;
            elem.style.width = width + '%';
          }
        };

        const options = {
          title: $scope.model.title,
          count: $scope.dash.count,
          query: $scope.model.query,
          filters: filterBar.getFilters(),
          exportWithTables: $scope.model.exportWithTables,
          exportWithQuery: $scope.model.exportWithQuery,
          toDownload: download
        };

        if ($scope.dash.indexPattern) {
          options.timefilter = {
            fieldName: $scope.dash.indexPattern.timeFieldName,
            timeRange: $scope.timefilter.time
          };
        };

        const savedVisualization = Private(savedVisualizationProvider);
        const dashPanelsToExport = [];
        Promise.all($scope.panels.map(panel => savedVisualization.get(panel.id)))
          .then((visualizationsOnDashboard) => {
            if ($scope.expandedPanel !== null) {
              const vis = _.find(visualizationsOnDashboard, visualization => visualization.id === $scope.expandedPanel.id);
              dashPanelsToExport.push({
                vis: vis.vis,
                $element: $document.find('dashboard-panel[panel="expandedPanel"]'),
                expanded: true,
                id: vis.id
              });
            } else {
              const usedIndexes = [];
              visualizationsOnDashboard.forEach(vis => {
                const panel = _.find($scope.panels, panel => panel.id === vis.id
                  && usedIndexes.indexOf(panel.panelIndex) === -1);
                usedIndexes.push(panel.panelIndex);
                dashPanelsToExport.push({
                  vis: vis.vis,
                  $element: $document.find('#dashboardPanel-' + panel.panelIndex),
                  expanded: false,
                  id: vis.id
                });
              });
            };
            exportDashboardHelper(dashPanelsToExport, options)
              .then(() => {
                $scope.dashboardExportInProgress = false;
              });
          }).catch((e) => {
            notify.error(e);
            $scope.dashboardExportInProgress = false;
          });
      };
      //kibi: end



      // kibi: Merge the parameters saved on kibi_appstate_param
      const passedState = JSON.parse(HashedItemStoreSingleton.getItem('kibi_appstate_param'));
      if (passedState && passedState.dashboardOptions) {
        _.assign($scope.opts.dashboard, _.cloneDeep(passedState.dashboardOptions));
        delete passedState.dashboardOptions;
        HashedItemStoreSingleton.setItem('kibi_appstate_param', JSON.stringify(passedState));
      }
      // kibi: end

      $scope.$emit('application.load');
    }
  };
});
