import { uiModules } from 'ui/modules';
import chrome from 'ui/chrome';
import angular from 'angular';
import React from 'react';
import { render, unmountComponentAtNode } from 'react-dom';
import template from '../partials/gremlin_console.html';
import copy from 'copy-to-clipboard';

import {
  EuiCallOut,
  EuiInMemoryTable,
  EuiPanel,
  EuiPage,
  EuiPageBody,
  EuiSpacer,
  EuiPageContentBody,
  EuiToolTip
} from '@elastic/eui';
const app = uiModules.get('kibana');

app.directive('gremlinConsole', function (queryEngineClient, createNotifier, ontologyModel, config) {
  const NOTIFY = createNotifier('Gremlin Console');
  return {
    template,
    controller: ($scope, $element) => {
      const copyToClipboard = (text) => {
        copy(text);
        NOTIFY.info('Copied to Clipboard!');
      };
      const relationTable = $element.find('#gremlinRelationTable')[0];
      const columns = [
        {
          name: 'Source Entity',
          field: 'sourceEntity'
        },
        {
          name: 'Relation',
          render: (item) => {
            return (
              <div>
                <p>{item.relation}</p>
                <p>
                  <EuiToolTip position="right" content="Copy to Clipboard">
                    <a onClick={() => copyToClipboard(item.relationId)}>{item.relationId}</a>
                  </EuiToolTip>
                </p>
              </div>
            );
          }
        },
        {
          name: 'Destination Entity',
          field: 'destEntity'
        },
        {
          name: 'Inverse Relation',
          render: (item) => {
            return (
              <div>
                <p>{item.inverseRelation}</p>
                <p>
                  <EuiToolTip position="right" content="Copy to Clipboard">
                    <a onClick={() => copyToClipboard(item.inverseRelationId)}>{item.inverseRelationId}</a>
                  </EuiToolTip>
                </p>
              </div>
            );
          }
        }
      ];
      const search = {
        box: {
          incremental: true,
          schema: true
        }
      };
      const pagination = {
        initialPageSize: 5,
        pageSizeOptions: [5, 10, 20]
      };
      $scope.init = () => {
        ontologyModel.getRelationList()
          .then((res) => {
            $scope.relations = [];
            res.forEach((ele) => {
              $scope.relations.push({
                sourceEntity: ele.domain.label,
                destEntity: ele.range.label,
                relation: ele.directLabel,
                relationId: ele.id,
                inverseRelation: ele.inverseLabel,
                inverseRelationId: ele.inverseOf
              });
            });
            render(
              <div>
                <EuiPage>
                  <EuiPageBody>
                    <EuiPageContentBody>
                      <EuiPanel
                        paddingSize="m"
                        hasShadow
                        betaBadgeLabel="Relations"
                        betaBadgeTitle="Relation Search Table"
                      >
                        <EuiCallOut
                          title={
                            'The Siren gremlin requires you to specify the relation using their Ids. ' +
                            'Use this table to search the required relation ids.'
                          }
                          size="s"
                          iconType="graphApp"
                        />
                        <EuiSpacer size="s" />
                        <EuiInMemoryTable
                          items={$scope.relations}//.concat(configsArray)}
                          itemId="id"
                          columns={columns}
                          search={search}
                          pagination={pagination}
                        />
                      </EuiPanel>
                    </EuiPageContentBody>
                  </EuiPageBody>
                </EuiPage>
              </div>,
              relationTable);
          });
      };
      $scope.init();
      $scope.$on('$destroy', () => unmountComponentAtNode(relationTable));
      const setQuery = (query) => {
        $scope.gremlinQuery = query;
      };
      const setResult = (result) => {
        if (typeof result === "object") {
          try {
            $scope.gremlinResult = JSON.stringify(result, null, 2);
            return $scope.gremlinResult;
          } catch (e) {
            //Do Nothing
          }
        }
        $scope.gremlinResult = result;
        return $scope.gremlinResult;
      };
      $scope.testGremlinQuery = () => {
        $scope.clear();
        const jsonData = {
          queries: [$scope.gremlinQuery],
          limit: config.get('siren:graphExpansionLimit'),
        };
        queryEngineClient.gremlin({
          method: 'POST',
          headers: {
            'Accept': 'application/json',
            'Content-Type': 'application/json; charset=UTF-8'
          },
          json: jsonData
        }).then((results) => {
          if (results.data) {
            if (results.data.constructor === Array && results.data[0].constructor === Array) {
              return setResult(results.data);
            }
            if (results.data.error) {
              return setResult(results.data.error);
            }
          }
          if (results.error) {
            return setResult(results.error);
          }
          return setResult(`Something went wrong while executing script: ${$scope.gremlinQuery} .`);
        }).catch(({ data }) => {
          return setResult(`${data.statusCode} | ${data.error} | ${data.message}`);
        });
      };
      $scope.clear = () => {
        $scope.gremlinResult = '';
      };
      setQuery('g.V("company/Company/Jh62o2YBu8ppBliP6u_w").bothE("80f63ed7-00f0-489f-8cf5-ee2cfbbb83fd").count()');
    }
  };
});
