import _ from 'lodash';
import moment from 'moment';
import 'ui/paginated_table';
import fieldNameHtml from './field_name.html';
import fieldTypeHtml from './field_type.html';
import fieldControlsHtml from '../field_controls.html';
import { uiModules } from 'ui/modules';
import { FieldWildcardProvider } from 'ui/field_wildcard';
import template from './indexed_fields_table.html';

// kibi: imports
import './indexed_fields_table.less';
import joinFields from 'plugins/investigate_core/management/sections/indices/join_fields';
import { SavedObjectsClientProvider } from 'ui/saved_objects';
import { FingerprintsProvider } from 'ui/kibi/quick_relations/fingerprints';
// kibi: end

uiModules.get('apps/management')
  .directive('indexedFieldsTable', function (Private, $filter, indexPatterns, $document, ontologyModel, Notifier) {
    const yesTemplate = '<i class="fa fa-check" aria-label="yes"></i>';
    const noTemplate = '';
    const filter = $filter('filter');
    const { fieldWildcardMatcher } = Private(FieldWildcardProvider);
    const savedObjectsClient = Private(SavedObjectsClientProvider);
    const fingerprints = Private(FingerprintsProvider);
    const notify = new Notifier({
      location: 'Indexed Fields Table'
    });

    return {
      restrict: 'E',
      template,
      scope: true,
      link: function ($scope, element) {
        const rowScopes = []; // track row scopes, so they can be destroyed as needed
        $scope.perPage = 25;

        // kibi: Added alternative 'fingerprints' view
        const standardColumns = [
          { title: 'name' },
          { title: 'type' },
          { title: 'format' },
          { title: 'searchable', info: 'These fields can be used in the filter bar' },
          { title: 'aggregatable' , info: 'These fields can be used in visualization aggregations' },
          { title: 'excluded', info: 'Fields that are excluded from _source when it is fetched' },
          // kibi: added primary key and single value properties
          { title: 'primary key', info: 'Fields that are a primary key of an Entity Identifier', sortable: false },
          { title: 'single value', info: 'Fields that are not arrays', sortable: false },
          { title: 'controls', sortable: false }
        ];

        const fingerprintColumns = [
          { title: 'name' },
          { title: 'type' },
          { title: 'tags' },
          { title: 'cardinality' },
          { title: 'range' },
          { title: 'samples' }
        ];

        const DEFAULT_FIELDS_TAB_VIEW = 'default';
        $scope.tabView = DEFAULT_FIELDS_TAB_VIEW;

        $scope.toggleTabView = function (tabView) {
          $scope.tabView = ($scope.tabView === tabView) ? DEFAULT_FIELDS_TAB_VIEW : tabView;
        };

        // kibi: changed watcher from tabView -> selectedTab
        $scope.$watch('tabView', updateSelectedTab);

        function updateSelectedTab(tabView) {
          refreshRows()
            .then(() => {
              $scope.columns = (tabView === DEFAULT_FIELDS_TAB_VIEW)
                ? standardColumns
                : fingerprintColumns;
            });
        }

        $scope.$on('aboutToRefreshFieldsList', function setDefaultView() {
        // Staying in the fingerprints view would trigger a fingerprints update, which
        // is already handled by the fields list refresh.

          $scope.tabView = DEFAULT_FIELDS_TAB_VIEW;
        });

        $scope.$on('refreshedFieldsList', refreshRows);
        // kibi: end

        $scope.$watchMulti(['[]indexPattern.fields', 'fieldFilter', 'indexedFieldTypeFilter'], refreshRows);

        function refreshRows() {
        // clear and destroy row scopes
          _.invoke(rowScopes.splice(0), '$destroy');
          const fields = filter($scope.indexPattern.getNonScriptedFields(), {
            name: $scope.fieldFilter,
            type: $scope.indexedFieldTypeFilter
          });

          // kibi: View-specific columns moved in a separate function

          // kibi: get 'relations' from ontology
          return ontologyModel.getRelationList()
            .then(relations => fields.map(function (field) {
              const childScope = _.assign($scope.$new(), {
                field: field,
                // kibi: add information about joined fields
                joinedTo: joinFields(relations, $scope.indexPattern.id, field.name)
              });
              rowScopes.push(childScope);

              return [
                {
                  markup: fieldNameHtml,
                  scope: childScope,
                  value: field.displayName,
                  attr: {
                    'data-test-subj': 'indexedFieldName'
                  }
                },
                {
                  markup: fieldTypeHtml,
                  scope: childScope,
                  value: field.type,
                  attr: {
                    'data-test-subj': 'indexedFieldType'
                  }
                }
              ];
            }))
            // kibi: Concatenate view-specific columns
            .then(rows => {
              const concatRowsPromise = ($scope.tabView === DEFAULT_FIELDS_TAB_VIEW)
                ? standardRowData(fields)
                : fingerprintRowData(fields);

              return concatRowsPromise
                .then(concatRows => {
                  $scope.rows = _(rows)
                    .zip(concatRows)
                    .map(_.flatten)
                    .value();
                });
            });
        // kibi: end
        }

        // kibi: View-specific columns preparation

        function standardRowData(fields) {
          const sourceFilters = $scope.indexPattern.sourceFilters && $scope.indexPattern.sourceFilters.map(f => f.value) || [];
          const fieldWildcardMatch = fieldWildcardMatcher(sourceFilters);
          _.find($scope.editSections, { index: 'indexedFields' }).count = fields.length; // Update the tab count

          return Promise.resolve(
            fields.map((field, f) => {
              const childScope = rowScopes[f];
              const excluded = fieldWildcardMatch(field.name);

              return [
                _.get($scope.indexPattern, ['fieldFormatMap', field.name, 'type', 'title']),
                {
                  markup: field.searchable ? yesTemplate : noTemplate,
                  value: field.searchable
                },
                {
                  markup: field.aggregatable ? yesTemplate : noTemplate,
                  value: field.aggregatable
                },
                {
                  markup: excluded ? yesTemplate : noTemplate,
                  value: excluded
                },
                // kibi: added primary key and single value columns
                {
                  markup: field.primaryKey ? yesTemplate : noTemplate,
                  value: field.primaryKey
                },
                {
                  markup: field.singleValue ? yesTemplate : noTemplate,
                  value: field.singleValue
                },
                {
                  markup: fieldControlsHtml,
                  scope: childScope
                }
              ];
            })
          );
        }

        function getFingerprints(indexPattern) {
          const fpDocId = 'fingerprint:' + indexPattern.title;

          return savedObjectsClient
            .get('fingerprint', fpDocId)                        // Can fail with 404
            .then(sobj => sobj.attributes.version === fingerprints.currentVersion
              ? JSON.parse(sobj.attributes.json)                // Can throw if bad doc
              : Promise.reject());
        }

        function fieldTermsFormatter(field) {
          switch (field.type) {
            case 'date':
            // Abbreviated datetime formatting with minutes resolution
            // e.g.: Sun, Sep 8, 2013 2:00 AM
              return dateVal => moment(dateVal).format('llll');
          }

          return _.identity;
        }

        function fingerprintRowData(fields) {
        // Adopting a 'soft' policy for errors on document retrieval - we'll
        // assume an empty fingerprint document in case no fingerprints for the
        // index was found (404) or the retrieved document was malformed (shouldn't
        // happen, but just to be sure)

          const { indexPattern } = $scope;

          return getFingerprints(indexPattern)
            .catch(() => Promise.resolve([ indexPattern ])
            // Try to regenerate index pattern fingerprints
              .then(fingerprints.selectIndexPatterns)
              .then(fingerprints.calculate)
              .then(() => notify.info("Fingerprints generated successfully"))
              .catch(err => { err && notify.error(err); })
              .then(() => getFingerprints(indexPattern)))
            .catch(_.noop)                                      // Fallback to empty doc
            .then(indexFp => indexFp || {})
            .then(function formatIndexFingerprints(indexFp) {
              return fields.map(function formatFieldFingeprints(field) {
                const termsFormatter = fieldTermsFormatter(field);

                // Applying empty document structure if no data is present
                const fieldFp = _.defaults({}, indexFp[field.name], {
                  tags: [],
                  attributes: {},
                  terms: []
                });

                // Using a dash in place of missing values
                const tags = _(fieldFp.tags).keys().join(', ') || '-';
                const cardinality = fieldFp.attributes.cardinality || '-';
                const range = {
                  markup: (
                    fieldFp.attributes.range &&
                  _(fieldFp.attributes.range)
                    .map(termsFormatter)
                    .join('<br>..<br>')
                  ) || '-'
                };
                const terms = {
                  markup: _(fieldFp.terms)
                    .slice(0, 5)
                    .map(termsFormatter)
                    .map(_.escape)
                    .join('<br>') || '-'
                };

                return [ tags, cardinality, range, terms ];
              });
            });
        }

        // kibi: end

        // kibi: fix table legend on top on scrolling
        $document.on('scroll.fixTableLegend', () => {
          const HEIGHT_TO_ADD_FIXED_LEGEND = 0;
          const tableContainer = element.find('paginate');

          // this condition executes logic when table was rendered
          if (tableContainer[0]) {
            const tableLegend = element.find('paginate thead');
            const tableLegendColumns = tableLegend.find('th');
            const clonedTableLegend = element.find('.cloned-table-legend');
            const topTablePosition = tableContainer.offset().top - $document.scrollTop();

            if (topTablePosition < HEIGHT_TO_ADD_FIXED_LEGEND && !tableLegend.hasClass('siren-fixed-table-legend')) {
            // use cloned element to keep width for columns
              tableLegend.clone().addClass('cloned-table-legend').insertBefore(tableLegend);
              // keep fixed width for columns of legend
              tableLegendColumns.each(columnIndex => {
                const tableLegendColumn = tableLegendColumns.eq(columnIndex);
                tableLegendColumn.css('width', tableLegendColumn.outerWidth());
              });
              // fix legend on top of the window
              tableLegend.addClass('siren-fixed-table-legend');
            } else if (topTablePosition > HEIGHT_TO_ADD_FIXED_LEGEND && tableLegend.hasClass('siren-fixed-table-legend')) {
            // return original position of legend
              tableLegend.removeClass('siren-fixed-table-legend');
              // remove cloned element
              clonedTableLegend.remove();
              // remove fixed width for columns of legend
              tableLegendColumns.each(columnIndex => {
                tableLegendColumns.eq(columnIndex).css('width', '');
              });
            };
          };
        });

        $scope.$on('$destroy', () => {
          $document.off('.fixTableLegend');
        });
      // kibi: end
      }
    };
  });
