'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Url = undefined;

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _highlight_html = require('../helpers/highlight/highlight_html');

var _field_format = require('./field_format');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const whitelistUrlSchemes = ['http://', 'https://'];

class Url extends _field_format.FieldFormat {
  constructor(params) {
    super(params);
    this._compileTemplate = _lodash2.default.memoize(this._compileTemplate);
  }

  _formatUrl(value) {
    const template = this.param('urlTemplate');
    if (!template) return value;

    return this._compileTemplate(template)({
      value: encodeURIComponent(value),
      rawValue: value
    });
  }

  _formatLabel(value, url = this._formatUrl(value)) {
    const template = this.param('labelTemplate');
    if (!template) return url;

    return this._compileTemplate(template)({
      value: value,
      url: url
    });
  }

  _compileTemplate(template) {
    const parts = template.split(Url.templateMatchRE).map(function (part, i) {
      // trim all the odd bits, the variable names
      return i % 2 ? part.trim() : part;
    });

    return function (locals) {
      // replace all the odd bits with their local var
      let output = '';
      let i = -1;
      while (++i < parts.length) {
        if (i % 2) {
          if (locals.hasOwnProperty(parts[i])) {
            const local = locals[parts[i]];
            output += local == null ? '' : local;
          }
        } else {
          output += parts[i];
        }
      }

      return output;
    };
  }
}

exports.Url = Url;
Url.id = 'url';
Url.title = 'Url';
Url.fieldType = ['number', 'boolean', 'date', 'ip', 'string', 'murmur3', 'unknown', 'conflict'];

Url.templateMatchRE = /{{([\s\S]+?)}}/g;
Url.paramDefaults = {
  type: 'a',
  urlTemplate: null,
  labelTemplate: null
};

Url.urlTypes = [{ id: 'a', name: 'Link' }, { id: 'img', name: 'Image' }];

Url.prototype._convert = {
  text: function (value) {
    return this._formatLabel(value);
  },
  html: function (rawValue, field, hit, parsedUrl) {
    const url = _lodash2.default.escape(this._formatUrl(rawValue));
    const label = _lodash2.default.escape(this._formatLabel(rawValue, url));

    switch (this.param('type')) {
      case 'img':
        // kibi: adding error images for empty image url and any other error
        const error = rawValue === '' ? 'Empty image url' : 'Error loading image';
        const errorImage = `<svg xmlns:svg="http://www.w3.org/2000/svg" xmlns="http://www.w3.org/2000/svg" version="1.1" height="24" width="116">
        <g>
        <rect x="0" y="0" width="116" height="24" stroke="#dedede" fill="#ffffff"></rect>
        <text x="8" y="17" font-size="12" fill="#2d2d2d" font-family="Open Sans">${error}</text>
        </g>
        </svg>`;

        const encodedErrorImage = 'data:image/svg+xml,' + encodeURIComponent(errorImage);

        return `<img src="${url}" alt="${label}" title="${label}" onerror="this.onerror=null;this.src='${encodedErrorImage}'">`;
      // kibi: end
      default:
        const inWhitelist = whitelistUrlSchemes.some(scheme => url.indexOf(scheme) === 0);
        if (!inWhitelist && !parsedUrl) {
          return url;
        }

        let prefix = '';
        /**
         * This code attempts to convert a relative url into a kibana absolute url
         *
         * SUPPORTED:
         *  - /app/kibana/
         *  - ../app/kibana
         *  - #/discover
         *
         * UNSUPPORTED
         *  - app/kibana
         */
        if (!inWhitelist) {
          // Handles urls like: `#/discover`
          if (url[0] === '#') {
            prefix = `${parsedUrl.origin}${parsedUrl.pathname}`;
          }
          // Handle urls like: `/app/kibana` or `/xyz/app/kibana`
          else if (url.indexOf(parsedUrl.basePath || '/') === 0) {
              prefix = `${parsedUrl.origin}`;
            }
            // Handle urls like: `../app/kibana`
            else {
                prefix = `${parsedUrl.origin}${parsedUrl.basePath}/app/`;
              }
        }

        let linkLabel;

        if (hit && hit.highlight && hit.highlight[field.name]) {
          linkLabel = (0, _highlight_html.getHighlightHtml)(label, hit.highlight[field.name]);
        } else {
          linkLabel = label;
        }

        return `<a href="${prefix}${url}" target="_blank">${linkLabel}</a>`;
    }
  }
};
