'use strict';

var _plugin_api = require('./plugin_api');

var _plugin_api2 = _interopRequireDefault(_plugin_api);

var _util = require('util');

var _lodash = require('lodash');

var _collection = require('../../utils/collection');

var _collection2 = _interopRequireDefault(_collection);

var _transform_deprecations = require('../config/transform_deprecations');

var _deprecation = require('../../deprecation');

var _joi = require('joi');

var _joi2 = _interopRequireDefault(_joi);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const byIdCache = Symbol('byIdCache');
const pluginApis = Symbol('pluginApis');

async function addPluginConfig(pluginCollection, plugin) {
  const { config, server, settings } = pluginCollection.kbnServer;

  const transformedSettings = (0, _transform_deprecations.transformDeprecations)(settings);
  const pluginSettings = (0, _lodash.get)(transformedSettings, plugin.configPrefix);
  const deprecations = plugin.getDeprecations();
  const transformedPluginSettings = (0, _deprecation.createTransform)(deprecations)(pluginSettings, message => {
    server.log(['warning', plugin.configPrefix, 'config', 'deprecation'], message);
  });

  const configSchema = await plugin.getConfigSchema();
  config.extendSchema(configSchema, transformedPluginSettings, plugin.configPrefix);
}

function disablePluginConfig(pluginCollection, plugin) {
  // when disabling a plugin's config we remove the existing schema and
  // replace it with a simple schema/config that only has enabled set to false
  const { config } = pluginCollection.kbnServer;
  config.removeSchema(plugin.configPrefix);
  const schema = _joi2.default.object({ enabled: _joi2.default.bool() });
  config.extendSchema(schema, { enabled: false }, plugin.configPrefix);
}

module.exports = class Plugins extends _collection2.default {

  constructor(kbnServer) {
    super();
    this.kbnServer = kbnServer;
    this[pluginApis] = new Set();
  }

  async new(path) {
    const api = new _plugin_api2.default(this.kbnServer, path);
    this[pluginApis].add(api);

    const output = [].concat(require(path)(api) || []);

    if (!output.length) return;

    // clear the byIdCache
    this[byIdCache] = null;

    for (const plugin of output) {
      if (!plugin instanceof api.Plugin) {
        throw new TypeError('unexpected plugin export ' + (0, _util.inspect)(plugin));
      }

      await addPluginConfig(this, plugin);
      this.add(plugin);
    }
  }

  async disable(plugin) {
    disablePluginConfig(this, plugin);
    this.delete(plugin);
  }

  get byId() {
    return this[byIdCache] || (this[byIdCache] = (0, _lodash.indexBy)([...this], 'id'));
  }

  getPluginApis() {
    return this[pluginApis];
  }

};
