'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedObjectsClient = exports.V6_TYPE = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; // kibi: all crud methods have been modified to go through the legacy
// saved objects API model in order to support middleware and mappings.

var _boom = require('boom');

var _boom2 = _interopRequireDefault(_boom);

var _uuid = require('uuid');

var _uuid2 = _interopRequireDefault(_uuid);

var _lodash = require('lodash');

var _lib = require('./lib');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const V6_TYPE = exports.V6_TYPE = 'doc';

class SavedObjectsClient {
  constructor(kibanaIndex, mappings, callAdminCluster, savedObjectsApi) {
    this.errors = _lib.errors;

    this._kibanaIndex = kibanaIndex;
    this._mappings = mappings;
    this._callAdminCluster = callAdminCluster;
    // kibi: added by kibi
    this._savedObjectsApi = savedObjectsApi;
  }

  getStatus() {
    return this._savedObjectsApi.status;
  }

  /**
   * Persists an object
   *
   * @param {string} type
   * @param {object} attributes
   * @param {object} [options={}]
   * @property {string} [options.id] - force id on creation, not recommended
   * @property {boolean} [options.overwrite=false]
   * @returns {promise} - { id, type, version, attributes }
  */
  async create(type, attributes = {}, options = {}, req) {
    const model = this._savedObjectsApi.getModel(type);
    let id = '';
    if (options.id) {
      id = options.id.startsWith(`${type}:`) ? options.id : `${type}:${options.id}`;
    } else {
      id = `${type}:${_uuid2.default.v1()}`;
    }

    const body = {
      type,
      [type]: attributes
    };

    let response;
    // Note: try/catch block needed to correctly propagate errors
    try {
      if (options.id && !options.overwrite) {
        response = await model.create(id, body, req);
      } else {
        response = await model.update(id, body, req);
      }
    } catch (error) {
      throw (0, _lib.decorateEsError)(error);
    }

    return (0, _lib.normalizeEsDoc)(response, { type, attributes });
  }

  /**
   * Creates multiple documents at once
   *
   * @param {array} objects - [{ type, id, attributes }]
   * @param {object} [options={}]
   * @property {boolean} [options.overwrite=false] - overrides existing documents
   * @property {string} [options.format=v5]
   * @returns {promise} - [{ id, type, version, attributes, error: { message } }]
   */
  async bulkCreate(objects, options = {}) {
    const { format = 'v5' } = options;

    const bulkCreate = format === 'v5' ? _lib.v5BulkCreate : _lib.v6BulkCreate;
    const response = await this._withKibanaIndex('bulk', {
      body: bulkCreate(objects, options),
      refresh: 'wait_for'
    });

    const items = (0, _lodash.get)(response, 'items', []);
    const missingTypesCount = items.filter(item => {
      const method = Object.keys(item)[0];
      return (0, _lodash.get)(item, `${method}.error.type`) === 'type_missing_exception';
    }).length;

    const formatFallback = format === 'v5' && items.length > 0 && items.length === missingTypesCount;

    if (formatFallback) {
      return this.bulkCreate(objects, Object.assign({}, options, { format: 'v6' }));
    }

    return (0, _lodash.get)(response, 'items', []).map((resp, i) => {
      const method = Object.keys(resp)[0];
      const { type, attributes } = objects[i];

      return (0, _lib.normalizeEsDoc)(resp[method], {
        id: resp[method]._id,
        type,
        attributes,
        error: resp[method].error ? { message: (0, _lodash.get)(resp[method], 'error.reason') } : undefined
      });
    });
  }

  /**
   * Deletes an object
   *
   * @param {string} type
   * @param {string} id
   * @returns {promise}
   */
  async delete(type, id, req) {
    const docId = id.startsWith(`${type}:`) ? id : `${type}:${id}`;
    const response = await this._withKibanaIndex('delete', {
      id: docId,
      type: type,
      refresh: 'wait_for'
    }, req);

    if ((0, _lodash.get)(response, 'deleted') === 0) {
      throw _lib.errors.decorateNotFoundError(_boom2.default.notFound());
    }
  }

  /**
   * @param {object} [options={}]
   * @property {string} options.type
   * @property {string} options.search
   * @property {string} options.searchFields - see Elasticsearch Simple Query String
   *                                        Query field argument for more information
   * @property {integer} [options.page=1]
   * @property {integer} [options.perPage=20]
   * @property {string} options.sortField
   * @property {string} options.sortOrder
   * @property {array|string} options.fields
   * @returns {promise} - { saved_objects: [{ id, type, version, attributes }], total, per_page, page }
   */
  async find(options = {}, req) {
    const {
      type,
      search,
      searchFields,
      page = 1,
      perPage = 20,
      sortField,
      sortOrder,
      fields,
      exclude, // kibi: added,
      excludedIndices, // kibi: added
      ignoreBroken
    } = options;

    const esOptions = {
      // kibi: start added support for exclude option
      _source: {
        includes: (0, _lib.includedFields)(type, fields),
        excludes: exclude ? exclude : []
      },
      // kibi: end
      type: options.type,
      size: perPage,
      from: perPage * (page - 1),
      options: {
        search, searchFields, type, sortField, sortOrder
      },
      createFindQuery: function (mapping) {
        return (0, _lib.createFindQuery)(mapping, {
          search: this.options.search,
          searchFields: this.options.searchFields,
          type: this.options.type,
          sortField: this.options.sortField,
          sortOrder: this.options.sortOrder
        });
      },
      excludedIndices
    };

    // kibi: search using the model
    const model = this._savedObjectsApi.getModel(type);
    const response = await model.search(esOptions.size, esOptions, req, esOptions._source.excludes, ignoreBroken);
    // kibi: end

    return {
      saved_objects: (0, _lodash.get)(response, 'hits.hits', []).map(hit => {
        return (0, _lib.normalizeEsDoc)(hit);
      }),
      total: (0, _lodash.get)(response, 'hits.total', 0),
      per_page: perPage,
      page

    };
  }
  /**
   * Returns an array of objects and counts
   *
   * @param {array} objects - an array of types to get the count of saved objects for each
   * @param {object} request - request object with server credentials (not the current investigate user)
   * @returns {promise} - [ { dashboard: 10, index-pattern: 15 }]
   * @example
   *
   * bulkCount([{ type: 'config' }, { type: 'index-pattern' }], request)
  */
  async bulkCount(objects = [], request) {
    if (objects.length === 0) {
      return {};
    }

    const promises = [];

    for (const object of objects) {
      if (!object.type) {
        throw new Error('Type not specified in bulkCount request');
      }

      promises.push(this._withKibanaIndex('search', {
        type: object.type,
        size: 0
      }, request).then(objects => {
        return { [object.type]: objects.hits.total };
      }).catch(error => {
        switch (error.status) {
          case 404:
          case 403:
            return { [object.type]: 0 };
          default:
            throw error;
        }
      }));
    }

    return Promise.all(promises);
  }

  /**
   * Returns an array of objects by id
   *
   * @param {array} objects - an array ids, or an array of objects containing id and optionally type
   * @returns {promise} - { saved_objects: [{ id, type, version, attributes }] }
   * @example
   *
   * bulkGet([
   *   { id: 'one', type: 'config' },
   *   { id: 'foo', type: 'index-pattern' }
   * ])
   */
  async bulkGet(objects = [], request) {
    // kibi: the logic of bulkGet has been altered to send a request
    // for each document for compatibility with the document based middleware.
    if (objects.length === 0) {
      return { saved_objects: [] };
    }

    const responses = [];
    for (const doc of objects) {
      // NOTE: type is required by the routes, however the description of
      // the method states that it is optional, so we check that it is set
      // explicitly to catch that.
      if ((0, _lodash.isEmpty)(doc.type)) {
        throw new Error('Type not specified in bulkGet request.');
      }

      try {
        responses.push((await this.get(doc.type, doc.id, request)));
      } catch (error) {
        let errorBody;
        switch (error.status) {
          case 404:
            errorBody = {
              type: 'not_found',
              reason: error.message,
              status: 404,
              found: false
            };
            break;
          case 403:
            errorBody = {
              type: 'security_exception',
              reason: error.message,
              status: 403,
              found: false
            };
            break;
          case 401:
            throw error;
          default:
            throw error; // TODO: fatal errors do not seem to be handled in
          // discover, only logged to console.
        }
        responses.push((0, _lodash.merge)({ error: errorBody }, doc));
      }
    }

    return { saved_objects: responses };
  }

  /**
   * Gets a single object
   *
   * @param {string} type
   * @param {string} id
   * @returns {promise} - { id, type, version, attributes }
   */
  async get(type, id, req) {
    // kibi: use saved objects API model
    const model = this._savedObjectsApi.getModel(type);
    // Note: try/catch block needed to correctly propagate errors
    try {
      const docId = id.startsWith(`${type}:`) ? id : `${type}:${id}`;
      const response = await model.get(docId, req);
      return (0, _lib.normalizeEsDoc)(response);
    } catch (error) {
      throw (0, _lib.decorateEsError)(error);
    }
    // kibi: end
  }

  /**
   * Updates an object
   *
   * @param {string} type
   * @param {string} id
   * @param {object} [options={}]
   * @property {integer} options.version - ensures version matches that of persisted object
   * @returns {promise}
   */
  async update(type, id, attributes, options = {}, req) {
    const model = this._savedObjectsApi.getModel(type);
    id = id.replace(`${type}:`, '');
    const V6_ID = `${type}:${id}`;
    const body = {
      type,
      [type]: attributes
    };

    // Note: try/catch block needed to correctly propagate errors
    try {
      const response = await model.update(V6_ID, body, req);
      return (0, _lib.normalizeEsDoc)(response, { id: V6_ID, type, attributes });
    } catch (error) {
      throw (0, _lib.decorateEsError)(error);
    }
  }

  /**
   * Partially updates an existing object
   *
   * @param {string} type
   * @param {string} id
   * @param {object} [options={}]
   * @property {integer} options.version - ensures version matches that of persisted object
   * @returns {promise}
   */
  async patch(type, id, attributes, options = {}, req) {
    const model = this._savedObjectsApi.getModel(type);
    const documentId = id.startsWith(`${type}:`) ? id : `${type}:${id}`;

    // Note: try/catch block needed to correctly propagate errors
    try {
      const response = await model.patch(documentId, attributes, req);
      return (0, _lib.normalizeEsDoc)(response, { id, type, attributes });
    } catch (error) {
      throw (0, _lib.decorateEsError)(error);
    }
  }

  async _withKibanaIndex(method, params, req) {
    try {
      let model;
      switch (method) {
        case 'bulk':
          // TODO implement in our model and call our model
          return await this._callAdminCluster('bulk', _extends({}, params, {
            index: this._kibanaIndex
          }));
        case 'delete':
          model = this._savedObjectsApi.getModel(params.type);
          return await model.delete(params.id, _lib.createIdQuery, req);
        case 'search':
          return await this._callAdminCluster('search', {
            body: _extends({}, (0, _lib.createFindQuery)(null, params)),
            index: this._kibanaIndex
          });
        case 'msearch':
          throw new Error('plain msearch is not supported');
      }
    } catch (err) {
      throw (0, _lib.decorateEsError)(err);
    }
  }

  // kibi: added function
  /**
   * Gets the format of the requested document
   *
   * @returns {promise}
   */
  async getDocumentFormat(req) {
    const model = this._savedObjectsApi.getModel('config');

    try {
      const format = await model.getDocumentFormat(req);
      return {
        statusCode: 200,
        version: format
      };
    } catch (error) {
      if (error.status === 404) {
        // no config document found
        return {
          statusCode: 404
        };
      } else {
        throw error;
      }
    };
  }
  // kibi: end
}
exports.SavedObjectsClient = SavedObjectsClient;
SavedObjectsClient.errors = _lib.errors;
