'use strict';

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _fs_stubs = require('./__tests__/__mocks__/_fs_stubs');

var _cgroup = require('./cgroup');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

jest.mock('fs', () => ({
  readFile: jest.fn()
}));

describe('Control Group', function () {
  const fsStub = (0, _fs_stubs.cGroups)();

  beforeAll(() => {
    _fs2.default.readFile.mockImplementation(_fs_stubs.readFileMock);
  });

  afterEach(() => {
    (0, _fs_stubs.setMockFiles)();
  });

  describe('readControlGroups', () => {
    it('parses the file', async () => {
      (0, _fs_stubs.setMockFiles)({ '/proc/self/cgroup': fsStub.cGroupContents });
      const cGroup = await (0, _cgroup.readControlGroups)();

      expect(cGroup).toEqual({
        freezer: '/',
        net_cls: '/',
        net_prio: '/',
        pids: '/',
        blkio: '/',
        memory: '/',
        devices: '/user.slice',
        hugetlb: '/',
        perf_event: '/',
        cpu: `/${fsStub.hierarchy}`,
        cpuacct: `/${fsStub.hierarchy}`,
        cpuset: `/${fsStub.hierarchy}`,
        'name=systemd': '/user.slice/user-1000.slice/session-2359.scope'
      });
    });
  });

  describe('readCPUStat', () => {
    it('parses the file', async () => {
      (0, _fs_stubs.setMockFiles)({ '/sys/fs/cgroup/cpu/fakeGroup/cpu.stat': fsStub.cpuStatContents });
      const cpuStat = await (0, _cgroup.readCPUStat)('fakeGroup');

      expect(cpuStat).toEqual({
        number_of_elapsed_periods: 0,
        number_of_times_throttled: 10,
        time_throttled_nanos: 20
      });
    });

    it('returns default stats for missing file', async () => {
      (0, _fs_stubs.setMockFiles)();
      const cpuStat = await (0, _cgroup.readCPUStat)('fakeGroup');

      expect(cpuStat).toEqual({
        number_of_elapsed_periods: -1,
        number_of_times_throttled: -1,
        time_throttled_nanos: -1
      });
    });
  });

  describe('getAllStats', () => {
    it('can override the cpu group path', async () => {
      (0, _fs_stubs.setMockFiles)({
        '/proc/self/cgroup': fsStub.cGroupContents,
        [`${fsStub.cpuAcctDir}/cpuacct.usage`]: '357753491408',
        '/sys/fs/cgroup/cpu/docker/cpu.cfs_period_us': '100000',
        '/sys/fs/cgroup/cpu/docker/cpu.cfs_quota_us': '5000',
        '/sys/fs/cgroup/cpu/docker/cpu.stat': fsStub.cpuStatContents
      });

      const stats = await (0, _cgroup.getAllStats)({ cpuPath: '/docker' });

      expect(stats).toEqual({
        cpuacct: {
          control_group: `/${fsStub.hierarchy}`,
          usage_nanos: 357753491408
        },
        cpu: {
          control_group: '/docker',
          cfs_period_micros: 100000,
          cfs_quota_micros: 5000,
          stat: {
            number_of_elapsed_periods: 0,
            number_of_times_throttled: 10,
            time_throttled_nanos: 20
          }
        }
      });
    });

    it('handles an undefined control group', async () => {
      (0, _fs_stubs.setMockFiles)({
        '/proc/self/cgroup': '',
        [`${fsStub.cpuAcctDir}/cpuacct.usage`]: '357753491408',
        [`${fsStub.cpuDir}/cpu.stat`]: fsStub.cpuStatContents,
        [`${fsStub.cpuDir}/cpu.cfs_period_us`]: '100000',
        [`${fsStub.cpuDir}/cpu.cfs_quota_us`]: '5000'
      });

      const stats = await (0, _cgroup.getAllStats)();

      expect(stats).toBe(null);
    });

    it('can override the cpuacct group path', async () => {
      (0, _fs_stubs.setMockFiles)({
        '/proc/self/cgroup': fsStub.cGroupContents,
        '/sys/fs/cgroup/cpuacct/docker/cpuacct.usage': '357753491408',
        [`${fsStub.cpuDir}/cpu.cfs_period_us`]: '100000',
        [`${fsStub.cpuDir}/cpu.cfs_quota_us`]: '5000',
        [`${fsStub.cpuDir}/cpu.stat`]: fsStub.cpuStatContents
      });

      const stats = await (0, _cgroup.getAllStats)({ cpuAcctPath: '/docker' });

      expect(stats).toEqual({
        cpuacct: {
          control_group: '/docker',
          usage_nanos: 357753491408
        },
        cpu: {
          control_group: `/${fsStub.hierarchy}`,
          cfs_period_micros: 100000,
          cfs_quota_micros: 5000,
          stat: {
            number_of_elapsed_periods: 0,
            number_of_times_throttled: 10,
            time_throttled_nanos: 20
          }
        }
      });
    });

    it('extracts control group stats', async () => {
      (0, _fs_stubs.setMockFiles)(fsStub.files);
      const stats = await (0, _cgroup.getAllStats)();

      expect(stats).toEqual({
        cpuacct: {
          control_group: `/${fsStub.hierarchy}`,
          usage_nanos: 357753491408
        },
        cpu: {
          control_group: `/${fsStub.hierarchy}`,
          cfs_period_micros: 100000,
          cfs_quota_micros: 5000,
          stat: {
            number_of_elapsed_periods: 0,
            number_of_times_throttled: 10,
            time_throttled_nanos: 20
          }
        }
      });
    });

    it('returns null when all files are missing', async () => {
      (0, _fs_stubs.setMockFiles)();
      const stats = await (0, _cgroup.getAllStats)();
      expect(stats).toBeNull();
    });

    it('returns null if CPU accounting files are missing', async () => {
      (0, _fs_stubs.setMockFiles)({
        '/proc/self/cgroup': fsStub.cGroupContents,
        [`${fsStub.cpuDir}/cpu.stat`]: fsStub.cpuStatContents
      });
      const stats = await (0, _cgroup.getAllStats)();

      expect(stats).toBeNull();
    });

    it('returns -1 stat values if cpuStat file is missing', async () => {
      (0, _fs_stubs.setMockFiles)({
        '/proc/self/cgroup': fsStub.cGroupContents,
        [`${fsStub.cpuAcctDir}/cpuacct.usage`]: '357753491408',
        [`${fsStub.cpuDir}/cpu.cfs_period_us`]: '100000',
        [`${fsStub.cpuDir}/cpu.cfs_quota_us`]: '5000'
      });
      const stats = await (0, _cgroup.getAllStats)();

      expect(stats).toEqual({
        cpu: {
          cfs_period_micros: 100000,
          cfs_quota_micros: 5000,
          control_group: `/${fsStub.hierarchy}`,
          stat: {
            number_of_elapsed_periods: -1,
            number_of_times_throttled: -1,
            time_throttled_nanos: -1
          }
        },
        cpuacct: {
          control_group: `/${fsStub.hierarchy}`,
          usage_nanos: 357753491408
        }
      });
    });
  });
});
