import { QueryBuilderFactory } from 'ui/kibi/helpers/query_builder';
import { parseWithPrecision } from 'ui/kibi/utils/date_math_precision';
import { bytesToHumanReadable } from 'ui/kibi/helpers/siren_translate_size_helper';
import uuid from 'uuid';
import moment from 'moment';
import { cloneDeep, each, filter, get } from 'lodash';
import { filterHelper } from 'ui/kibi/components/dashboards360/filter_helper';
import { coatHelper } from 'ui/kibi/components/dashboards360/coat_helper';

export function SirenDataExportHelperProvider(
  Private, timefilter, $rootScope, getAppState, globalState, chrome, config, createNotifier, kibiVersion) {
  return class SirenDataExportHelper {

    constructor({ form, source, indexPattern, stamp, useAppState, useGlobalTime, attrs, exportedFileName, omitNull, siren } = {}) {
      this._form = form;
      this._source = source;
      this._indexPattern = indexPattern;
      this._stamp = stamp;
      this._useAppState = useAppState;
      this._useGlobalTime = useGlobalTime;
      this._attrs = attrs;
      this._exportedFileName = exportedFileName;
      this._omitNull = omitNull;
      this._siren = siren;

      this.updateTimeFromGlobalState = (event) => {
        if (event) {
          event.preventDefault();
        };
        // ***** Refresh time button is hidden for now
        // const timeButton = this._form.querySelector('div.time-container button.refresh-time');
        // if (timefilter.time.mode === 'absolute') {
        //   timeButton.style.display = 'none';
        // } else {
        //   timeButton.style.display = 'inline-block';
        // };
        // *****
        let dateFormat = config.get('dateFormat');
        if (
          dateFormat.indexOf('HH') === -1 &&
          dateFormat.indexOf('mm') === -1 &&
          dateFormat.indexOf('ss') === -1
        ) {
          // update time format if it doesn't have all time parts (hours and minutes and seconds)
          dateFormat += ' HH:mm:ss';
        };
        this._savedTimefilter = {
          from: parseWithPrecision(timefilter.time.from, false, $rootScope.sirenTimePrecision),
          to: parseWithPrecision(timefilter.time.to, true, $rootScope.sirenTimePrecision)
        };
        this._form.querySelector('div.time-container div.time span.time-from').innerHTML =
          moment(this._savedTimefilter.from).format(dateFormat);
        this._form.querySelector('div.time-container div.time span.time-to').innerHTML =
          moment(this._savedTimefilter.to).format(dateFormat);
      };

      const executeUpdateEstimatedSizeOnChanges = () => {
        this.executeUpdateEstimatedSizeOnChanges();
        this.updateTimeFromGlobalState();
      };

      if (this._useAppState === true) {
        this.appState = getAppState();
        this.appState.on('save_with_changes', executeUpdateEstimatedSizeOnChanges);
      };
      if (this._useGlobalTime === true && this._indexPattern.timeFieldName) {
        this._savedTimefilter = {
          from: parseWithPrecision(timefilter.time.from, false, $rootScope.sirenTimePrecision),
          to: parseWithPrecision(timefilter.time.to, true, $rootScope.sirenTimePrecision)
        };
        this.updateTimeFromGlobalState();
        this._form.querySelector('div.time-container button.refresh-time').onclick = this.updateTimeFromGlobalState;
        globalState.on('save_with_changes', executeUpdateEstimatedSizeOnChanges);
      };

      this.destroyForm = () => {
        if (this._interval) {
          clearInterval(this._interval);
        };

        if (this._form && this._form.parentNode) {
          this._form.parentNode.removeChild(this._form);
          if (this._attrs.exportFormIsOpened) {
            delete this._attrs.exportFormIsOpened;
          };
        };

        if (this._useAppState === true) {
          this.appState.off('save_with_changes', executeUpdateEstimatedSizeOnChanges);
        };

        if (this._useGlobalTime === true && this._indexPattern.timeFieldName) {
          globalState.off('save_with_changes', executeUpdateEstimatedSizeOnChanges);
        };
      };
    };

    getSelectedFieldsArray() {
      const fields = [];
      Array.prototype.forEach.call(
        this._form.querySelector('div.fields-container div.fields div.fields-in').querySelectorAll('input'), field => {
          if (field.checked) {
            fields.push(field.value);
          }
        }
      );
      return fields;
    };

    getSelectedFieldsValue() {
      const fields = this.getSelectedFieldsArray();
      return fields.length > 0 ? fields.join(',') : '*';
    };

    updateEstimatedSize({
      format = 'csv',
      size = 'all',
      callWithTotalSizeUpdating = false
    } = {}) {
      const COLOR_FOR_WARNING_SIZE = '#b62516';
      const COLOR_FOR_NORMAL_SIZE = '#b6d6e0';
      const BIG_FILE_THRESHOLD_IN_BYTES = 1073741824;
      if (callWithTotalSizeUpdating) {
        this.updateTimeFromGlobalState();
        size = 'all';
      };

      const query = JSON.stringify(this.getAppliedQuery().query);
      const _source = this.getSelectedFieldsValue();
      const body = {
        indexPatternId: this._indexPattern.id,
        query,
        format,
        size,
        _source,
        kbn_version: kibiVersion,
        omitNull: this._omitNull
      };
      const xhr = new XMLHttpRequest();
      xhr.open("POST", chrome.getBasePath() + '/export/estimate', true);
      xhr.setRequestHeader("Content-Type", "application/json");
      xhr.send(JSON.stringify(body));
      const estimateSizeValue = this._form.querySelector('span.estimated-size');
      estimateSizeValue.innerHTML = '<i class="kuiIcon fa-sync fa-spin"></i>';
      const estimatedSizeContainer = this._form.querySelector('div.estimated-size-container');
      estimatedSizeContainer.style.color = COLOR_FOR_NORMAL_SIZE;
      estimatedSizeContainer.style.borderColor = COLOR_FOR_NORMAL_SIZE;
      xhr.onreadystatechange = () => {
        if (xhr.readyState !== 4) {
          return;
        } else if (xhr.status !== 200) {
          createNotifier({ location: 'Data Export' })
            .error(`Could not estimate due to error: [${xhr.status}] ${xhr.responseText}`);
          return;
        }

        const parseResponse = JSON.parse(xhr.responseText);
        estimateSizeValue.innerHTML = bytesToHumanReadable(parseResponse.estimatedSize);
        if (callWithTotalSizeUpdating) {
          this._indexSize = parseResponse.totalHits;
          const maxSizeContainer = this._form.querySelector('span.max-size');
          const selectedForm = document.forms['sirenDataExport-' + this._stamp];
          if (selectedForm) {
            const limitExceedsHits = selectedForm.elements.limit.value > parseResponse.totalHits;
            maxSizeContainer.style.display = limitExceedsHits ? 'inline' : 'none';
          }
        };

        if (parseResponse.estimatedSize > BIG_FILE_THRESHOLD_IN_BYTES) {
          estimatedSizeContainer.style.color = COLOR_FOR_WARNING_SIZE;
          estimatedSizeContainer.style.borderColor = COLOR_FOR_WARNING_SIZE;
        };
      };
    };

    getAppliedQuery() {
      const filters = this._source.filter ? cloneDeep(this._source.filter) : [];
      const queries = this._source.query ? [this._source.query] : [];

      const filtersForThisRequest = [...filters];
      const queriesForThisRequest = [...queries];

      if (this._useAppState === true) {
        const isTheRequestFromForeignWidgetDashboard = coatHelper.isItForeingWidgetsCoat(this._siren);
        const isVisPresentInCoat = coatHelper.isVisPresentInTheCoat(this._siren);

        if (isTheRequestFromForeignWidgetDashboard && isVisPresentInCoat) {
          // here push query into allFilters as we do not make a distinction between filter and query
          const allFilters = [...filters, ...this.appState.filters, ... queries, this.appState.query];
          filtersForThisRequest.push(...filterHelper.getMainForThisRequest(allFilters, this._siren));
          const foreignSearchJoinFilters = filterHelper.composeJoinFilterForForeignVis({
            _sirenFromVis: this._siren,
            allFilters,
            time: null,
            sirenTimePrecision: $rootScope.sirenTimePrecision,
            useRootNodeAsFocus: false
          });
          filtersForThisRequest.push(...foreignSearchJoinFilters);
        } else {
          filtersForThisRequest.push(...this.appState.filters);
          queriesForThisRequest.push(this.appState.query);
        }
      }

      if (this._useGlobalTime) {
        const timeFieldName = this._indexPattern.timeFieldName;

        let addTimeFilter = false;
        if (timeFieldName) {
          const isItFromDash360EnabledVis = this._siren && get(this._siren, 'coat.node') !== null;
          if (isItFromDash360EnabledVis) {
            if (get(this._siren, 'coat.node.d.entity.useGlobalTimeFilter') === true) {
              addTimeFilter = true;
            }
          } else {
            addTimeFilter = true;
          }
        }

        if (addTimeFilter) {
          const savedTimefilter = this.getSavedTimefilter();
          const timeFieldFilter = {
            range: {
              [timeFieldName]: {
                gte: savedTimefilter.from.valueOf(),
                lte: savedTimefilter.to.valueOf(),
                format: 'epoch_millis'
              }
            }
          };
          filtersForThisRequest.push(timeFieldFilter);
        }
      }

      const queryBuilder = Private(QueryBuilderFactory);
      return queryBuilder(filtersForThisRequest, queriesForThisRequest);
    };

    executeUpdateEstimatedSizeOnChanges() {
      const selectedForm = document.forms['sirenDataExport-' + this._stamp];
      if (!selectedForm) {
        return;
      }

      const userFormElements = selectedForm.elements;
      const format = userFormElements.format.value;
      this.updateEstimatedSize({
        format,
        size: userFormElements.limit.value,
        callWithTotalSizeUpdating: true
      });
    };

    getIndexSize() {
      return this._indexSize;
    };

    getSavedTimefilter() {
      return this._savedTimefilter;
    };

    submitForm(sortValue) {

      function getCookie(name) {
        const value = `; ${document.cookie}`;
        const parts = value.split(`; ${name}=`);
        if (parts.length === 2) return parts.pop().split(';').shift();
      };

      function deleteCookie(name) {
        document.cookie = `${name}=;expires=Thu, 01 Jan 1970 00:00:01 GMT;`;
      };

      const query = this.getAppliedQuery();
      const queryString = JSON.stringify(query.query);
      const cookieName = uuid.v1();
      const userFormElements = document.forms[`sirenDataExport-${this._stamp}`].elements;
      const iframe = this._form.querySelector('iframe');
      const innerDoc = iframe.contentDocument || iframe.contentWindow.document;
      const form = innerDoc.forms[0];
      const formElements = form.elements;
      if (this._exportedFileName) {
        formElements.filename.value = this._exportedFileName;
      };
      formElements.indexPatternId.value = this._indexPattern.id;
      formElements.query.value = queryString;
      formElements.format.value = userFormElements.format.value;
      formElements.size.value = userFormElements.limit.value;
      formElements._source.value = this.getSelectedFieldsValue();
      formElements.formatted.value = userFormElements.formatted.value === 'true';
      formElements.omitNull.value = userFormElements.omitNull.value === 'true';
      formElements.sort.value = sortValue ? `[${sortValue}]` : '';
      formElements.cookieName.value = cookieName;
      formElements.kbn_version.value = kibiVersion;

      const self = this;
      this._interval = setInterval(function () {
        // detect the
        const cookie = getCookie(cookieName);
        if (cookie) {
          clearInterval(this._interval);
          deleteCookie(cookieName);
          if (cookie === 'ok') {
            deleteCookie(cookieName);
            self.destroyForm();
          }
          // for now we do not close the box on error
        }
      }, 50);

      form.submit();
    };

  };
};
