'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScrollStream = undefined;

var _stream = require('stream');

var _process = require('process');

var _process2 = _interopRequireDefault(_process);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _perf_hooks = require('perf_hooks');

var _field_types = require('./field_types');

var _stream_helpers = require('./common/stream_helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Stream that automatically scrolls
 * through elasticsearch query results.
 */
class ScrollStream extends _stream.Readable {
  /**
   * @param {object} client
   * @param {object} indexInfo
   * @param {object} query
   * @param {object} options
   */
  constructor(client, indexInfo, query, options = {}) {
    const defaultOpts = {
      timeout: '60s',
      docsToFetch: 'all',
      normalize: false,
      pretty: false,
      _source: '*',
      debug: false,
      batchSize: 1000,
      objectMode: false,
      sort: undefined,
      omitNull: false
    };
    options = Object.assign(defaultOpts, options);
    super(options);
    this._client = client;
    this._indexInfo = indexInfo;
    this._query = query;
    this._timeout = options.timeout;
    this._docsToFetch = options.docsToFetch;
    this._normalize = options.normalize;
    this._pretty = options.pretty;
    this._debug = options.debug;
    this._batchSize = options.batchSize;
    this._source = (0, _stream_helpers.parseSource)(options._source);
    this._sort = options.sort;
    this._omitNull = options.omitNull;
    this._fields = [];

    // Internal state
    this._reading = false;
    this._scrollId = '';
    this._filter_path = 'hits.hits._source,hits.hits.fields,_scroll_id';
    this._totalBatches = 0;
    this._batchesFetched = 0;
    this._docsFetched = 0;

    // Debug state
    this._startTime = _perf_hooks.performance.now();
  }

  /**
   * Read next response from elasticsearch
   * and push to stream buffer.
   */
  async _read() {
    if (this._reading) {
      return false;
    }

    this._reading = true;

    try {
      let response;
      do {
        response = await this._fetchBatch();
      } while (this.push(response));
    } catch (err) {
      this.destroy(err);
    }

    this._reading = false;
  }

  async _fetchBatch() {
    let response;
    const startTime = _perf_hooks.performance.now();

    if (this._docsFetched === 0) {
      this._log('\nRequest parameters:\n' + ` Index pattern: '${this._indexInfo.title}'\n` + ` Query: '${JSON.stringify(this._query, undefined, 2)}'\n` + ` Timeout: '${this._timeout}'`);
      this._log('============Starting search============');

      this._docsToFetch = await this._getDocCount();
      this._batchSize = Math.min(this._docsToFetch, this._batchSize);
      this._batchSize = Math.max(1, this._batchSize);
      this._totalBatches = Math.ceil(this._docsToFetch / this._batchSize);
      response = await this._fetchFirstBatch();
    } else if (this._docsFetched < this._docsToFetch) {
      response = await this._fetchNextBatch();
    } else {
      const timeTaken = (_perf_hooks.performance.now() - this._startTime).toFixed(2);
      this._log(`=====Scroll finished  (${timeTaken}ms taken)=====`);

      return null;
    }

    const timeTaken = _perf_hooks.performance.now() - startTime;

    if (this._docsFetched + response.hits.hits.length > this._docsToFetch) {
      const remainingHits = this._docsToFetch - this._docsFetched;
      response.hits.hits = response.hits.hits.slice(0, remainingHits);
    }

    this._batchesFetched++;

    const memUsedInMB = _process2.default.memoryUsage().rss / 1024 / 1024;
    this._log(`Received batch. [${this._batchesFetched}/${this._totalBatches}] ` + `(${timeTaken.toFixed(2)}ms taken)\n` + `Memory in use: ${memUsedInMB} mb`);

    _lodash2.default.forEach(response.hits.hits, hit => {
      // Merge script/stored fields into _source
      hit = Object.assign(hit._source, hit.fields);

      _lodash2.default.forEach(this._fields, field => {
        const fieldInfo = _lodash2.default.get(hit, field.name);
        if (this._omitNull && fieldInfo === null) {
          _lodash2.default.set(hit, field.name, '');
        } else if (fieldInfo && this._normalize) {
          _lodash2.default.set(hit, field.name, (0, _field_types.createFieldType)(fieldInfo, field.type).normalized());
        }
      });
    });

    let finalResponse;
    if (this._readableState.objectMode) {
      finalResponse = response.hits.hits.map(hit => hit._source);
    } else {
      finalResponse = this._responseToString(response);
    }
    this._docsFetched += response.hits.hits.length;
    return finalResponse;
  }

  /**
   * Initial elasticsearch query.
   */
  async _fetchFirstBatch() {
    this._fields = await (0, _stream_helpers.filterFields)(this._client, this._indexInfo.title, this._indexInfo.fields, this._source);
    const body = (0, _stream_helpers.buildRequestBody)(this._fields, this._source, this._query, this._sort);
    this._log(`\nRequest body:\n${JSON.stringify(body, undefined, 2)}`);
    const response = await this._client('siren_search', {
      index: this._indexInfo.title,
      scroll: this._timeout,
      filter_path: 'hits.total,' + this._filter_path,
      body,
      size: this._batchSize
    });
    this._scrollId = response._scroll_id;
    return response;
  }

  /**
   * Fetch the next batch of documents using the elasticsearch scroll API.
   * @returns {object}
   */
  async _fetchNextBatch() {
    const response = await this._client('siren_scroll', {
      scrollId: this._scrollId,
      scroll: this._timeout,
      filter_path: this._filter_path
    });

    this._scrollId = response._scroll_id;
    return response;
  }

  /**
   * Helper method that stringifies the source fields
   * of an elasticsearch response.
   * @param {object} response
   * @returns {string}
   */
  _responseToString(response) {
    const finished = this._docsFetched + response.hits.hits.length === this._docsToFetch;
    let jsonString = response.hits.hits.map(hit => JSON.stringify(hit._source, undefined, this._pretty ? 2 : 0)).join(',\n');
    jsonString = (this._docsFetched === 0 ? '[' : '') + jsonString + (finished ? ']' : ',\n');

    return jsonString;
  }

  /**
   * Helper method that sets maxDocs to
   * every available document, or to a user-specified number.
   * @param {object} response
   */
  _setMaxDocs(response) {
    if (this._docsToFetch === 'all') {
      this._docsToFetch = response.hits.total;
    } else {
      this._docsToFetch = Math.min(this._docsToFetch, response.hits.total);
    }

    this._totalBatches = Math.ceil(this._docsToFetch / this._batchSize);
  }

  _log(message) {
    if (this._debug) {
      this.emit('log', message);
    }
  }

  async _getDocCount() {
    const response = await this._client('siren_search', {
      index: this._indexInfo.title,
      filter_path: 'hits.total',
      body: {
        query: this._query
      },
      size: 0
    });

    if (this._docsToFetch === 'all') {
      return response.hits.total;
    } else {
      return Math.min(this._docsToFetch, response.hits.total);
    }
  }

  /**
   * Reimplementation of stream.Readable._destroy() that clears the
   * elasticsearch scroll before destroying the stream.
   * @param {string} err
   * @param {function} callback
   */
  _destroy(err, callback) {
    this._client('clearScroll', { scrollId: this._scrollId }).then(() => super._destroy(err, callback)).catch(() => super._destroy(err, callback));
  }
}
exports.ScrollStream = ScrollStream;
