'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _create_scroll_stream = require('../lib/create_scroll_stream.js');

var _stream_helpers = require('../lib/streams/common/stream_helpers');

var _boom = require('boom');

var _boom2 = _interopRequireDefault(_boom);

var _joi = require('joi');

var _joi2 = _interopRequireDefault(_joi);

var _moment = require('moment');

var _moment2 = _interopRequireDefault(_moment);

var _stream = require('stream');

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class EstimateStream extends _stream.Writable {
  constructor(docsToFetch, batchSize, options) {
    super(options);
    this.averageSize = 0;
    this.batchCount = 0;

    this.batchSize = batchSize;
    this.batchesToFetch = docsToFetch / batchSize;
  }

  estimate() {
    const estimatedSize = Math.round(this.averageSize * this.batchesToFetch);
    return estimatedSize;
  }

  _write(chunk, encoding, callback) {
    const batchLength = chunk.length;
    const prev = this.averageSize * this.batchCount;
    this.batchCount++;
    this.averageSize = (prev + batchLength) / this.batchCount;
    callback();
  }
}

const sharedValidationScheme = {
  size: _joi2.default.alternatives([_joi2.default.number().integer().positive(), _joi2.default.string().empty('').valid(['all'])]).default('all'),
  format: _joi2.default.string().empty('').valid(['csv', 'json']).default('json'),
  pretty: _joi2.default.boolean().default(false),
  normalize: _joi2.default.boolean().default(false),
  debug: _joi2.default.boolean().default(false),
  formatted: _joi2.default.boolean().default(false),
  omitNull: _joi2.default.boolean().default(false),
  kbn_version: _joi2.default.string().empty('')
};

exports.default = server => {

  server.route({
    path: '/export',
    config: {
      validate: {
        payload: Object.assign({
          indexPatternId: _joi2.default.string().required(),
          query: _joi2.default.object().empty('').default({ match_all: {} }),
          _source: _joi2.default.alternatives([_joi2.default.boolean(), _joi2.default.array(), _joi2.default.object(), _joi2.default.string()]).empty('').default('*'),
          sort: _joi2.default.alternatives([_joi2.default.object(), _joi2.default.array()]).empty('').default(undefined),
          filename: _joi2.default.string().empty('').default(() => `export_${(0, _moment2.default)().format('YYYY-MM-DD-HH-mm-ss')}`, 'File name.'),
          cookieName: _joi2.default.string().default('myCookie')
        }, sharedValidationScheme)
      }
    },

    method: 'POST',
    handler: async (req, reply) => {
      req.payload.query = await req.server.plugins.query_engine.translateToES(req.server, req, JSON.stringify(req.payload.query));
      const filename = req.payload.filename;
      const format = req.payload.format;
      const expires = new Date(new Date().getTime() + 86409000).toUTCString();
      try {
        const stream = await (0, _create_scroll_stream.createScrollStream)(req);

        reply(stream).header('Content-Disposition', `attachment; filename="${filename}.${format}"`).header('Content-Type', 'application/x-download').header("Set-Cookie", `${req.payload.cookieName}=ok; expires=${expires}`);
      } catch (err) {
        const errResponse = _boom2.default.wrap(err, 400);
        errResponse.output.headers['Set-Cookie'] = `${req.payload.cookieName}=err; expires=${expires}`;
        reply(errResponse);
      }
    }
  });

  const savedSearchHandler = async (req, reply) => {
    const expires = new Date(new Date().getTime() + 86409000).toUTCString();
    try {
      const kibanaIndex = req.server.config().get('kibana.index');
      const dataCluster = req.server.plugins.elasticsearch.getCluster('data');
      const client = (...args) => dataCluster.callWithRequest(req, ...args);

      const savedSearch = await client('get', {
        index: kibanaIndex,
        type: '_all',
        id: req.params.searchId
      });
      const parsedSearchSource = JSON.parse(savedSearch._source.search.kibanaSavedObjectMeta.searchSourceJSON);

      req.payload.indexPatternId = parsedSearchSource.index;
      req.payload._source = savedSearch._source.search.columns || '*';
      req.dashboardSearchSource = req.dashboardSearchSource || [];
      parsedSearchSource.query = parsedSearchSource.query || { match_all: {} };
      parsedSearchSource.filter = parsedSearchSource.filter || [];
      req.payload.query = {
        bool: {
          must: [parsedSearchSource.query, ...req.dashboardSearchSource, ...parsedSearchSource.filter.map(entry => _lodash2.default.omit(entry, ['meta', '$state']))]
        }
      };
      req.payload.sort = _lodash2.default.set({}, savedSearch._source.search.sort[0], savedSearch._source.search.sort[1]);

      const filename = req.payload.filename;
      const format = req.payload.format;
      const stream = await (0, _create_scroll_stream.createScrollStream)(req);

      reply(stream).header('Content-Disposition', `attachment; filename="${filename}.${format}"`).header('Content-Type', 'application/x-download').header("Set-Cookie", `${req.payload.cookieName}=ok; expires=${expires}`);
    } catch (err) {
      const errResponse = _boom2.default.wrap(err, 400);
      errResponse.output.headers['Set-Cookie'] = `${req.payload.cookieName}=err; expires=${expires}`;
      reply(errResponse);
    }
  };

  server.route({
    path: '/export/saved_search/{searchId}',
    config: {
      validate: {
        payload: Object.assign({
          filename: _joi2.default.string().empty('').default(() => `export_${(0, _moment2.default)().format('YYYY-MM-DD-HH-mm-ss')}`, 'File name.')
        }, sharedValidationScheme),
        params: {
          searchId: _joi2.default.string().required()
        }
      }
    },

    method: 'POST',
    handler: savedSearchHandler
  });

  server.route({
    path: '/export/dashboard/{dashboardId}',
    config: {
      validate: {
        payload: Object.assign({
          filename: _joi2.default.string().empty('').default(() => `export_${(0, _moment2.default)().format('YYYY-MM-DD-HH-mm-ss')}`, 'File name.')
        }, sharedValidationScheme),
        params: {
          dashboardId: _joi2.default.string().required()
        }
      }
    },

    method: 'POST',
    handler: async (req, reply) => {
      const expires = new Date(new Date().getTime() + 86409000).toUTCString();
      try {
        const kibanaIndex = req.server.config().get('kibana.index');
        const dataCluster = req.server.plugins.elasticsearch.getCluster('data');
        const client = (...args) => dataCluster.callWithRequest(req, ...args);
        const dashboardResponse = await client('get', {
          index: kibanaIndex,
          type: '_all',
          id: req.params.dashboardId
        });

        const coat = JSON.parse(dashboardResponse._source.dashboard.coatJSON);
        const rootNode = _lodash2.default.find(coat.items, item => item.d.isRoot === true);
        let savedSearchId;
        if (rootNode) {
          savedSearchId = rootNode.d.entity.id;
        }

        if (!savedSearchId) {
          throw new Error(`Dashboard '${dashboardResponse._source.dashboard.title}' does not have an associated saved search.`);
        }

        const searchSource = JSON.parse(dashboardResponse._source.dashboard.kibanaSavedObjectMeta.searchSourceJSON);
        req.dashboardSearchSource = searchSource.filter.map(entry => {
          return entry.query ? entry.query : _lodash2.default.omit(entry, ['meta', '$state']);
        });
        if (searchSource.query) {
          req.dashboardSearchSource.concat(searchSource.query);
        }
        req.params.searchId = savedSearchId;
        await savedSearchHandler(req, reply);
      } catch (err) {
        const errResponse = _boom2.default.wrap(err, 400);
        errResponse.output.headers['Set-Cookie'] = `${req.payload.cookieName}=err; expires=${expires}`;
        reply(errResponse);
      }
    }
  });

  server.route({
    path: '/export/estimate',
    config: {
      validate: {
        payload: Object.assign({
          indexPatternId: _joi2.default.string().required(),
          query: _joi2.default.object().empty('').default({ match_all: {} }),
          _source: _joi2.default.alternatives([_joi2.default.boolean(), _joi2.default.array(), _joi2.default.object(), _joi2.default.string()]).empty('').default('*'),
          sort: _joi2.default.alternatives([_joi2.default.object(), _joi2.default.array()]).empty('').default(undefined)
        }, sharedValidationScheme)
      }
    },

    method: 'POST',
    handler: async (req, reply) => {
      try {
        req.payload.query = await req.server.plugins.query_engine.translateToES(req.server, req, JSON.stringify(req.payload.query));
        const desiredDocCount = await getDocCount(req);
        const sampleSize = 20;
        req.payload.size = sampleSize;
        const stream = await (0, _create_scroll_stream.createScrollStream)(req, { fixedBatchSize: sampleSize });
        const estimateStream = new EstimateStream(desiredDocCount, sampleSize);
        stream.pipe(estimateStream);

        const myPromise = new Promise((resolve, reject) => stream.on("end", () => resolve({
          estimatedSize: estimateStream.estimate(),
          totalHits: desiredDocCount
        })));
        reply(myPromise);
      } catch (err) {
        const errResponse = _boom2.default.wrap(err, 400);
        reply(errResponse);
      }
    }
  });
};

async function getDocCount(req) {
  const params = req.payload;
  const docsToFetch = params.size;
  const kibanaIndex = req.server.config().get('kibana.index');
  const dataCluster = req.server.plugins.elasticsearch.getCluster('data');
  const client = (...args) => dataCluster.callWithRequest(req, ...args);
  const indexInfo = await (0, _stream_helpers.fetchIndexInfo)(client, kibanaIndex, params.indexPatternId);
  const indexTitle = indexInfo.title;
  const response = await client('siren_search', {
    index: indexTitle,
    filter_path: 'hits.total',
    body: {
      query: params.query
    },
    size: 0
  });
  if (docsToFetch === 'all') {
    return response.hits.total;
  } else {
    return Math.min(docsToFetch, response.hits.total);
  }
}
module.exports = exports['default'];
