'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _federate_version_parser = require('./utils/federate_version_parser');

var _compatibility_status = require('./utils/compatibility_status');

var _compatibility_status2 = _interopRequireDefault(_compatibility_status);

var _node_compatibility_resolver = require('./node_compatibility_resolver');

var _node_compatibility_resolver2 = _interopRequireDefault(_node_compatibility_resolver);

var _constants = require('./constants');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Resolves compatibility status of ES cluster with Investigate
 */
class CompatibilityResolver {

  static _isCompatible(compatibilityStatus) {
    return compatibilityStatus >= _compatibility_status2.default.MIN_SUPPORTED;
  }

  static getIncompatibleNodeReason(compatibilityStatus, federateVersion) {
    if (!CompatibilityResolver._isCompatible(compatibilityStatus)) {
      switch (compatibilityStatus) {
        case _compatibility_status2.default.FEDERATE_NOT_FOUND:
          return `The node does not have Federate.`;
        case _compatibility_status2.default.INVALID_VERSION:
          return `The node appears to have an invalid Federate version: ${federateVersion}`;
        case _compatibility_status2.default.TOO_OLD:
          return `The node has an older than minimum supported version of Federate: ${federateVersion}`;
        case _compatibility_status2.default.TOO_HIGH:
          return `The Federate installed is higher than the target version: ${federateVersion}, please upgrade Investigate ` + `or install a supported version of Federate.`;
        default:
          return `Not compatible with ${federateVersion}`;
      }
    }
    return `${federateVersion} is compatible.`;
  }

  constructor(nodes, compatibility) {
    this.compatibilityMatrix = Object.freeze({
      MIN_SUPPORTED: new _federate_version_parser.FederateVersionParser(compatibility.min_supported),
      FEATURE_COMPLETE: new _federate_version_parser.FederateVersionParser(compatibility.feature_complete),
      TARGET: new _federate_version_parser.FederateVersionParser(compatibility.target)
    });
    this.rawNodes = nodes;
    let compatibilityStatus;
    this.nodes = [];
    this.incompatibleNodes = [];
    for (const key in nodes) {
      if (nodes.hasOwnProperty(key)) {
        const node = nodes[key];
        let federateVersion;
        for (let i = 0; i < node.plugins.length; i++) {
          if (node.plugins[i].name === _constants.SIREN_FEDERATE_PLUGIN) {
            federateVersion = node.plugins[i].version;
            break;
          }
        }
        const resolvedNode = new _node_compatibility_resolver2.default(federateVersion, this.compatibilityMatrix);
        this.nodes.push(resolvedNode);
        if (compatibilityStatus === undefined) {
          compatibilityStatus = resolvedNode.getCompatibilityStatus();
        } else if (compatibilityStatus > resolvedNode.getCompatibilityStatus()) {
          compatibilityStatus = resolvedNode.getCompatibilityStatus();
        }
        if (!resolvedNode.isCompatible()) {
          this.incompatibleNodes.push({
            node,
            reason: CompatibilityResolver.getIncompatibleNodeReason(resolvedNode.getCompatibilityStatus(), federateVersion)
          });
        }
      }
    }
    this.compatibilityStatus = compatibilityStatus;
  }

  isCompatible() {
    return CompatibilityResolver._isCompatible(this.compatibilityStatus);
  }

  getIncompatibleNodes() {
    return this.incompatibleNodes;
  }

  getCompatibilityStatus() {
    return this.compatibilityStatus;
  }
}
exports.default = CompatibilityResolver;
module.exports = exports['default'];
