'use strict';

var _ = require('./');

describe('LicenseParser', () => {
  let federateResolver;
  let licenseParser;
  let dummyUnlimitedLicense;
  let oldUnlimitedLicense;

  const noLicense = {
    license: {
      content: {},
      isValid: false
    }
  };
  const communityLicense = {
    license_content: {
      description: 'Siren Community License',
      issue_date: '-',
      permissions: {
        federate: {
          max_concurrent_sessions: '1',
          max_nodes: '1'
        },
        investigate: {
          max_dashboards: '12',
          max_graph_nodes: '500',
          max_virtual_indices: '5'
        }
      },
      valid_date: '-'
    },
    license_validation: {
      is_valid: true
    }
  };

  beforeEach(() => {
    federateResolver = {
      checkApiAvailability: jest.fn()
    };
    licenseParser = new _.LicenseParser(federateResolver);
    dummyUnlimitedLicense = {
      license_content: {
        permissions: {
          federate: {
            max_concurrent_sessions: '-1',
            max_nodes: '-1',
            session_timeout: '-1'
          },
          investigate: {
            max_dashboards: '-1',
            max_graph_nodes: '-1',
            max_virtual_indices: '-1'
          }
        },
        valid_date: 'date',
        issue_date: 'date'
      },
      license_validation: {
        is_valid: true
      }
    };
    oldUnlimitedLicense = {
      license: {
        content: {
          'issue-date': 'date',
          'graph-browser': 'true',
          'max-nodes': '12',
          company: 'Siren Solutions',
          'valid-date': 'date',
          version: '0.2'
        },
        isValid: true
      }
    };
  });

  const setupApiAvailability = function (boolean) {
    federateResolver.checkApiAvailability.mockReturnValueOnce(boolean);
  };

  const checkFederateResolverCalls = function () {
    expect(federateResolver.checkApiAvailability).toBeCalledWith('LICENSE_API_V2');
    expect(federateResolver.checkApiAvailability.mock.calls.length).toBe(1);
  };

  describe('latest license api', () => {
    beforeEach(() => {
      setupApiAvailability(true);
    });

    it('returns license untouched', () => {
      setupApiAvailability(true);
      const license = 'latest-license';
      expect(licenseParser.ensureLatest(license)).toEqual(license);
      checkFederateResolverCalls();
    });
  });

  describe('old license api', () => {
    beforeEach(() => {
      setupApiAvailability(false);
    });

    it('converts to community license if no license', () => {
      expect(licenseParser.ensureLatest(noLicense)).toEqual(communityLicense);
      checkFederateResolverCalls();
    });

    it('graph browser: enabled', () => {
      expect(licenseParser.ensureLatest(oldUnlimitedLicense)).toEqual(dummyUnlimitedLicense);
      checkFederateResolverCalls();
    });

    it('graph browser: disabled', () => {
      oldUnlimitedLicense.license.content['graph-browser'] = false;
      dummyUnlimitedLicense.license_content.permissions.investigate.max_graph_nodes = 500;

      expect(licenseParser.ensureLatest(oldUnlimitedLicense)).toEqual(dummyUnlimitedLicense);
      checkFederateResolverCalls();
    });
  });
}); /* global expect, jest */
