'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeRequest = executeRequest;

var _http = require('http');

var _http2 = _interopRequireDefault(_http);

var _https = require('https');

var _https2 = _interopRequireDefault(_https);

var _check_domain = require('./check_domain.js');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Requests image in url and return image stream if
 * - url is a valid image url,
 * - url domain is allowed,
 * - image dimension doesn't exceeds maximum allowable size.
 * - request response time doesn't exceeds maximum allowable timeout.
 *
 * @param {string} url
 * @param {object} serverConfig
 * @returns {object}
 */
function executeRequest(url, serverConfig) {
  return new Promise((resolve, reject) => {
    const allowedDomains = serverConfig.get('investigate_core.image_proxy.allowed_domains');
    const domain = new URL(url).hostname;
    if (!(0, _check_domain.checkDomain)(domain, allowedDomains)) {
      reject(new Error(`${domain} not allowed. To allow add ${domain} to investigate_core.image_proxy.allowed_domains list in investigate.yml`));
    }

    const isHttps = /^https/.test(url);
    const httpClient = isHttps ? _https2.default : _http2.default;
    const timeout = serverConfig.get('investigate_core.image_proxy.timeout');
    const request = httpClient.request(url, function (response) {
      const maxImageSize = serverConfig.get('investigate_core.image_proxy.max_image_size');
      const statusCode = response.statusCode;
      const contentType = response.headers['content-type'];
      const contentLength = response.headers['content-length'];
      if (statusCode !== 200) {
        reject(new Error(`Responded with status code ${statusCode}`));
      } else if (!contentType.includes('image/')) {
        reject(new Error('No image found'));
      } else if (contentLength > maxImageSize) {
        reject(new Error(`Image size (${contentLength}) exceeds maximum allowed size of (${maxImageSize})`));
      }
      const chunks = [];
      response.on('data', function (chunk) {
        chunks.push(chunk);
      });
      response.on('error', function (err) {
        reject(new Error(err));
      });
      response.on('end', function (data) {
        const ret = Buffer.concat(chunks);
        resolve({
          statusCode: statusCode,
          contentType: contentType,
          contentLength: contentLength,
          body: ret
        });
      });
    });
    request.setTimeout(timeout, function () {
      reject(new Error('Connection timed-out'));
    });
    request.on('error', function (err) {
      reject(new Error(err));
    });
    request.end();
  });
}
