'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 15.
 *
 * Looks for:
 *
 * - the "discover:sampleSize" property in config
 * - the "pageSize" property in "visState" property of visualizations
 *
 * Then:
 *
 * - if "discover:sampleSize" or "visState" is a string parse it to number
 */
class Migration15 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');
    this._configQuery = {
      "query": {
        "bool": {
          "filter": { "term": { "_id": "kibi" } }
        }
      }
    };
    this._visQuery = {
      'query': {
        'exists': { 'field': 'visState' }
      }
    };
  }

  static get description() {
    return 'Check "discover:sampleSize" property in config and "pageSize" property in "visState" property of visualizations ' + 'if any of them a string, parse to number';
  }

  async _fetchConfig() {
    this._config = '';
    if (this._index) {
      let configs;
      try {
        configs = await this.scrollSearch(this._index, 'config', this._configQuery);
      } catch (error) {
        if (error.status === 404) {
          return 0;
        } else {
          throw error;
        }
      }
      if (configs && configs.length > 0) {
        const kibiConfig = configs[0];
        if (_lodash2.default.isString(kibiConfig._source['discover:sampleSize'])) {
          this._config = kibiConfig;
        }
      }
    }
  }

  async _fetchVisualizations() {
    this._visualizations = [];
    let visualizations;
    try {
      visualizations = await this.scrollSearch(this._index, 'visualization', this._visQuery);
    } catch (error) {
      if (error.status === 404) {
        return 0;
      } else {
        throw error;
      }
    }

    _lodash2.default.each(visualizations, visualization => {
      const visState = JSON.parse(visualization._source.visState);
      if (_lodash2.default.isString(_lodash2.default.get(visState.params, 'pageSize'))) {
        this._visualizations.push(visualization);
      }
    });
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    await this._fetchConfig();
    await this._fetchVisualizations();
    if (this._config || !_lodash2.default.isEmpty(this._visualizations)) {
      return this._visualizations.length + (this._config ? 1 : 0);
    }
    return 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let upgraded = 0;
    await this._fetchConfig();
    await this._fetchVisualizations();
    if (this._visualizations.length === 0 && !this._config) {
      return upgraded;
    }

    let body = '';
    this._logger.info('Checking "discover:sampleSize" property in config and ' + '"pageSize" property in "visState" property of visualizations');

    if (_lodash2.default.get(this._config._source, 'discover:sampleSize')) {
      const sampleSize = _lodash2.default.get(this._config._source, 'discover:sampleSize');
      if (_lodash2.default.isString(sampleSize)) {
        const sampleSizeInt = parseInt(sampleSize);

        this._logger.info('Parsing "discover:sampleSize" property in [ ' + this._config._id + ' ] to number');
        body += JSON.stringify({
          update: {
            _index: this._config._index,
            _type: this._config._type,
            _id: this._config._id
          }
        }) + '\n' + JSON.stringify({
          doc: {
            'discover:sampleSize': sampleSizeInt
          }
        }) + '\n';
        upgraded++;
      }
    }

    _lodash2.default.each(this._visualizations, visualization => {
      const visState = JSON.parse(visualization._source.visState);
      if (_lodash2.default.get(visState.params, 'pageSize') && _lodash2.default.isString(visState.params.pageSize)) {
        const pageSizeInt = parseInt(visState.params.pageSize);
        visState.params.pageSize = pageSizeInt;
        visualization._source.visState = visState;
        const visStateObjectString = JSON.stringify(visualization._source.visState);

        this._logger.info('Parsing "pageSize" property in [ ' + visualization._id + ' ] to number');
        body += JSON.stringify({
          update: {
            _index: visualization._index,
            _type: visualization._type,
            _id: visualization._id
          }
        }) + '\n' + JSON.stringify({
          doc: {
            visState: visStateObjectString
          }
        }) + '\n';
        upgraded++;
      }
    });

    if (upgraded > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return upgraded;
  }
}
exports.default = Migration15;
module.exports = exports['default'];
