'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 2.
 *
 * Looks for saved templates having version set to `1` and:
 *
 * - renames attributes with the `st_` prefix in the name
 * - bumps version to `2`.
 */

class Migration2 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'template';
    this._query = {
      query: {
        match: {
          version: 1
        }
      }
    };
  }

  static get description() {
    return 'Upgrade saved templates from version 1 to version 2';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    return await this.countHits(this._index, this._type, this._query);
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    let body = '';
    for (const obj of objects) {
      const upgraded = this._upgradeObject(obj);
      body += JSON.stringify({
        update: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      }) + '\n' + JSON.stringify({ doc: upgraded }) + '\n';
    }
    await this._client.bulk({
      refresh: true,
      body: body
    });
    return objects.length;
  }

  /**
   * Upgrades a single object.
   *
   * @param {Object} obj The object to upgrade.
   */
  _upgradeObject(obj) {
    const upgraded = (0, _lodash.transform)(obj._source, (result, value, key) => {
      if (key.indexOf('st_') === 0) {
        key = key.replace(/^st_/, '');
        if (obj._source.hasOwnProperty(key)) {
          this._logger.warning(`Template with id ${obj._id} already contains an attribute named ${key}` + `, will not remove attribute st_${key}`);
          return;
        }
        result[key] = value;
      }
    });
    upgraded.version = 2;
    return upgraded;
  }

}
exports.default = Migration2;
module.exports = exports['default'];
