'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _gremlin_server = require('../../../../../server/gremlin_server/gremlin_server');

var _gremlin_server2 = _interopRequireDefault(_gremlin_server);

var _gremlin_server_errors = require('../../../../../server/gremlin_server/gremlin_server_errors');

var _server_ontology_client = require('../../ontology/server_ontology_client');

var _server_ontology_client2 = _interopRequireDefault(_server_ontology_client);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 28.
 *
 * As we try to eliminate all direct usage of index-patterns in our app
 *
 * Looks for:
 * all visualization with "index" property set in kibanaSavedObjectMeta.searchSourceJSON which points to existing index-pattern
 * where a corresponding root saved search is present
 *
 * Then when found:
 * - delete this "index" property and replace it with "savedSearchId" property
 */
class Migration28 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');

    this._visType = 'visualization';
    this._indexPatternType = 'index-pattern';

    this._ontologyType = 'ontology-model';
    this._ontologyId = 'default-ontology';
    this._gremlinStartTimeout = configuration.gremlinStartTimeout;
  }

  static get description() {
    return 'Detach visualizations from index-patterns and attach them to root saved searches.';
  }

  async _findAllUpgradeableVis(callbackOnEachFound, useInCount) {
    const gremlin = new _gremlin_server2.default(this._server, this._gremlinStartTimeout);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    try {
      await gremlin.start('Migration 28');

      const visualizations = await this.scrollSearch(this._index, this._visType);
      const indexPatterns = await this.scrollSearch(this._index, this._indexPatternType);
      const entities = await serverOntologyClient.getEntities();

      _lodash2.default.each(visualizations, hit => {
        if (hit._source.kibanaSavedObjectMeta && hit._source.kibanaSavedObjectMeta.searchSourceJSON) {
          const searchSourceJSON = hit._source.kibanaSavedObjectMeta.searchSourceJSON;
          try {
            const searchSource = JSON.parse(searchSourceJSON);
            if (searchSource.index) {
              // check if such index-pattern exist
              const indexPattern = _lodash2.default.find(indexPatterns, '_id', searchSource.index);

              if (!indexPattern) {
                this._logger.warning(`Detected visualisation [${hit._id}] attached to nonexistent index pattern [${searchSource.index}].` + 'Such visualisation can be fixed only by directly editing of the saved object ' + 'at /management/siren/objects by removing the incorrect index pattern in ' + 'kibanaSavedObjectMeta.searchSourceJSON property and ' + 'adding an existing saved search to savedSearchId property');
              } else {
                // check if there is a corresponding root saved search
                const foundRootSavedSearch = _lodash2.default.find(entities, entity => {
                  return entity.type === 'SAVED_SEARCH' && (entity.parentId === 'null' || !entity.parentId) && entity.indexPattern === searchSource.index;
                });

                if (foundRootSavedSearch) {

                  callbackOnEachFound({
                    found: true,
                    savedSearchId: foundRootSavedSearch.id,
                    visSearchSource: searchSource,
                    visHit: hit
                  });
                }
              }
            }
          } catch (e) {
            // should not happen if it happens the object is broken anyway
            this._logger.error('Detected broken kibanaSavedObjectMeta.searchSourceJSON object in saved search: ' + hit._id);
          }
        }
      });
      this._logger.info('All done stopping Siren Gremlin Server');
      await gremlin.stop('Migration 28 clean');
    } catch (e) {
      if (useInCount) {
        this._logger.error('Migration 28 - count operation');
      }
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 28');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 28');
        this._logger.error(e);
        await gremlin.stop('Migration 28 error');
      } else {
        this._logger.error('Could not perform Migration 28');
        this._logger.error(e);
        await gremlin.stop('Migration 28 error');
      }
      throw e;
    }
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let count = 0;
    await this._findAllUpgradeableVis(function (res) {
      if (res.found) {
        count++;
      }
    }, true);
    return count;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    let count = 0;
    let body = '';
    await this._findAllUpgradeableVis(function (res) {
      if (res.found) {
        const _source = res.visHit._source;
        const searchSource = res.visSearchSource;
        const savedSearchId = res.savedSearchId;

        delete searchSource.index;
        _source.savedSearchId = savedSearchId;
        _source.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSource);

        body += JSON.stringify({
          update: {
            _index: res.visHit._index,
            _type: res.visHit._type,
            _id: res.visHit._id
          }
        }) + '\n' + JSON.stringify({ doc: _source }) + '\n';
        count++;
      }
    });

    if (count > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return count;
  }

}
exports.default = Migration28;
module.exports = exports['default'];
