'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 30.
 *
 * Checks if siren:indexExclusionRegexList is defined in the configuration;
 * if so, tries to parse the JSON as an array and convert it to a list of
 * index patterns and puts it in siren:excludedIndices.
 *
 * The siren:indexExclusionRegexList is then removed.
 *
 * If siren:excludedIndices is already set it does nothing.
 */

const EXCLUSION_REGEX_KEY = 'siren:indexExclusionRegexList';
const EXCLUDED_INDICES_KEY = 'siren:excludedIndices';

class Migration30 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'config';
    this._query = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'siren'
            }
          }]
        }
      }
    };
  }

  static get description() {
    return `Convert ${EXCLUSION_REGEX_KEY} to ${EXCLUDED_INDICES_KEY}`;
  }

  _isExclusionRegexUpgradeable(exclusionRegex) {
    if (typeof exclusionRegex !== 'string') {
      const msg = `Expected ${EXCLUSION_REGEX_KEY} to be a string but got ${exclusionRegex}`;
      this._logger.error(msg);
      throw new Error(msg);
    } else {
      try {
        exclusionRegex = JSON.parse(exclusionRegex);
      } catch (e) {
        const msg = `Could not parse ${EXCLUSION_REGEX_KEY}`;
        this._logger.error(msg, e);
        throw new Error(msg);
      }
    }
    if (!Array.isArray(exclusionRegex)) {
      const msg = `Could not parse ${EXCLUSION_REGEX_KEY} as an array.`;
      this._logger.error(msg);
      throw new Error(msg);
    }
    return true;
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    const docSource = objects[0]._source;
    const exclusionRegex = docSource[EXCLUSION_REGEX_KEY];
    const excludedIndices = docSource[EXCLUDED_INDICES_KEY];
    if (excludedIndices) {
      return 0;
    }
    if (exclusionRegex) {
      const isUpgradeable = this._isExclusionRegexUpgradeable(exclusionRegex);
      if (isUpgradeable === true) {
        return 1;
      }
    }
    return 0;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    let count = 0;
    const doc = objects[0];
    const docSource = doc._source;
    const exclusionRegex = docSource[EXCLUSION_REGEX_KEY];
    const excludedIndices = docSource[EXCLUDED_INDICES_KEY];
    if (excludedIndices) {
      return 0;
    }
    if (typeof exclusionRegex === 'undefined' || exclusionRegex === null) {
      return 0;
    }

    let exclusionRegexArray;
    if (typeof exclusionRegex !== 'string') {
      const msg = `Expected ${EXCLUSION_REGEX_KEY} to be a string but got ${exclusionRegex}.`;
      throw new Error(msg);
    } else {
      try {
        exclusionRegexArray = JSON.parse(exclusionRegex);
      } catch (e) {
        const msg = `Could not parse ${EXCLUSION_REGEX_KEY}`;
        this._logger.error(msg);
        throw new Error(msg);
      }
    }

    let transformedValue;
    if (Array.isArray(exclusionRegexArray)) {
      count = 1;
      const transformed = [];
      exclusionRegexArray.forEach(regex => {
        if (regex) {
          transformed.push(regex.replace(/\\./g, '.').replace(/\.\*/g, '*'));
        }
      });
      if (transformed.length > 0) {
        transformedValue = transformed.join(',');
      }
    } else {
      const msg = `Could not parse ${EXCLUSION_REGEX_KEY} as an array.`;
      this._logger.error(msg);
      throw new Error(msg);
    }

    if (count > 0) {
      const body = JSON.stringify({
        update: {
          _index: doc._index,
          _type: doc._type,
          _id: doc._id
        }
      }) + '\n' + JSON.stringify({
        script: `ctx._source.remove('${EXCLUSION_REGEX_KEY}');` + `ctx._source['${EXCLUDED_INDICES_KEY}'] = '${transformedValue}'`
      }) + '\n';

      const response = await this._client.bulk({
        refresh: true,
        body: body
      });

      if (response.errors === true) {
        this._logger.error(`Got an error while setting ${EXCLUDED_INDICES_KEY}: ${JSON.stringify(response)}`);
        throw new Error(`Could not set ${EXCLUDED_INDICES_KEY}.`);
      }
    }
    return count;
  }
}
exports.default = Migration30;
module.exports = exports['default'];
