'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 31.
 *
 * Looks for an obsolete object of type "session"
 * in kibana.index and removes all found
 *
 * This objects were created a long time ago and are obsolete
 *
 */

class Migration31 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._server = configuration.server;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._type = 'session';
    this._query = {
      query: {
        match: {
          version: 1
        }
      }
    };
  }

  static get description() {
    return 'Find obsolete objects of type "session" in kibana.index and remove them';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }

    return await this.countHits(this._index, this._type, this._query);
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    const count = objects.length;
    if (count === 0) {
      return 0;
    }

    let body = '';
    for (const obj of objects) {
      body += JSON.stringify({
        delete: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      }) + '\n';
    }

    await this._client.bulk({
      refresh: true,
      body: body
    });

    return count;
  }

}
exports.default = Migration31;
module.exports = exports['default'];
