'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _gremlin_server = require('../../../../../server/gremlin_server/gremlin_server');

var _gremlin_server2 = _interopRequireDefault(_gremlin_server);

var _gremlin_server_errors = require('../../../../../server/gremlin_server/gremlin_server_errors');

var _server_ontology_client = require('../../ontology/server_ontology_client');

var _server_ontology_client2 = _interopRequireDefault(_server_ontology_client);

var _check_es_version = require('../../elasticsearch/check_es_version');

var _requirefrom = require('requirefrom');

var _requirefrom2 = _interopRequireDefault(_requirefrom);

var _entity_type = require('../../ontology/entity_type');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const joiToMapping = (0, _requirefrom2.default)('src/siren_core_plugins/saved_objects_api/lib/model/')('_joi_to_mapping');
/**
 * Investigate Core - Index Migration from ES5 to ES6 format.
 *
 *  Creates bulk index request for all Objects in .siren index
 *  Add mapping to object as objects of new types are found.
 *  if any object had to be migrated
 *      Delete .siren index
 *      Create .siren index with mapping
 *      send bulk index request
 */

class IndexMigration extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._logger = configuration.logger;
    this._index = configuration.config.get('kibana.index');
    this._type = 'doc';
    this._server = configuration.server;

    this._ontologyType = 'doc';
    this._ontologyId = 'ontology-model:default-ontology';
    this._registeredTypes = (0, _lodash.map)(this._server.plugins.saved_objects_api.getTypes(), 'type');
  }

  static get description() {
    return 'Upgrade Investigate index to Elasticsearch 6 compatible format';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    // should trigger for both 5 and 6
    if (!(esVersion.major === 5 || esVersion.major === 6)) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index);
    return objects.reduce((count, obj) => {
      if (this._isUpgradeable(obj)) {
        return count + 1;
      }
      return count;
    }, 0);
  }

  addMapping(requestBody, type) {
    // Ignore as these types are old and should be removed. "sentinl-script" will not have been removed if the user removed Sentinl
    // from his installation, so it is better to just not create the mapping
    const ignoredTypes = ['sentinl-script', 'session'];

    if (this._registeredTypes.includes(type)) {
      const schema = this._server.plugins.saved_objects_api.getModel(type).schema;
      if (schema) {
        requestBody.properties[type] = {
          dynamic: true,
          properties: joiToMapping(schema)
        };
      }
    } else if (!ignoredTypes.includes(type)) {
      this._logger.warning(`Saved object schema not registered for: ${type}. It may no longer be used.`);
    }
  }

  manageMapping(typeSet, type, mappingBody) {
    if (!typeSet.has(type)) {
      typeSet.add(type);
      try {
        this.addMapping(mappingBody, type);
      } catch (err) {
        this._logger.error(err);
      }
    }
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    // should trigger for both 5 and 6
    if (!(esVersion.major === 5 || esVersion.major === 6)) {
      return 0;
    }

    const idMap = [];
    const objects = await this.scrollSearch(this._index);
    if (objects.length === 0) {
      return 0;
    }
    const body = [];
    let upgraded = 0;
    const mappingBody = {
      dynamic: true,
      properties: {
        type: {
          type: 'keyword'
        },
        updated_at: {
          type: 'date'
        }
      }
    };

    const typeSet = new Set();
    for (const obj of objects) {
      const modified = this._upgradeDocument(obj);

      if (!modified) {
        this.manageMapping(typeSet, obj._source.type, mappingBody);
        body.push({
          index: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        });
        body.push(obj._source);
        continue;
      }
      this.manageMapping(typeSet, obj._type, mappingBody);
      body.push({
        index: {
          _index: obj._index,
          _type: this._type,
          _id: `${obj._type}:${obj._id}`
        }
      });
      body.push(obj._source);
      idMap.push({
        id: obj._id,
        type: obj._type,
        newId: `${obj._type}:${obj._id}`
      });
      upgraded++;
    }

    if (upgraded > 0) {
      //Delete previous index and push mapping
      if (typeSet.size > 0) {
        await this._client.indices.delete({
          index: this._index
        });
        await this._client.indices.create({
          index: this._index,
          body: {
            mappings: {
              [this._type]: mappingBody
            }
          }
        });
      }

      //Bulk indexing request on newly created index
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }

    if (idMap.length > 0) {
      await this._upgradeOntologyModel(idMap);
      await this._upgradeVisualizations(idMap);
      await this._upgradeDashboards(idMap);
      await this._upgradeDashboardgroups(idMap);
      await this._upgradeSearches(idMap);
      await this._upgradeConfig(idMap);
      await this._upgradeSavedGraphs(idMap);
      await this._upgradeRelationalGraph(idMap);
    }
    return upgraded;
  }

  async _upgradeRelationalGraph(idMap) {
    const object = await this._client.get({
      index: this._index,
      type: this._type,
      ignore: [404],
      id: this._ontologyId
    });
    if (object.found && object._source['ontology-model']['relational-graph']) {
      const relationalGraphItems = object._source['ontology-model']['relational-graph'].items;
      (0, _lodash.each)(relationalGraphItems, graphItem => {
        if (graphItem.type === 'node' && graphItem.d.entityType === _entity_type.EntityType.SAVED_SEARCH) {
          const searchFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === graphItem.id && item.type === 'search';
          });
          if (searchFound) {
            graphItem.id = searchFound.newId;
          }
        } else if (graphItem.type === 'link') {
          let searchFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === graphItem.id1 && item.type === 'search';
          });
          if (searchFound) {
            graphItem.id1 = searchFound.newId;
          }
          searchFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === graphItem.id2 && item.type === 'search';
          });
          if (searchFound) {
            graphItem.id2 = searchFound.newId;
          }
        }
      });

      object._source['ontology-model']['relational-graph'].items = relationalGraphItems;
      await this._client.index({
        index: this._index,
        type: this._type,
        id: this._ontologyId,
        body: object._source,
        refresh: true
      });
    };
  }

  async _upgradeSearches(idMap) {
    const query = {
      query: {
        match: {
          type: 'search'
        }
      }
    };
    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      const searchSourceJSON = JSON.parse(obj._source.search.kibanaSavedObjectMeta.searchSourceJSON);

      let idFound = false;
      if (searchSourceJSON.index) {
        idFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === searchSourceJSON.index && item.type === 'index-pattern';
        });
        if (idFound) {
          searchSourceJSON.index = idFound.newId;
        }
      }

      let savedFiltersFound = false;
      if (searchSourceJSON.filter && searchSourceJSON.filter.length > 0) {
        (0, _lodash.each)(searchSourceJSON.filter, filter => {
          if (filter.meta && filter.meta.index) {
            const idFound = (0, _lodash.find)(idMap, item => {
              return item.id === filter.meta.index && item.type === 'index-pattern';
            });
            if (idFound) {
              filter.meta.index = idFound.newId;
              savedFiltersFound = true;
            }
          }
        });
      }

      if (idFound || savedFiltersFound) {
        obj._source.search.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  _upgradeSequentialJoinButton(button, idMap) {
    // NOTE:
    // do not try to do anything with "button.indexRelationId"
    // as the relations live only in ontology and the ids should be the same

    if (button.sourceDashboardId) {
      const itemFound = (0, _lodash.find)(idMap, function (item) {
        return item.id === button.sourceDashboardId && item.type === 'dashboard';
      });
      if (itemFound) {
        button.sourceDashboardId = itemFound.newId;
      } else {
        this._logger.warning('Could not find upgraded id for dashboard: ' + button.sourceDashboardId);
      }
    }

    if (button.targetDashboardId) {
      const itemFound = (0, _lodash.find)(idMap, function (item) {
        return item.id === button.targetDashboardId && item.type === 'dashboard';
      });
      if (itemFound) {
        button.targetDashboardId = itemFound.newId;
      } else {
        this._logger.warning('Could not find upgraded id for dashboard: ' + button.targetDashboardId);
      }
    }

    return button;
  }

  async _upgradeVisualizations(idMap) {
    const query = {
      query: {
        match: {
          type: 'visualization'
        }
      }
    };
    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      const value = obj._source.visualization.savedSearchId;
      const idFound = (0, _lodash.find)(idMap, function (item) {
        return item.id === value && item.type === 'search';
      });
      if (idFound) {
        obj._source.visualization.savedSearchId = idFound.newId;
      }

      const visState = JSON.parse(obj._source.visualization.visState);
      let searchSourceJSON;
      if (obj._source.visualization && obj._source.visualization.kibanaSavedObjectMeta && obj._source.visualization.kibanaSavedObjectMeta.searchSourceJSON) {
        searchSourceJSON = JSON.parse(obj._source.visualization.kibanaSavedObjectMeta.searchSourceJSON);
      }

      let templateFound = false;
      if (visState.params && visState.params.templateId) {
        const visStateTemplateId = visState.params.templateId;
        templateFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === visStateTemplateId && item.type === 'template';
        });
        if (templateFound) {
          visState.params.templateId = templateFound.newId;
        }
      }

      let contextualScriptsFound = false;
      if (visState.params && visState.params.contextualScripts) {
        contextualScriptsFound = true;
        for (const script of visState.params.contextualScripts) {
          const itemFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === script.id && item.type === 'script';
          });
          if (itemFound) {
            script.id = itemFound.newId;
          }
        }
      }

      let lensScriptsFound = false;
      if (visState.params && visState.params.lensScripts) {
        lensScriptsFound = true;
        for (const script of visState.params.lensScripts) {
          const itemFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === script.id && item.type === 'script';
          });
          if (itemFound) {
            script.id = itemFound.newId;
          }
        }
      }

      let onUpdateScriptsFound = false;
      if (visState.params && visState.params.onUpdateScripts) {
        onUpdateScriptsFound = true;
        for (const script of visState.params.onUpdateScripts) {
          const itemFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === script.id && item.type === 'script';
          });
          if (itemFound) {
            script.id = itemFound.newId;
          }
        }
      }

      let indexPatternFound = false;
      if (searchSourceJSON && searchSourceJSON.index) {
        const indexPatternId = searchSourceJSON.index;
        indexPatternFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === indexPatternId && item.type === 'index-pattern';
        });
        if (indexPatternFound) {
          searchSourceJSON.index = indexPatternFound.newId;
        }
      }
      let savedFiltersFound = false;
      if (searchSourceJSON.filter && searchSourceJSON.filter.length > 0) {
        (0, _lodash.each)(searchSourceJSON.filter, filter => {
          if (filter.meta && filter.meta.index) {
            const idFound = (0, _lodash.find)(idMap, item => {
              return item.id === filter.meta.index && item.type === 'index-pattern';
            });
            if (idFound) {
              filter.meta.index = idFound.newId;
              savedFiltersFound = true;
            }
          }
        });
      }

      let expansionScriptFound = false;
      if (visState.params && visState.params.expansionScript) {
        expansionScriptFound = true;
        const itemFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === visState.params.expansionScript && item.type === 'script';
        });
        if (itemFound) {
          visState.params.expansionScript = itemFound.newId;
        }
      }

      let sequentialJoinVisFound = false;
      if (visState.type === 'kibi_sequential_join_vis' && (0, _lodash.get)(visState, 'params.buttons') && visState.params.buttons.length > 0) {
        sequentialJoinVisFound = true;
        visState.params.buttons = visState.params.buttons.map(button => this._upgradeSequentialJoinButton(button, idMap));
      }

      let timelineGroupsFound = false;
      if (visState.type === 'kibi_timeline' && (0, _lodash.get)(visState, 'params.groups') && visState.params.groups.length > 0) {
        timelineGroupsFound = true;
        visState.params.groups = visState.params.groups.map(group => {
          const indexPatternFound = (0, _lodash.find)(idMap, item => item.id === group.indexPatternId && item.type === 'index-pattern');
          if (indexPatternFound) {
            group.indexPatternId = indexPatternFound.newId;
          }

          const searchFound = (0, _lodash.find)(idMap, item => item.id === group.savedSearchId && item.type === 'search');
          if (searchFound) {
            group.savedSearchId = searchFound.newId;
          }

          return group;
        });
      }

      const objectsFound = templateFound || contextualScriptsFound || lensScriptsFound || onUpdateScriptsFound || expansionScriptFound || sequentialJoinVisFound || timelineGroupsFound;

      if (objectsFound) {
        obj._source.visualization.visState = JSON.stringify(visState);
      }

      if (indexPatternFound || savedFiltersFound) {
        obj._source.visualization.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
      }

      if (idFound || objectsFound || indexPatternFound || savedFiltersFound) {
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  async _upgradeDashboards(idMap) {
    const query = {
      query: {
        match: {
          type: 'dashboard'
        }
      }
    };
    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      let savedSearchIdFound = false;
      if (obj._source.dashboard.savedSearchId) {
        const value = obj._source.dashboard.savedSearchId;
        savedSearchIdFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === value && item.type === 'search';
        });
        if (savedSearchIdFound) {
          obj._source.dashboard.savedSearchId = savedSearchIdFound.newId;
        }
      }

      let oldPanelFound = false;

      if (obj._source.dashboard.panelsJSON) {
        const panelsJSON = JSON.parse(obj._source.dashboard.panelsJSON);
        for (const panel of panelsJSON) {
          const idFound = (0, _lodash.find)(idMap, function (item) {
            return item.id === panel.id && item.type === panel.type;
          });
          if (idFound) {
            oldPanelFound = true;
            panel.id = idFound.newId;
          }
        }

        if (oldPanelFound) {
          obj._source.dashboard.panelsJSON = JSON.stringify(panelsJSON);
        }
      }

      let uiStateScriptsFound = false;
      if (obj._source.dashboard.uiStateJSON) {
        const uiStateJSON = JSON.parse(obj._source.dashboard.uiStateJSON);
        (0, _lodash.each)(uiStateJSON, obj => {
          if (obj.settings && obj.settings.lenses && obj.settings.lenses.lens) {
            (0, _lodash.each)(obj.settings.lenses.lens, lens => {
              if (lens.state && lens.state.scriptId) {
                const idFound = (0, _lodash.find)(idMap, item => {
                  return item.id === lens.state.scriptId && item.type === 'script';
                });
                if (idFound) {
                  uiStateScriptsFound = true;
                  lens.state.scriptId = idFound.newId;
                }
              }
            });
          }
        });

        if (uiStateScriptsFound) {
          obj._source.dashboard.uiStateJSON = JSON.stringify(uiStateJSON);
        }
      }

      let indexPatternFound = false;
      let savedFiltersFound = false;

      if (obj._source.dashboard.kibanaSavedObjectMeta && obj._source.dashboard.kibanaSavedObjectMeta.searchSourceJSON) {

        const searchSourceJSON = JSON.parse(obj._source.dashboard.kibanaSavedObjectMeta.searchSourceJSON);

        if (searchSourceJSON.index) {
          const idFound = (0, _lodash.find)(idMap, item => {
            return item.id === searchSourceJSON.index && item.type === 'index-pattern';
          });
          if (idFound) {
            searchSourceJSON.index = idFound.newId;
            indexPatternFound = true;
          }
        }

        if (searchSourceJSON.filter && searchSourceJSON.filter.length > 0) {
          (0, _lodash.each)(searchSourceJSON.filter, filter => {
            if (filter.meta && filter.meta.index) {
              const idFound = (0, _lodash.find)(idMap, item => {
                return item.id === filter.meta.index && item.type === 'index-pattern';
              });
              if (idFound) {
                filter.meta.index = idFound.newId;
                savedFiltersFound = true;
              }
            }
          });
        }
        if (indexPatternFound || savedFiltersFound) {
          obj._source.dashboard.kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
        }
      }

      if (oldPanelFound || savedSearchIdFound || uiStateScriptsFound || indexPatternFound || savedFiltersFound) {
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  async _upgradeDashboardgroups(idMap) {
    const query = {
      query: {
        match: {
          type: 'dashboardgroup'
        }
      }
    };
    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      const dashboards = JSON.parse(obj._source.dashboardgroup.dashboards);
      let oldDashboardIdFound = false;
      for (const dashboard of dashboards) {
        const value = dashboard.id;
        const idFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === value && item.type === 'dashboard';
        });
        if (idFound) {
          oldDashboardIdFound = true;
          dashboard.id = idFound.newId;
        }
      }
      if (oldDashboardIdFound) {
        obj._source.dashboardgroup.dashboards = JSON.stringify(dashboards);
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  async _upgradeConfig(idMap) {
    const query = {
      query: {
        match: {
          type: 'config'
        }
      }
    };
    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      const defaultIndex = obj._source.config.defaultIndex;
      const defaultIndexFound = (0, _lodash.find)(idMap, function (item) {
        return item.id === defaultIndex && item.type === 'index-pattern';
      });
      if (defaultIndexFound) {
        obj._source.config.defaultIndex = defaultIndexFound.newId;
      }

      const defaultDashboardId = obj._source.config['siren:defaultDashboardId'];
      const defaultDashboardIdFound = (0, _lodash.find)(idMap, function (item) {
        return item.id === defaultDashboardId && item.type === 'dashboard';
      });
      if (defaultDashboardIdFound) {
        obj._source.config['siren:defaultDashboardId'] = defaultDashboardIdFound.newId;
      }
      if (defaultIndexFound || defaultDashboardIdFound) {
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  async _upgradeOntologyModel(idMap) {
    const gremlin = new _gremlin_server2.default(this._server);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    try {
      await gremlin.start('Migration 33');

      const replaceIds = function (entity) {
        const response = {
          changed: false,
          id: entity.id
        };

        const idFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === entity.id && item.type === 'search';
        });
        if (idFound) {
          response.changed = true;
          entity.id = idFound.newId;
        }

        const indexPatternFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === entity.indexPattern && item.type === 'index-pattern';
        });
        if (indexPatternFound) {
          response.changed = true;
          entity.indexPattern = indexPatternFound.newId;
        }

        const parentIdFound = (0, _lodash.find)(idMap, function (item) {
          return item.id === entity.parentId && item.type === 'search';
        });
        if (parentIdFound) {
          response.changed = true;
          entity.parentId = parentIdFound.newId;
        }

        return response;
      };

      // NOTE:
      // Important grab both before we start to save or update any
      // as e.g. deleting the entity will delete the relations attached to it automatically
      let entities;
      let relations;
      try {
        entities = await serverOntologyClient.getEntities();
      } catch (e) {
        this._logger.error(e);
        throw new Error('Could not fetch entities. Will not migrate any entities');
      }
      try {
        relations = await serverOntologyClient.getRelations();
      } catch (e) {
        this._logger.error(e);
        throw new Error('Could not fetch relations. Will not migrate any relations');
      }

      if (entities && entities.length > 0) {
        // first we replace ids and resave all entities
        for (const entity of entities) {
          if (entity.type === 'SAVED_SEARCH') {
            const response = replaceIds(entity);

            if (response.changed) {
              await serverOntologyClient.deleteEntity(response.id);
              await serverOntologyClient.createEntity(entity);
            } else {
              this._logger.error('Could not find replacemnet id to update SAVED_SEARCH entity: ' + response.id);
            }
          }
        }
      }

      if (relations && relations.length > 0) {
        // now re-save all relations at once
        for (const relation of relations) {
          replaceIds(relation.domain);
          replaceIds(relation.range);
        };
        await serverOntologyClient.saveRelations(relations);
      }

      this._logger.info('All done stopping Siren Gremlin Server');
      await gremlin.stop('Migration 33 clean');
    } catch (e) {
      if (e instanceof _gremlin_server_errors.GremlinStopError) {
        this._logger.error('Could not stop the Siren Gremlin Server for Migration 33');
        this._logger.error(e);
      } else if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 33');
        this._logger.error(e);
        await gremlin.stop('Migration 33 error');
      } else {
        this._logger.error('Could not perform Migration 33');
        this._logger.error(e);
        await gremlin.stop('Migration 33 error');
      }
      throw e;
    }
  }

  async _upgradeSavedGraphs(idMap) {
    const query = {
      query: {
        match: {
          type: 'graph'
        }
      }
    };

    const objects = await this.scrollSearch(this._index, this._type, query);
    await Promise.all(objects.map(async obj => {
      let modified = false;
      const graphObject = JSON.parse(obj._source.graph.graph);
      (0, _lodash.each)(graphObject.items, item => {
        if (item.type === 'node') {
          const idFound = (0, _lodash.find)(idMap, function (obj) {
            return obj.id === item.indexPattern && obj.type === 'index-pattern';
          });
          if (idFound) {
            item.indexPattern = idFound.newId;
            modified = true;
          }
        }
      });

      if (modified) {
        obj._source.graph.graph = JSON.stringify(graphObject);
        const body = JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        await this._client.bulk({
          refresh: true,
          body: body
        });
      }
    }));
  }

  /**
   * Checks if a document is upgradeable.
   */
  _isUpgradeable(object) {
    if (object._type !== this._type) {
      return true;
    }
    return false;
  }

  /**
   * Upgrades a ES5 document to ES6.
   *
   * @param {Object} The document to upgrade.
   * @retval {Boolean} true if the document has been modified.
   * @retval {Boolean} false if the document has not been modified.
   */
  _upgradeDocument(doc) {
    let modified = false;
    if (this._isUpgradeable(doc)) {
      modified = true;
      const source = doc._source;
      doc._source = {
        type: doc._type,
        updated_at: new Date().toISOString()
      };
      doc._source[doc._type] = source;
    }
    return modified;
  }

}
exports.default = IndexMigration;
module.exports = exports['default'];
