'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 9.
 *
 * Looks for:
 *
 * - old gremlin queries
 *
 * Then:
 *
 * - remove them as they are not needed anymore.
 */
class Migration9 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');
    this._type = 'query';
  }

  static get description() {
    return 'Delete gremlin server saved queries.';
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const items = await this.scrollSearch(this._index, ['query', 'datasource']);
    const tinkerpop3Datasources = items.reduce((array, obj) => {
      if (obj._type === 'datasource') {
        const datasourceType = obj._source.datasourceType;
        if (datasourceType === 'tinkerpop3') {
          array.push(obj._id);
        }
      }
      return array;
    }, []);

    return items.reduce((count, obj) => {
      if (obj._type === 'query') {
        const datasourceId = obj._source.datasourceId;
        if (_lodash2.default.includes(tinkerpop3Datasources, datasourceId)) {
          count++;
        }
      }
      return count;
    }, 0);
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const items = await this.scrollSearch(this._index, ['query', 'datasource']);
    const tinkerpop3Datasources = items.reduce((array, obj) => {
      if (obj._type === 'datasource') {
        const datasourceType = obj._source.datasourceType;
        if (datasourceType === 'tinkerpop3') {
          array.push(obj._id);
        }
      }
      return array;
    }, []);

    const tinkerPopQueryIds = items.reduce((array, obj) => {
      if (obj._type === 'query') {
        const datasourceId = obj._source.datasourceId;
        if (_lodash2.default.includes(tinkerpop3Datasources, datasourceId)) {
          array.push(obj._id);
        }
      }
      return array;
    }, []);

    const bulkRequests = tinkerPopQueryIds.reduce((array, id) => {
      array.push({
        delete: {
          _index: this._index,
          _type: this._type,
          _id: id
        }
      });
      return array;
    }, []);

    if (bulkRequests.length) {
      await this._client.bulk({
        refresh: true,
        body: bulkRequests
      });
    }

    return bulkRequests.length;
  }

}
exports.default = Migration9;
module.exports = exports['default'];
