'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _json_helper = require('../../json/json_helper');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const PREFIX = 'index-pattern:';
const testValueFunction = function (value) {
  return value.indexOf(PREFIX) !== 0;
};

/**
 * Investigate Core - Migration 36.
 *
 * Find all searches, visualizations and  dashboards with broken references to index pattern
 * with value not prefixed with "index-pattern:"
 *
 * if found try to see that corresponding index pattern (with prefix) exists
 * if yes update the reference
 * if not print warning about the problem
 *
 * This migration was written to cover the upgrade from 10.1.x with already broken objects to 10.2.x
 */

class Migration36 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._server = configuration.server;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;

    this._defaultOntologyTitle = 'Default Ontology';
    this._id = 'ontology-model:default-ontology';
    this._type = 'doc';
    this._indexPatternQuery = {
      query: {
        match: {
          type: 'index-pattern'
        }
      }
    };
    this._visualizationQuery = {
      query: {
        match: {
          type: 'visualization'
        }
      }
    };
    this._dashboardQuery = {
      query: {
        match: {
          type: 'dashboard'
        }
      }
    };
    this._searchQuery = {
      query: {
        match: {
          type: 'search'
        }
      }
    };
  }

  static get description() {
    return 'Find all searches, dashboards and visualizations with broken references to index-pattern searchSourceJSON and fix them';
  }

  _isUpgradeable(obj, type) {
    if (obj._source[type] && obj._source[type].kibanaSavedObjectMeta.searchSourceJSON) {
      const searchSourceJSON = JSON.parse(obj._source[type].kibanaSavedObjectMeta.searchSourceJSON);
      if (searchSourceJSON.index && testValueFunction(searchSourceJSON.index)) {
        return true;
      }
      if (searchSourceJSON.filter) {
        for (const filter of searchSourceJSON.filter) {
          if (filter && filter.meta && filter.meta.index && testValueFunction(filter.meta.index)) {
            // if therte is a wrong reference in the filter meta
            return true;
          }
          if (filter && filter.join_sequence) {
            // if there is a wrong reference in the join_sequence filter
            // here we have to traverse the json and check all pattern properties
            const foundBrokenJoinFilter = (0, _json_helper.findPropertyInJson)(filter, 'pattern', testValueFunction);
            if (foundBrokenJoinFilter) {
              return true;
            }
          }
        }
      }
    }
    return false;
  }

  _upgradeSingle(obj, type, indexPatternIds) {
    let upgraded = false;
    const searchSourceJSON = JSON.parse(obj._source[type].kibanaSavedObjectMeta.searchSourceJSON);
    if (searchSourceJSON.index && testValueFunction(searchSourceJSON.index)) {
      const foundIndexPattern = indexPatternIds.indexOf(PREFIX + searchSourceJSON.index);
      upgraded = true;
      if (foundIndexPattern === -1) {
        this._logger.warning(`Could not fix reference to non existing index-pattern object inside ${type} id: ${obj._id}, searchSourceJSON property`);
      }
      searchSourceJSON.index = PREFIX + searchSourceJSON.index;
    }
    if (searchSourceJSON.filter) {

      const fixValueFunction = value => {
        const foundIndex = indexPatternIds.indexOf(PREFIX + value);
        upgraded = true;
        if (foundIndex === -1) {
          this._logger.warning(`Could not fix reference to non existing index-pattern object inside ${type} id: ${obj._id}, searchSourceJSON property`);
        }
        return PREFIX + value;
      };

      for (let i = 0; i < searchSourceJSON.filter.length; i++) {
        const filter = searchSourceJSON.filter[i];
        if (filter && filter.meta && filter.meta.index && testValueFunction(filter.meta.index)) {
          const foundIndexPattern = indexPatternIds.indexOf(PREFIX + filter.meta.index);
          upgraded = true;
          if (foundIndexPattern === -1) {
            this._logger.warning(`Could not fix reference to non existing index-pattern object inside ${type} id: ${obj._id}, searchSourceJSON property`);
          }
          filter.meta.index = PREFIX + filter.meta.index;
        }
        if (filter && filter.join_sequence && (0, _json_helper.findPropertyInJson)(filter, 'pattern', testValueFunction)) {
          const fixedFilter = (0, _json_helper.fixPropertyInJson)(filter, 'pattern', fixValueFunction);
          searchSourceJSON.filter[i] = fixedFilter;
        }
      }
    }
    if (upgraded) {
      obj._source[type].kibanaSavedObjectMeta.searchSourceJSON = JSON.stringify(searchSourceJSON);
      return JSON.stringify({
        update: {
          _index: obj._index,
          _type: obj._type,
          _id: obj._id
        }
      }) + '\n' + JSON.stringify({ doc: _extends({}, obj._source) }) + '\n';
    }
    return '';
  }

  async count() {
    const searches = await this.scrollSearch(this._index, this._type, this._searchQuery);
    const dashboards = await this.scrollSearch(this._index, this._type, this._dashboardQuery);
    const visualizations = await this.scrollSearch(this._index, this._type, this._visualizationQuery);
    let count = 0;
    for (const obj of searches) {
      if (this._isUpgradeable(obj, 'search')) {
        count++;
      }
    }
    for (const obj of dashboards) {
      if (this._isUpgradeable(obj, 'dashboard')) {
        count++;
      }
    }
    for (const obj of visualizations) {
      if (this._isUpgradeable(obj, 'visualization')) {
        count++;
      }
    }
    return count;
  }

  async upgrade() {
    const indexPatterns = await this.scrollSearch(this._index, this._type, this._indexPatternQuery);
    const indexPatternIds = indexPatterns.map(ip => ip._id);
    const searches = await this.scrollSearch(this._index, this._type, this._searchQuery);
    const dashboards = await this.scrollSearch(this._index, this._type, this._dashboardQuery);
    const visualizations = await this.scrollSearch(this._index, this._type, this._visualizationQuery);
    let count = 0;
    let body = '';
    for (const obj of searches) {
      if (this._isUpgradeable(obj, 'search')) {
        body += this._upgradeSingle(obj, 'search', indexPatternIds);
        count++;
      }
    }
    for (const obj of dashboards) {
      if (this._isUpgradeable(obj, 'dashboard')) {
        body += this._upgradeSingle(obj, 'dashboard', indexPatternIds);
        count++;
      }
    }
    for (const obj of visualizations) {
      if (this._isUpgradeable(obj, 'visualization')) {
        body += this._upgradeSingle(obj, 'visualization', indexPatternIds);
        count++;
      }
    }
    if (body.length > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }

    return count;
  }

}
exports.default = Migration36;
module.exports = exports['default'];
