'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Dashboard showTimePicker option - Migration 42.
 *
 * kibi-internal Issue: #10277
 * Set showTimePicker option to True (When not defined) in all dashboard objects even on the non-timebased one
 */

class Migration42 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._type = 'doc';
    this._query = {
      query: {
        match: {
          type: 'dashboard'
        }
      }
    };
  }

  static get description() {
    return 'Set showTimePicker option to True (When not defined) in all dashboard objects.';
  }

  /**
   * @param  {Object} dashboardObject
   * @return {Boolean}
   */
  static _upgradeRequired(dashboardObject) {
    const objectSource = dashboardObject._source;
    const optionsJSON = JSON.parse(objectSource[objectSource.type].optionsJSON);
    if (!optionsJSON.showTimePicker && optionsJSON.showTimePicker !== false) {
      return true;
    }
    return false;
  }

  async count() {
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    const migrationCount = objects.reduce((count, obj) => {
      if (Migration42._upgradeRequired(obj)) {
        count++;
      }
      return count;
    }, 0);
    return migrationCount;
  }

  /**
   * Sets optionsJSON.showTimePicker: true
   * @param  {Object} dashboardObject
   * @return {Object} upgraded object
   */
  static _upgradeObjectSource(dashboardObject) {
    const objectSource = dashboardObject._source;
    const optionsJSON = JSON.parse(objectSource[objectSource.type].optionsJSON);
    optionsJSON.showTimePicker = true;
    objectSource[objectSource.type].optionsJSON = JSON.stringify(optionsJSON);
    return dashboardObject;
  }

  async upgrade() {
    const migrationCount = await this.count();
    if (migrationCount > 0) {
      const objects = await this.scrollSearch(this._index, this._type, this._query);
      const bulkIndex = [];
      let upgradeCount = 0;
      objects.forEach(obj => {
        if (Migration42._upgradeRequired(obj)) {
          const { _index, _type, _id, _source } = Migration42._upgradeObjectSource(obj);
          bulkIndex.push({
            index: { _index, _type, _id }
          });
          bulkIndex.push(_source);
          upgradeCount++;
        }
      });
      if (upgradeCount > 0) {
        await this._client.bulk({
          refresh: true,
          body: bulkIndex
        });
      }
      return upgradeCount;
    }
    return 0;
  }

}
exports.default = Migration42;
module.exports = exports['default'];
