import _ from 'lodash';
import uiRoutes from 'ui/routes';
import { uiModules } from 'ui/modules';
import template from 'plugins/investigate_core/management/sections/data_model/index.html';
import refreshFieldsPopupTemplate from '../templates/refresh_fields_popup.html';
import '../styles/data_model.less';
import 'ui/timefilter';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { DataModelErrorType } from './error_type';
import { CrudType } from '../services/crud_type';
import { DataModelPermissionsProvider } from '../services/data_model_permissions';

import 'plugins/investigate_core/ui/directives/siren_entity_options/siren_entity_options';
import 'plugins/investigate_core/ui/directives/siren_entity_relations/siren_entity_relations';
import IndexOptionsHelperFactory from 'plugins/investigate_core//ui/directives/siren_entity_options/helpers/index_options_helper';
import { QuickDashboardProvider } from 'ui/kibi/quick_dashboard/quick_dashboard';
import { FindRelationsProvider } from 'ui/kibi/quick_relations/find_relations';
import { FingerprintsProvider } from 'ui/kibi/quick_relations/fingerprints';
import { OntologyWrapperProvider } from 'ui/kibi/quick_relations/ontology_wrapper';
import { GuessFieldsProvider } from 'ui/kibi/quick_dashboard/guess_fields';
import { MessageBoxProvider } from 'ui/kibi/modals/message_box';
import '../advanced_options/advanced_options';
import { EntityModalsProvider } from '../modals/modals';
import {
  ResolveObjectsOnStartViewProvider,
  ResolveObjectsOnCreateViewProvider,
  ResolveObjectsOnEntityViewProvider,
} from './resolvers';

// imports needed by 3 field tabs
import 'plugins/kibana/management/sections/indices/edit_index_pattern/edit_index_pattern';
import { IndicesEditSectionsProvider } from 'plugins/kibana/management/sections/indices/edit_index_pattern/edit_sections';

//  imports needed for the "dicsover" data part
import angular from 'angular';
import moment from 'moment';
import { getSort } from 'ui/doc_table/lib/get_sort';
import * as columnActions from 'ui/doc_table/actions/columns';
import 'ui/doc_table';
import 'ui/visualize';
import 'ui/notify';
import 'ui/fixed_scroll';
import 'ui/directives/validate_json';
import 'ui/filters/moment';
import 'ui/courier';
import 'ui/index_patterns';
import 'ui/state_management/app_state';
import 'ui/timefilter';
import 'ui/share';
import { VisProvider } from 'ui/vis';
import { DocTitleProvider } from 'ui/doc_title';
import { UtilsBrushEventProvider } from 'ui/utils/brush_event';
import PluginsKibanaDiscoverHitSortFnProvider from 'plugins/kibana/discover/_hit_sort_fn';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { FilterManagerProvider } from 'ui/filter_manager';
import { AggTypesBucketsIntervalOptionsProvider } from 'ui/agg_types/buckets/_interval_options';
import { stateMonitorFactory } from 'ui/state_management/state_monitor_factory';
import { getDefaultQuery } from 'ui/parse_query';

import { parseWithPrecision } from 'ui/kibi/utils/date_math_precision';
import 'plugins/kibana/discover/directives/timechart';

uiRoutes
  .when('/management/siren/datamodel', {
    template,
    reloadOnSearch: false,
    resolve: {
      objects: function (Private) {
        const resolveObjectsOnStartView = Private(ResolveObjectsOnStartViewProvider);
        return resolveObjectsOnStartView();
      }
    }
  })
  .when('/management/siren/datamodel/new/virtual_entity', {
    template,
    reloadOnSearch: false,
    resolve: {
      newVirtualEntity: function () {return true; },
      objects: function (Private) {
        const resolveObjectsOnCreateView = Private(ResolveObjectsOnCreateViewProvider);
        return resolveObjectsOnCreateView();
      }
    }
  })
  .when('/management/siren/datamodel/new/search_entity/:indexPattern?', {
    template,
    reloadOnSearch: false,
    resolve: {
      newSearchEntity: function () {return true; },
      objects: function (Private) {
        const resolveObjectsOnCreateView = Private(ResolveObjectsOnCreateViewProvider);
        return resolveObjectsOnCreateView();
      }
    }
  })
// Have a specific route for each type to avoid a clash with different routes on /ontology path
  .when('/management/siren/datamodel/SAVED_SEARCH/:id/:selectedTab?', {
    template,
    reloadOnSearch: false,
    mapBreadcrumbs($route, breadcrumbs) {
      const entity = $route.current.locals.objects.entity;
      return breadcrumbs.map(crumb => {
        if (crumb.id !== entity.id) {
          return crumb;
        }

        return {
          ...crumb,
          display: entity.label
        };
      });
    },
    resolve: {
      objects: function (Private, $route) {
        const resolveObjectsOnEntityView = Private(ResolveObjectsOnEntityViewProvider);
        return resolveObjectsOnEntityView($route.current.params.id);
      }
    }
  })
  .when('/management/siren/datamodel/VIRTUAL_ENTITY/:id/:selectedTab?', {
    template,
    reloadOnSearch: false,
    resolve: {
      objects: function (Private, $route) {
        const resolveObjectsOnEntityView = Private(ResolveObjectsOnEntityViewProvider);
        return resolveObjectsOnEntityView($route.current.params.id);
      }
    }
  });

function controller(
  $scope, $route, ontologyModel, kbnUrl, $injector, $timeout,
  $element, Private, createNotifier, $window,
  AppState, timefilter, config, Promise, $rootScope, courier, dataModel, confirmModal, kibiState) {

  const modals = Private(EntityModalsProvider);
  const indexOptionsHelpers = Private(IndexOptionsHelperFactory);
  const findRelations = Private(FindRelationsProvider);
  const fingerprints = Private(FingerprintsProvider);
  const messageBox = Private(MessageBoxProvider);
  const dataModelPermissions = Private(DataModelPermissionsProvider);

  ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
  dataModel = Private(OntologyWrapperProvider).forDataModel(dataModel);

  $scope.mustShowAutorelationButtons = function () {
    return dataModel.getAutoRelationsData().relations.length > 0;
  };

  const notify = createNotifier({
    location: 'Data Model editor'
  });
  $scope.EntityType = EntityType;
  $scope.DataModelErrorType = DataModelErrorType;

  $scope.$on('totalHitsChanged', (event,totalHits) => {
    $scope.totalHits = totalHits;
  });
  // START bits needed by 3 field tabs
  $scope.defaultIndex = config.get('defaultIndex');

  $scope.goEditIndexPatternField = function (field) {
    return kbnUrl.change(`management/siren/datamodel/editIndexPatternField/${$scope.entity.id}/field/${field.name}`);
  };
  $scope.goAddIndexPatternScriptedField = function () {
    return kbnUrl.change(`management/siren/datamodel/editIndexPatternField/${$scope.entity.id}/create-field/`);
  };
  $scope.$watch('indexPattern.fields', function (fields) {
    if (fields) {
      $scope.editSections = Private(IndicesEditSectionsProvider)($scope.indexPattern);
      $scope._refreshFilters();
    }
  });

  // kibi: moved from edit index pattern page
  $scope._refreshFilters = function () {
    const indexedFieldTypes = [];
    const scriptedFieldLanguages = [];
    if ($scope.indexPattern.fields) {
      $scope.indexPattern.fields.forEach(field => {
        if (field.scripted) {
          scriptedFieldLanguages.push(field.lang);
        } else {
          indexedFieldTypes.push(field.type);
        }
      });
    }

    $scope.indexedFieldTypes = _.uniq(indexedFieldTypes);
    $scope.scriptedFieldLanguages = _.uniq(scriptedFieldLanguages);
  };

  $scope.changeFilter = function (filter, val) {
    $scope[filter] = val || ''; // null causes filter to check for null explicitly
  };
  // kibi: end

  $scope.refreshFields = function () {
    const modal = messageBox({
      class: 'kibi-message-box--confirm',
      content: refreshFieldsPopupTemplate,
      buttons: ['Cancel', 'Refresh fields'],
      doFingerprints: true
    });

    return modal.show()
      .then(ok => {
        if (!ok) { return; }

        const { indexPattern, savedSearch: ssearch, entity } = $scope;

        // Inform fields table that we're about to refresh the fields list, this
        // should make sure it doesn't interfere with fingerprints refresh
        $scope.$broadcast('aboutToRefreshFieldsList');

        return fingerprints.delete(indexPattern.title)
          .then(() => indexPattern.refreshFields())
          .then(() => fingerprints.applyDefaultFieldMetadata(entity, ssearch)
            .then(() => modal.scope.doFingerprints && fingerprints
              .selectSavedSearches([ ssearch ])
              .then(fingerprints.calculate)
              .then(fps => fingerprints.applyToFieldMetadata(entity, ssearch, fps))
              .then(() => notify.info("Fingerprints generated successfully")))
            // Notify errors but not canceled operation, either way don't stop
            .catch(err => err && notify.error(err)))
          .finally(() => $scope.$broadcast('refreshedFieldsList'));
      });
  };

  config.bindToScope($scope, 'defaultIndex');

  $scope.setDefaultPattern = function () {
    config.set('defaultIndex', $scope.indexPattern.id);
  };
  // END bits needed by 3 field tabs

  $scope.newVirtualEntity = $route.current.locals.newVirtualEntity;
  $scope.newSearchEntity = $route.current.locals.newSearchEntity;
  $scope.savedSearch = $route.current.locals.objects.savedSearch;
  $scope.savedEid = $route.current.locals.objects.savedEid;
  $scope.entities = $route.current.locals.objects.entities;
  $scope.entity = $route.current.locals.objects.entity;
  $scope.entityRelations = $route.current.locals.objects.entityRelations;
  $scope.relationCountToDisplay = 0;
  $scope.autoRelationCountToDisplay = 0;

  if ($route.current.locals.objects.entity && $route.current.locals.objects.entity._objects.error) {
    $scope.errorType = $route.current.locals.objects.entity._objects.errorType;
    $scope.error = $route.current.locals.objects.entity._objects.error;
    $scope.errorRecoveredIndexPatternAttributes =  $route.current.locals.objects.entity._objects.errorRecoveredIndexPatternAttributes;
  }

  $scope.isRelationalGraphAvailable = $injector.has('sirenRelationalGraphDirective');

  function getSelectedTab() {
    const eidOrChildTabs = ['info', 'data', 'relations', 'graph'];
    let selectedTab = $route.current.params.selectedTab || 'info';
    if (
      $scope.entity &&
      (
        $scope.entity.type === EntityType.VIRTUAL_ENTITY ||
        ($scope.entity.type === EntityType.SAVED_SEARCH && $scope.entity.parentId !== null)
      ) &&
      eidOrChildTabs.indexOf($scope.selectedTab) === -1
    ) {
      // flip to info tab if you are on eid or child search
      selectedTab = 'info';
    }
    return selectedTab;
  }

  $scope.selectedTab = getSelectedTab();
  $scope.graphVisible = $scope.selectedTab === 'graph';
  $scope.graphControl = {};
  $scope.reloadGraph = () => {
    if ($scope.graphControl && $scope.graphControl.reload) {
      $scope.graphControl.reload();
    }
  };
  $scope.changeSelectedTab = function (tabName) {
    $scope.selectedTab = tabName;

    if (tabName === 'graph') {
      // make graph visible after user clicked on graph tab first time
      // this prevents to render graph in the corner when tab is not visible
      $scope.graphVisible = true;
    }
  };

  $scope.refreshEntities = function () {
    $scope.$broadcast('refresh.entities-sidebar');
  };

  $scope.showEntities = true;

  const changeTo = function (type, id, tab) {
    return kbnUrl.change('/management/siren/datamodel/{{type}}/{{id}}/{{tab}}', { type, id, tab });
  };

  $scope.changeSelectedEntity = function (entity) {
    if (entity && entity.type) {
      changeTo(entity.type, entity.id, $scope.selectedTab);
    }
  };

  const redirectTo = function (type, id, tab) {
    return kbnUrl.redirect('/management/siren/datamodel/{{type}}/{{id}}/{{tab}}', { type, id, tab });
  };

  const redirectToNextAvailableEntity = function () {
    return ontologyModel.getEntityList()
      .then(entities => {
        let found = _.find(entities, entity => {
          return entity.type === EntityType.SAVED_SEARCH && entity.parentId === null;
        });
        if (found) {
          return redirectTo(EntityType.SAVED_SEARCH, found.id, $scope.selectedTab);
        }
        found = _.find(entities, entity => {
          return entity.type === EntityType.VIRTUAL_ENTITY && entity.parentId === null;
        });
        if (found) {
          return redirectTo(EntityType.VIRTUAL_ENTITY, found.id, $scope.selectedTab);
        }

        return kbnUrl.redirect('/management/siren/datamodel/');
      });
  };

  const openDeleteSearchEntityModal = function () {
    const onConfirm = function () {
      dataModel.deleteSearch($scope.entity, $scope.savedSearch)
        .then(redirectToNextAvailableEntity)
        .catch(notify.error);
    };

    modals.deleteEntity(
      onConfirm,
      'Delete Search: ' + $scope.entity.label,
      'Are you sure you would like to delete it?'
    );
  };

  const openDeleteVirtualEntityModal = function () {
    const onConfirm = function () {
      dataModel.deleteEntityIdentifier($scope.entity)
        .then(redirectToNextAvailableEntity)
        .catch(notify.error);
    };

    modals.deleteEntity(
      onConfirm,
      'Delete Entity Identifier: ' + $scope.entity.label,
      'Are you sure you would like to delete it?'
    );
  };

  const openDeleteRootEntityModal = function () {
    // this is the root one print some warning and delete all children
    const childEntitiesToDelete = [];
    _.each($scope.entities, entity => {
      if (entity.parentId === $scope.entity.id) {
        childEntitiesToDelete.push(entity);
      }
    });
    const childNames = childEntitiesToDelete.map(entity => entity.label);
    const entitiesToDelete = childEntitiesToDelete.concat($scope.entity);

    const deleteEntities = function () {
      const promises = _.map(entitiesToDelete, entity => dataModel.deleteSearch(entity));
      Promise.all(promises)
        .then(redirectToNextAvailableEntity)
        .catch(notify.error);
    };

    if (childEntitiesToDelete.length > 0) {
      modals.deleteEntity(
        deleteEntities,
        `Delete Index Pattern Search: ${$scope.entity.label}`,
        `<div>
          The following child Searches will be deleted too:<br/>
          ${childNames.join('<br/>')}<br/>
          Are you sure you want to delete all searches?
        </div>`
      );
    } else {
      modals.deleteEntity(
        deleteEntities,
        `Delete Index Pattern Search: ${$scope.entity.label}.`,
        'Are you sure you want to delete the search?'
      );
    }
  };

  const deleteEntity = function () {
    if ($scope.entity.type === EntityType.SAVED_SEARCH) {
      if ($scope.entity.parentId !== null && $scope.entity.parentId !== 'null') {
        openDeleteSearchEntityModal();
      } else {
        openDeleteRootEntityModal();
      }
    } else if ($scope.entity.type === EntityType.VIRTUAL_ENTITY) {
      openDeleteVirtualEntityModal();
    }
  };

  $scope.saveObject = function () {
    if ($scope.entity.type === EntityType.VIRTUAL_ENTITY) {
      return dataModel.updateEntityIdentifier($scope.entity, $scope.savedEid)
        .then(savedId => {
          // A temporary autorelated EID can be created with a new id
          if (savedId === $scope.entity.id) {
            $scope.refreshEntities();
            $scope.reloadGraph();
          } else {
            changeTo($scope.entity.type, savedId, $scope.selectedTab);
          }
        });
    } else {
      if ($scope.savedSearch) {
        return $scope.opts.saveDataSource()     // Includes refreshEntities()
          .then(() => $scope.reloadGraph());
      } else {
        // Note:
        // Special case where underlying index did not exist
        //
        // User will be able to fix it on the edit page
        // the index-pattern will be recreated and saved with the same id
        // then we reload to correctly initialize the page again

        return dataModel.triggerBeforeSave()
          .then(() => {
            $window.location.reload();
          });
      }
    }
  };

  $scope.indexPatternError = false;
  $scope.$on('indexPatternError', (event, indexPatternError) => {
    $scope.indexPatternError = indexPatternError;
  });

  $scope.$on('indexPatternLabelEmpty', (event, indexPatternLabelEmpty) => {
    $scope.indexPatternLabelEmpty = indexPatternLabelEmpty;
  });

  $scope.isValid = function () {
    return !$scope.newVirtualEntity && !$scope.newSearchEntity &&
           !$scope.indexPatternError && !$scope.indexPatternLabelEmpty &&
           ($scope.entity && $scope.entity.label && !$scope.entity.$$scriptedLabelError) &&
           // disable if there were any errors, except if the error was about missing indices
           // as in this case user can edit/change the pattern and save
           ($scope.errorType === DataModelErrorType.INDEX_PATTERN_MISSING_DATA_INDICES || !$scope.errorType) &&
           !$scope.relationsError;
  };

  $scope.isDeleteValid = function () {
    return !$scope.newVirtualEntity && !$scope.newSearchEntity;
  };

  $scope.deleteObject = function () {
    deleteEntity();
  };

  // expose some methods to the navbar buttons
  [ 'isValid', 'saveObject', 'isDeleteValid', 'deleteObject'].forEach(name => {
    $element.data(name, $scope[name]);
  });

  function setCreateNewSearchButton($state) {
    if ($scope.entity.type !== EntityType.SAVED_SEARCH || $scope.entity.parentId !== null) {
      $scope.showCreateNewSearchButton = false;
      return;
    }

    const searchSourceJSON = JSON.parse($scope.savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
    const savedState = {
      filters: searchSourceJSON.filter, // tricky in searchSourceJSON is singular filter !!!
      query: searchSourceJSON.query
    };
    const actualState = {
      filters: $state.filters,
      query: $state.query
    };
    const stateDiff = kibiState.compareStates(actualState, savedState);
    $scope.showCreateNewSearchButton =
      !stateDiff.stateEqual &&
      (stateDiff.diff.extra.filters.length > 0 || stateDiff.diff.extra.query);
  }

  $scope.createNewChildSearch = function () {
    const onConfirm = function (label) {
      if (label) {
        $scope.entity.label = label;
        $scope.savedSearch.copyOnSave = true;
        $scope.savedSearch.title = label;
        $scope.opts.saveDataSource();
      }
    };
    modals.createNewChildSearchModal(onConfirm, $scope.entity.label);
  };

  const initializeSavedSearch = () => {
    $scope.intervalOptions = Private(AggTypesBucketsIntervalOptionsProvider);
    $scope.showInterval = false;

    $scope.intervalEnabled = function (interval) {
      return interval.val !== 'custom';
    };
    $scope.timefilter = timefilter;
    timefilter.enabled = true;

    // the saved savedSearch
    const Vis = Private(VisProvider);
    const docTitle = Private(DocTitleProvider);
    const brushEvent = Private(UtilsBrushEventProvider);
    const HitSortFn = Private(PluginsKibanaDiscoverHitSortFnProvider);
    const queryFilter = Private(FilterBarQueryFilterProvider);
    const filterManager = Private(FilterManagerProvider);

    const quickDashboard = Private(QuickDashboardProvider);
    const guessFields = Private(GuessFieldsProvider);

    $scope.indexPattern = $scope.savedSearch.searchSource.get('index');
    $scope.searchSource = $scope.savedSearch.searchSource;
    $scope.searchSource
      .highlightAll(true)
      .version(true);

    if ($scope.savedSearch.id) {
      docTitle.change($scope.savedSearch.title);
    }


    let stateMonitor;
    const $appStatus = $scope.appStatus = this.appStatus = {
      dirty: !$scope.savedSearch.id
    };
    const $state = $scope.state = new AppState(getStateDefaults());

    $scope.uiState = $state.makeStateful('uiState');

    function getStateDefaults() {
      return {
        query: $scope.searchSource.get('query') || getDefaultQuery(),
        sort: getSort.array($scope.savedSearch.sort, $scope.indexPattern),
        columns: $scope.savedSearch.columns.length > 0 ? $scope.savedSearch.columns : config.get('defaultColumns').slice(),
        index: $scope.indexPattern.id,
        interval: 'auto',
        filters: _.cloneDeep($scope.searchSource.getOwn('filter'))
      };
    }

    $state.index = $scope.indexPattern.id;
    $state.sort = getSort.array($state.sort, $scope.indexPattern);

    $scope.$watchCollection('state.columns', function () {
      $state.save();
    });

    $scope.opts = {
      // number of records to fetch, then paginate through
      sampleSize: config.get('discover:sampleSize'),
      timefield: $scope.indexPattern.timeFieldName,
      savedSearch: $scope.savedSearch,
      indexPatternList: [$scope.indexPattern],
      timefilter: $scope.timefilter
    };

    const init = _.once(function () {
      const showTotal = 5;
      $scope.failuresShown = showTotal;
      $scope.showAllFailures = function () {
        $scope.failuresShown = $scope.failures.length;
      };
      $scope.showLessFailures = function () {
        $scope.failuresShown = showTotal;
      };

      stateMonitor = stateMonitorFactory.create($state, getStateDefaults());
      stateMonitor.onChange((status) => {
        $appStatus.dirty = status.dirty || !$scope.savedSearch.id;
      });
      $scope.$on('$destroy', () => stateMonitor.destroy());

      $scope.updateDataSource()
        .then(function () {
          $scope.$listen(timefilter, 'fetch', function () {
            $scope.fetch();
          });

          $scope.$watchCollection('state.sort', function (sort) {
            if (!sort) return;

            // get the current sort from {key: val} to ["key", "val"];
            const currentSort = _.pairs($scope.searchSource.get('sort')).pop();

            // if the searchSource doesn't know, tell it so
            if (!angular.equals(sort, currentSort)) $scope.fetch();
          });

          // update data source when filters update
          $scope.$listen(queryFilter, 'update', function () {
            return $scope.updateDataSource().then(function () {
              $state.save();
            });
          });

          // update data source when hitting forward/back and the query changes
          $scope.$listen($state, 'fetch_with_changes', function (diff) {
            if (diff.indexOf('query') >= 0) $scope.fetch();
          });

          // fetch data when filters fire fetch event
          $scope.$listen(queryFilter, 'fetch', $scope.fetch);

          $scope.$watch('opts.timefield', function (timefield) {
            timefilter.enabled = !!timefield;
          });

          $scope.$watch('state.interval', function () {
            $scope.fetch();
          });

          $scope.$watch('vis.aggs', function () {
          // no timefield, no vis, nothing to update
          // kibi: added missing check about $scope.vis
            if (!$scope.opts.timefield || !$scope.vis) return;

            const buckets = $scope.vis.aggs.bySchemaGroup.buckets;

            if (buckets && buckets.length === 1) {
              $scope.bucketInterval = buckets[0].buckets.getInterval();
            }
          });

          $scope.$watchMulti([
            'rows',
            'fetchStatus'
          ], (function updateResultState() {
              let prev = {};
              const status = {
                LOADING: 'loading', // initial data load
                READY: 'ready', // results came back
                NO_RESULTS: 'none' // no results came back
              };

              function pick(rows, oldRows, fetchStatus) {
                // initial state, pretend we are loading
                if (rows == null && oldRows == null) return status.LOADING;

                const rowsEmpty = _.isEmpty(rows);
                // An undefined fetchStatus means the requests are still being
                // prepared to be sent. When all requests are completed,
                // fetchStatus is set to null, so it's important that we
                // specifically check for undefined to determine a loading status.
                const preparingForFetch = _.isUndefined(fetchStatus);
                if (preparingForFetch) return status.LOADING;
                else if (rowsEmpty && fetchStatus) return status.LOADING;
                else if (!rowsEmpty) return status.READY;
                else return status.NO_RESULTS;
              }

              return function () {
                const current = {
                  rows: $scope.rows,
                  fetchStatus: $scope.fetchStatus
                };

                $scope.resultState = pick(
                  current.rows,
                  prev.rows,
                  current.fetchStatus,
                  prev.fetchStatus
                );

                prev = current;
              };
            }()));

          $scope.searchSource.onError(function (err) {
            notify.error(err);
          }).catch(notify.fatal);

          function initForTime() {
            return setupVisualization().then($scope.updateTime);
          }

          return Promise.resolve($scope.opts.timefield && initForTime())
            .then(function () {
              init.complete = true;
              $state.replace();
              $scope.$emit('application.load');
            });
        });
    });

    $scope.opts.saveDataSource = function () {
      return $scope.updateDataSource()
        .then(function () {
          $scope.savedSearch.columns = $scope.state.columns;
          $scope.savedSearch.sort = $scope.state.sort;
          $scope.savedSearch.title = $scope.entity.label;

          const promise =
          $scope.savedSearch.copyOnSave ?
            dataModel.createSearch($scope.entity, $scope.savedSearch, $scope.entities) :
            dataModel.updateSearch($scope.entity, $scope.savedSearch);

          return promise
            .then(function (id) {
              stateMonitor.setInitialState($state.toJSON());

              if (id) {
                if (id !== $route.current.params.id) {
                  changeTo(EntityType.SAVED_SEARCH, id, $scope.selectedTab);
                } else {
                // Update defaults so that "reload saved query" functions correctly
                  $state.setDefaults(getStateDefaults());
                  docTitle.change($scope.savedSearch.lastSavedTitle);
                  $scope.refreshEntities();
                }
              }
            })
            .catch(notify.error);
        });
    };

    $scope.opts.fetch = $scope.fetch = function () {

      setCreateNewSearchButton($state);

      // ignore requests to fetch before the app inits
      if (!init.complete) return;

      $scope.updateTime();

      $scope.updateDataSource()
        .then(setupVisualization)
        .then(function () {
          $state.save();
          return courier.fetch();
        })
        .catch(notify.error);
    };

    $scope.searchSource.onBeginSegmentedFetch(function (segmented) {
      function flushResponseData() {
        $scope.hits = 0;
        $scope.faliures = [];
        $scope.rows = [];
        $scope.fieldCounts = {};
      }

      if (!$scope.rows) flushResponseData();

      const sort = $state.sort;
      const timeField = $scope.indexPattern.timeFieldName;

      /**
       * Basically an emum.
       *
       * opts:
       *   "time" - sorted by the timefield
       *   "non-time" - explicitly sorted by a non-time field, NOT THE SAME AS `sortBy !== "time"`
       *   "implicit" - no sorting set, NOT THE SAME AS "non-time"
       *
       * @type {String}
       */
      const sortBy = (function () {
        if (!_.isArray(sort)) return 'implicit';
        else if (sort[0] === '_score') return 'implicit';
        else if (sort[0] === timeField) return 'time';
        else return 'non-time';
      }());

      let sortFn = null;
      if (sortBy !== 'implicit') {
        sortFn = new HitSortFn(sort[1]);
      }

      $scope.updateTime();
      if (sort[0] === '_score') segmented.setMaxSegments(1);
      segmented.setDirection(sortBy === 'time' ? (sort[1] || 'desc') : 'desc');
      segmented.setSortFn(sortFn);
      segmented.setSize($scope.opts.sampleSize);

      // triggered when the status updated
      segmented.on('status', function (status) {
        $scope.fetchStatus = status;
      });

      segmented.on('first', function () {
        flushResponseData();
      });

      segmented.on('segment', notify.timed('handle each segment', function (resp) {
        if (resp._shards.failed > 0) {
          $scope.failures = _.union($scope.failures, resp._shards.failures);
          $scope.failures = _.uniq($scope.failures, false, function (failure) {
            return failure.index + failure.shard + failure.reason;
          });
        }
      }));

      segmented.on('mergedSegment', function (merged) {
        $scope.mergedEsResp = merged;
        $scope.hits = merged.hits.total;

        const indexPattern = $scope.searchSource.get('index');

        // the merge rows, use a new array to help watchers
        $scope.rows = merged.hits.hits.slice();

        notify.event('flatten hit and count fields', function () {
          let counts = $scope.fieldCounts;

          // if we haven't counted yet, or need a fresh count because we are sorting, reset the counts
          if (!counts || sortFn) counts = $scope.fieldCounts = {};

          $scope.rows.forEach(function (hit) {
            // skip this work if we have already done it
            if (hit.$$_counted) return;

            // when we are sorting results, we need to redo the counts each time because the
            // "top 500" may change with each response, so don't mark this as counted
            if (!sortFn) hit.$$_counted = true;

            const fields = _.keys(indexPattern.flattenHit(hit));
            let n = fields.length;
            let field;
            while (field = fields[--n]) {
              if (counts[field]) counts[field] += 1;
              else counts[field] = 1;
            }
          });
        });
      });

      segmented.on('complete', function () {
        if ($scope.fetchStatus.hitCount === 0) {
          flushResponseData();
        }

        $scope.fetchStatus = null;
      });
    }).catch(notify.fatal);

    // kibi: use parseWithPrecision()
    $scope.updateTime = function () {
      $scope.timeRange = {
        from: parseWithPrecision(timefilter.time.from, false, $rootScope.sirenTimePrecision),
        to: parseWithPrecision(timefilter.time.to, true, $rootScope.sirenTimePrecision)
      };
    };

    $scope.resetQuery = function () {
      throw 'Not implemented';
    };

    $scope.newQuery = function () {
      throw 'Not implemented';
    };

    $scope.updateDataSource = Promise.method(function updateDataSource() {
      $scope.searchSource
        .size($scope.opts.sampleSize)
        .sort(getSort($state.sort, $scope.indexPattern))
        .query(!$state.query ? null : $state.query)
        .set('filter', queryFilter.getFilters());
    });

    $scope.setSortOrder = function setSortOrder(columnName, direction) {
      $scope.state.sort = [columnName, direction];
    };

    // TODO: On array fields, negating does not negate the combination, rather all terms
    // kibi: added options parameter to support more like this queries
    $scope.filterQuery = function (field, values, operation, options) {
      $scope.indexPattern.popularizeField(field, 1);
      filterManager.add(field, values, operation, $state.index, options);
    };

    $scope.addColumn = function addColumn(columnName) {
      $scope.indexPattern.popularizeField(columnName, 1)
        .then(() => {
          columnActions.addColumn($scope.state.columns, columnName);
        });
    };

    $scope.removeColumn = function removeColumn(columnName) {
      $scope.indexPattern.popularizeField(columnName, 1)
        .then(() => {
          columnActions.removeColumn($scope.state.columns, columnName);
        });
    };

    $scope.moveColumn = function moveColumn(columnName, newIndex) {
      columnActions.moveColumn($scope.state.columns, columnName, newIndex);
    };

    $scope.toTop = function () {
      $window.scrollTo(0, 0);
    };

    // kibi: Added quick dashboard maker
    $scope.makeQuickDashboard = function () {
      return $scope.updateDataSource()
        .then(function () {
          $scope.savedSearch.columns = $scope.state.columns;
          $scope.savedSearch.sort = $scope.state.sort;

          return quickDashboard.create({
            entity: $scope.entity,
            savedSearch: $scope.savedSearch,
            fieldNames: $scope.state.columns,
            query: $scope.state.query,
            filters: $scope.state.filters,
            timeFilter: $scope.timefilter
          });
        });
    };

    $scope.guessFields = function () {
      const index = $scope.indexPattern;

      // The timeField is excluded since it's already implicitly considered by discover.
      // Apart from that, 'bad' fields are already filtered away by the ranking procedure.
      const timeField = index.fields.byName[index.timeFieldName];
      const fields = _.without(index.fields, timeField);

      return guessFields(index, fields, { savedSearch: $scope.savedSearch })
        .then(guessedFields => {
          if (!guessedFields) { return; }

          $scope.state.columns = _.map(guessedFields, 'name');
        });
    };
    // kibi: end

    let loadingVis;
    function setupVisualization() {
      // If we're not setting anything up we need to return an empty promise
      // kibi: added condition to check if there are any fields
      if (!$scope.opts.timefield  || $scope.indexPattern.fields.length === 0) return Promise.resolve();
      if (loadingVis) return loadingVis;

      const visStateAggs = [
        {
          type: 'count',
          schema: 'metric'
        },
        {
          type: 'date_histogram',
          schema: 'segment',
          params: {
            field: $scope.opts.timefield,
            interval: $state.interval
          }
        }
      ];

      // we have a vis, just modify the aggs
      if ($scope.vis) {
        const visState = $scope.vis.getEnabledState();
        visState.aggs = visStateAggs;

        $scope.vis.setState(visState);
        return Promise.resolve($scope.vis);
      }

      $scope.vis = new Vis($scope.indexPattern, {
        title: $scope.savedSearch.title,
        type: 'histogram',
        params: {
          addLegend: false,
          addTimeMarker: true
        },
        listeners: {
          click: function (e) {
            notify.log(e);
            timefilter.time.from = moment(e.point.x);
            timefilter.time.to = moment(e.point.x + e.data.ordered.interval);
            timefilter.time.mode = 'absolute';
          },
          brush: brushEvent($scope.state)
        },
        aggs: visStateAggs
      });

      $scope.searchSource.aggs(function () {
        $scope.vis.requesting();
        return $scope.vis.aggs.toDsl();
      });

      // stash this promise so that other calls to setupVisualization will have to wait
      loadingVis = new Promise(function (resolve) {
        $scope.$on('ready:vis', function () {
          resolve($scope.vis);
        });
      })
        .finally(function () {
        // clear the loading flag
          loadingVis = null;
        });

      return loadingVis;
    }

    init();
  };

  // kibi: Added quick relations builder
  $scope.buildAutoRelations = function () {
    // first check that the user has enough permission to start the process
    return Promise.all([
      // no need to check for READ permission for saved searches and index patterns
      // if user does not have these permissions the page will not load anyway
      dataModelPermissions.checkEntityIdPermissions(CrudType.CREATE),
      dataModelPermissions.checkRelationPermissions(CrudType.CREATE)
    ])
      .then(res => {
        if (_.every(res)) {
          return Promise.resolve()
            .then(findRelations.start)
            .then(findRelations.setAutoRelationsData)
            .then(() => $route.reload())
            .catch(err => { err && notify.error(err); });
        }
      });
  };

  $scope.saveAutoRelations = function () {
    confirmModal(
      'This will save all automatically generated relations. Are you sure you want to proceed?',
      {
        confirmButtonText: 'Save All Automatic Relations',
        onConfirm: () => {
          findRelations.applyAutoRelations()
            .then(({ savedAutoEntityIds }) => {
              // When the autorelations get applied, temporary entity
              // identifiers gain a new, persistent id - we have to redirect
              // there if our entity's id changed.
              const remappedEid = savedAutoEntityIds[$scope.entity.id];

              if (remappedEid) {
                redirectTo($scope.entity.type, remappedEid, $scope.selectedTab);
              } else {
                $route.reload();
              }
            })
            .catch(err => err && notify.error(err));
        }
      }
    );
  };

  $scope.clearAutoRelations = function () {
    confirmModal(
      'This will remove all automatically generated relations. Are you sure you want to proceed?',
      {
        confirmButtonText: 'Remove All Automatic Relations',
        onConfirm: () => {
          findRelations.clearAutoRelations();

          if ($scope.entity.autoRelation) {
            // Current entity was a temporary autorelation-created, which was
            // removed
            redirectToNextAvailableEntity();
          } else {
            $route.reload();
          }
        }
      }
    );
  };
  // kibi: end

  if ($scope.savedSearch) {
    initializeSavedSearch();
    $scope.$on('$destroy', $scope.savedSearch.destroy);
  }

  $scope.$watch('entity.id', (newValue, oldValue) => {
    if (newValue !== oldValue) {
      changeTo($scope.entity.type, $scope.entity.id, $scope.selectedTab);
    }
  });

  $scope.$on('siren:relationsChanged', function (event, obj) {
    $scope.relationCountToDisplay = obj.relationCount;
    $scope.autoRelationCountToDisplay = obj.autoRelationCount;
    $scope.relationsError = obj.relationError;
  });

  function refreshEntitieyRelations() {
    const resolveObjectsOnEntityView = Private(ResolveObjectsOnEntityViewProvider);
    const resolvedObject = resolveObjectsOnEntityView($route.current.params.id);
    resolvedObject.then(dataModelObject => {
      // We'll refresh entities too, since autorelations may change the state of
      // temporary EIDs
      $scope.entities = dataModelObject.entities;
      $scope.entityRelations = dataModelObject.entityRelations;

      $scope.refreshEntities();
    });
  };

  $scope.$on('siren:reload-entity-relations', refreshEntitieyRelations);
}

uiModules
  .get('apps/management', ['kibana'])
  .controller('DatamodelController', controller);
