import { find, has } from 'lodash';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { EntityValidatorProvider } from './entity_validator';
import { OntologyWrapperProvider } from 'ui/kibi/quick_relations/ontology_wrapper';

export function ResolveObjectsOnStartViewProvider(Private, kbnUrl, createNotifier, Promise) {
  return function resolveObjectsOnStartView() {
    const entityValidator = Private(EntityValidatorProvider);
    return entityValidator.findFirstValidEntity()
      .then(res => {
        if (res.entity) {
          kbnUrl.redirect(
            '/management/siren/datamodel/{{type}}/{{id}}/{{tab}}',
            { type: res.entity.type, id: res.entity.id, tab: 'info' }
          );
          return Promise.halt();
        }

        return { entities: res.permittedEntities };
      })
      .catch(error => {
        createNotifier({ location: 'Data model resolve' }).warning('Could not find valid entity.', error);
        return Promise.halt();
      });
  };
}

export function ResolveObjectsOnCreateViewProvider(Private, ontologyModel) {
  ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);

  return function resolveObjectsOnCreateView() {
    return ontologyModel.getEntityList()
      .then(entities => {
        return {
          entities
        };
      });
  };
}

export function ResolveObjectsOnEntityViewProvider(Private, ontologyModel, createNotifier, Promise) {
  ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
  const notifier = createNotifier({ location: 'Data model resolve' });

  // This should return entities and entity and savedSearch if there is a problem fetching savedSearch
  return async function resolveObjectsOnEntityView(id) {
    const objects = {};
    let brokenSearches = [];
    try {
      objects.entities = await ontologyModel.getEntityList();
      objects.entityRelations = await ontologyModel.getRelationsByDomain(id);
      brokenSearches = await ontologyModel.getBrokenSavedSearches();
    } catch (error) {
      notifier.error(`Could not resolve objects for entity '${id}': ${error.message}`);
      return Promise.halt();
    }

    if (brokenSearches.length > 0) {
      objects.entities = objects.entities.concat(brokenSearches);
    }
    objects.entity = find(objects.entities, 'id', id);

    if (objects.entity.type === EntityType.SAVED_SEARCH) {
      objects.savedSearch = objects.entity._objects.savedSearch || null;
      objects.indexPattern = objects.entity._objects.indexPattern ? objects.entity._objects.indexPattern.id : null;
    } else if (objects.entity.type === EntityType.VIRTUAL_ENTITY) {
      objects.savedEid = objects.entity._objects.savedEid;
    }
    return objects;
  };
}
