import _ from 'lodash';
import $ from 'jquery';
import dragula from 'dragula';
import { uiModules } from 'ui/modules';
import { CacheProvider } from 'ui/kibi/helpers/cache_helper';

uiModules
  .get('kibana')
  .directive('dashboardDraggableContainer', function ($rootScope, createNotifier,
    Private, dashboardGroups, savedDashboardGroups, savedDashboards, $timeout, kibiState) {
    const cache = Private(CacheProvider);
    const $scopes = new Map();
    // PLEASE READ: The declarated value of this constants CAN NOT be changed.
    const DUMMY_PLACEHOLDER_BETWEEN_DASHBOARDS = -1;
    const DUMMY_PLACEHOLDER_BETWEEN_GROUPS = -2;
    const DUMMY_PLACEHOLDER_FIRST_GROUP = -3;

    return {
      restrict: 'A',
      scope: true,
      controllerAs: 'dashboardDraggableContainerCtrl',
      controller($scope, $attrs, $parse, $element) {
        const element = $element.get(0);
        $scopes.set(element, $scope);

        $scope.$on('$destroy', function () {
          $scopes.delete(element);
        });

        this.linkDraggableItem = (el, $itemScope) => {
          const element = $(el).parent().get(0);
          $scope.drake.containers.push(element);
          $scopes.set(element, $itemScope);

          $itemScope.$on('$destroy', function () {
            $scopes.delete(element);
          });
        };

        this.linkContainer = (el, $itemScope) => {
          const element = $(el).get(0);
          $scope.drake.containers.push(element);
          $scopes.set(element, $itemScope);

          $itemScope.$on('$destroy', function () {
            $scopes.delete(element);
          });
        };

      },
      link($scope, $el, attr) {
        let notify = createNotifier({
          location: 'Dashboard Groups Editor'
        });
        let drake = dragula({
          isContainer: function (el) {
            return el.classList.contains('siren-graph-browser-vis')
            || el.classList.contains('dashboard-container');
          },
          accepts(el, target, source, sibling) {
            const sourceScope = $scopes.get(source);
            const targetScope = $scopes.get(target);
            if (targetScope && !sourceScope.isDashboard && !sourceScope.isVirtualGroup) {
            // the source is a group
              return !targetScope.isDashboard && !targetScope.isVirtualGroup && !targetScope.isDummy;
            }
            if (targetScope && sourceScope.isDashboard) {
              return targetScope.isDashboard || targetScope.isDummy;
            }
            if (targetScope && sourceScope.isVirtualGroup) {
              return targetScope.isDashboard || (targetScope.isDummy && !targetScope.isVirtualGroup);
            }
            return true;
          },
          moves(el, source, handle) {
            const itemScope = $scopes.get(source);
            if (!itemScope || !('dashboardDraggableItemCtrl' in itemScope)) {
              return; // only [draggable-item] is draggable
            }
            return true;
          },
          copy: true,
          mirrorContainer: $el.parent().get(0),
          removeOnSpill: false,
          grabDelay: 100
        });

        drake.on('drag', markDragging(true));
        drake.on('dragend', markDragging(false));
        drake.on('cancel', clearHoveredState());
        drake.on('over', over);
        drake.on('cloned', cloned);
        drake.on('drop', drop);
        $scope.drake = drake;

        $scope.$on('$destroy', function () {
          drake.destroy();
          drake = null;  // kibi: this helps to release the reference and completely garbage collect the drake object
          notify = null; // kibi: this helps to release the reference and completely garbage collect the notify object
          $el.remove();
        });

        function getDashboardId(elmt) {
          let dashId;
          // if it's a group child
          if (elmt.attributes['dash-id']) {
            dashId = elmt.attributes['dash-id'].value;
          } else {
            // else it's a virtual group (which is just a wrapped dashboard node)
            for (let i = 0; i < elmt.childNodes.length; i++) {
              const item = elmt.childNodes[i];
              if (item.tagName && item.tagName === 'DIV' && item.attributes['dash-id']) {
                dashId = item.attributes['dash-id'].value;
                break;
              }
            }
          }
          return dashId;
        }

        // Just emit the graph browser event here, as the drag shadow will trigger this event
        function cloned(clone, original, type) {
          const dashId = getDashboardId(original);

          if (dashId) {
            savedDashboards.get(dashId)
              .then(savedDashboard => {
                const searchId = savedDashboard.getMainSavedSearchId();
                kibiState.dragDashboardOnGraph({
                  showGraphMsg: true,
                  dashHasSearch: !!searchId
                });
              });
          }
        }

        function clearHoveredState() {
          kibiState.dragDashboardOnGraph({ showGraphMsg: false });
          return () => {
            $scopes.forEach(scope => {
              if (scope && scope.dashboardDraggableItemCtrl && scope.dashboardDraggableItemCtrl.getState) {
                delete scope.dashboardDraggableItemCtrl.getState().hovered;
                scope.$apply();
              }
            });
          };
        }

        function markDragging(isDragging) {
          return el => {
            if (!isDragging) {
              kibiState.dragDashboardOnGraph({ showGraphMsg: false });
            }
            const scope = $scopes.get($(el).parent().get(0));
            if (!scope) return;
            scope.isDragging = isDragging;
            $timeout(() => {
              $(el).addClass('dragging-pointer');
            });
            scope.$apply();
          };
        }

        function over(el, container, source) {
          // not used for now
        }

        function drop(el, target, source, sibling) {
          kibiState.dragDashboardOnGraph({ showGraphMsg: false });
          if (!target) {
            el.remove();
            drake.cancel(true);
            return;
          }
          $(el).removeClass('dragging-pointer');

          const sourceItemScope = $scopes.get(source);
          if (!sourceItemScope) return;
          const sourceItem = sourceItemScope.dashboardDraggableItemCtrl.getItem();
          const sourceGroup = sourceItemScope.dashboardDraggableItemCtrl.getGroup();

          if (target.classList.contains('siren-graph-browser-vis')) {
            const dashId = getDashboardId(el);

            el.remove();
            kibiState.dragDashboardOnGraph({ showGraphMsg: false });
            kibiState.dropDashboardOnGraph(dashId);

            drake.cancel(true);
          } else if (target.classList.contains('dashboard-container')) {
            // If we drop the dashboard outside the nav bar and
            // outside a valid div do nothing
            el.remove();
            drake.cancel(true);
          } else {
            $scope.isSaving = true;
            dashboardGroups.renumberGroups().then(() => {
              const targetItemScope = $scopes.get(target);
              if (!targetItemScope) return;
              const targetItem = targetItemScope.dashboardDraggableItemCtrl.getItem();
              const targetGroup = targetItemScope.dashboardDraggableItemCtrl.getGroup();

              if (sourceItemScope.isDashboard && !sourceGroup.virtual
                && targetItem < DUMMY_PLACEHOLDER_BETWEEN_DASHBOARDS) {
              // Removes a dashboard from one group and put in the correct order
                return savedDashboardGroups.get(sourceGroup.id).then(savedSourceGroup => {
                  const sourceItemId = savedSourceGroup.dashboards[sourceItem].id;
                  savedSourceGroup.dashboards.splice(sourceItem, 1);
                  return savedSourceGroup.save().then(() => {
                    return savedDashboards.get(sourceItemId).then(savedDashboard => {
                      if (targetItem === DUMMY_PLACEHOLDER_FIRST_GROUP) {
                        savedDashboard.priority = targetGroup.priority - 5;
                      } else {
                        savedDashboard.priority = targetGroup.priority - (sibling ? 5 : -5);
                      }
                      return savedDashboard.save();
                    });
                  });
                });
              }
              else if (sourceGroup.id === targetGroup.id && !sourceGroup.virtual && !targetGroup.virtual) {
              // Changes the dashboard order inside a group
                return savedDashboardGroups.get(sourceGroup.id).then(savedGroup => {
                  const dashboard = _.clone(savedGroup.dashboards[sourceItem]);
                  if (!sibling) {
                    savedGroup.dashboards.splice(sourceItem, 1);
                    savedGroup.dashboards.splice(targetItem, 0, dashboard);
                  } else {
                    savedGroup.dashboards.splice(sourceItem, 1);
                    const siblingItem = $scopes.get($(sibling).parent().get(0)).dashboardDraggableItemCtrl.getItem();
                    savedGroup.dashboards.splice(siblingItem, 0, dashboard);
                  }
                  return savedGroup.save();
                });
              }
              else if (sourceItemScope.isDashboard && !targetGroup.virtual
                     && targetItem > DUMMY_PLACEHOLDER_BETWEEN_GROUPS) {
              // Moves a dashboard from one group to another
                targetGroup.collapsed = false;
                return savedDashboardGroups.get(sourceGroup.id).then(savedSourceGroup => {
                  return savedDashboardGroups.get(targetGroup.id).then(savedTargetGroup => {
                    const actions = [];
                    const dashboard = _.clone(savedSourceGroup.dashboards[sourceItem]);
                    savedSourceGroup.dashboards.splice(sourceItem, 1);
                    actions.push(savedSourceGroup.save());
                    if (!sibling) {
                      savedTargetGroup.dashboards.splice(targetItem + 1, 0, dashboard);
                    } else {
                      const siblingItem = $scopes.get($(sibling).parent().get(0)).dashboardDraggableItemCtrl.getItem();
                      savedTargetGroup.dashboards.splice(siblingItem, 0, dashboard);
                    }
                    actions.push(savedTargetGroup.save());
                    return Promise.all(actions);
                  });
                });
              }
              else if (!sourceItemScope.isDashboard && sourceGroup.virtual
                     && targetItem < DUMMY_PLACEHOLDER_BETWEEN_DASHBOARDS) {
              // Changes the virtual group order
                return savedDashboards.get(sourceGroup.id).then(savedDashboard => {
                  if (targetItem === DUMMY_PLACEHOLDER_FIRST_GROUP) {
                    savedDashboard.priority = targetGroup.priority - 5;
                  } else {
                    savedDashboard.priority = targetGroup.priority - (sibling ? 5 : -5);
                  }
                  return savedDashboard.save();
                });
              }
              else if (!sourceItemScope.isDashboard && sourceGroup.virtual && !targetGroup.virtual
                     && targetItem > DUMMY_PLACEHOLDER_BETWEEN_GROUPS) {
              // Moves a virtual group into a group
                targetGroup.collapsed = false;
                return savedDashboardGroups.get(targetGroup.id).then(savedGroup => {
                  const dashboard = {
                    id: sourceGroup.id,
                    title: sourceGroup.title
                  };
                  if (!sibling) {
                    savedGroup.dashboards.splice(targetItem + 1, 0, dashboard);
                  } else {
                    const siblingItem = $scopes.get($(sibling).parent().get(0)).dashboardDraggableItemCtrl.getItem();
                    savedGroup.dashboards.splice(siblingItem, 0, dashboard);
                  }
                  return savedGroup.save();
                });
              }
              else if (!sourceItemScope.isDashboard && !sourceGroup.virtual) {
              // Changes the group order
                return savedDashboardGroups.get(sourceGroup.id).then(savedGroup => {
                  savedGroup.priority = targetGroup.priority + (sibling ? 5 : -5);
                  return savedGroup.save();
                });
              }
            })
              .then(cache.invalidate)
              .then(() => {
                source.remove();
                $scope.isSaving = false;
                $scope.$emit('kibi:dashboardgroup:changed', sourceGroup.id);
              })
              .catch((reason) => {
                $scope.isSaving = false;
                notify.error(reason);
              });
          }
        }

      }
    };

  });
