import angular from 'angular';
import _ from 'lodash';
import './dashboard_switcher';
import './dashboards_nav.less';
import './dashboard_top_toolbar';
import './dashboard_bottom_toolbar';
import './dashboard_nav_group_editor';
import './dashboard_draggable/dashboard_draggable_container';
import './dashboard_draggable/dashboard_draggable_item';
import './dashboard_draggable/dashboard_draggable_handle';
import './dashboard_draggable/dashboard_dummy_container';
import './new_dashboard_confirm/new_dashboard_confirm_promise';
import dashboardsNavTemplate from './dashboards_nav.html';
import { onDashboardPage } from 'ui/kibi/utils/on_page';
import { CacheProvider } from 'ui/kibi/helpers/cache_helper';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { DashboardViewMode } from 'plugins/kibana/dashboard/dashboard_view_mode';
import { HashedItemStoreSingleton } from 'ui/state_management/state_storage';
import { uiModules } from 'ui/modules';
import uiChrome from 'ui/chrome';

uiModules
  .get('kibana')
  .directive('dashboardsNav', ($rootScope, dashboardsNavState, globalNavState, createNotifier, dashboardGroups, kbnUrl,
    savedDashboards, Private, kibiState, $document, newDashboardConfirmPromise, $timeout, $window, getAppState, $location,
    useVisualizationForCountOnDashboard) => {
    return {
      restrict: 'E',
      replace: true,
      scope: true,
      template: dashboardsNavTemplate,
      link: ($scope, $element) => {
        const cache = Private(CacheProvider);
        const queryFilter = Private(FilterBarQueryFilterProvider);
        let notify = createNotifier({
          location: 'Dashboard Nav'
        });
        const COLLAPSED_WIDTH = 140;
        const HIDDED_WIDTH = 0;
        const SLIDER_WIDTH = 4;
        const BOTTOM_BAR_HEIGHT = 70;
        const DASHBOARD_TITLE_MARGIN_LEFT = 10;
        const DASHBOARD_TITLE_NO_INDICATORS_MARGIN = 8;
        const DASHBOARDS_INDICATORS_WIDTH = 70 + DASHBOARD_TITLE_MARGIN_LEFT;
        const VIRTUAL_DASHBOARDS_INDICATORS_WIDTH = DASHBOARDS_INDICATORS_WIDTH - DASHBOARD_TITLE_MARGIN_LEFT;
        const GROUPS_INDICATORS_WIDTH = 50;

        $scope.bar = $element;
        $scope.slider = $element.find('.dashboards-slider-handle');
        $scope.dashApp = $document.find('.app-container.dashboard-container');
        $scope.links = $element.find('.links');
        $scope.groupEditor = $element.find('.group-editor');
        $scope.isSaving = false;

        $scope.resizeParts = (count) => {
          $scope.bar.css('visibility', 'visible');
          $scope.links.css('visibility', 'visible');
          $scope.bar.css('width', count);
          $document.find('.toaster-container .toaster').css('margin-left', count);
          $scope.dashApp.css('margin-left', count);
          $scope.groupEditor.css('margin-left', count);
          $scope.links.css('width', count);
          $scope.slider.css('left', count - SLIDER_WIDTH);
          let parts = $element.find('.title');
          parts.css('width', (((count - GROUPS_INDICATORS_WIDTH) / count) * 100).toFixed(2) + '%');
          let value = (((count - DASHBOARDS_INDICATORS_WIDTH) / count) * 100).toFixed(2);
          parts = $element.find('.dashboard-nav-title');
          parts.css('width', value + '%');
          value = (((count - VIRTUAL_DASHBOARDS_INDICATORS_WIDTH) / count) * 100).toFixed(2);
          parts = $element.find('.dashboard-nav-title-virtual-group');
          parts.css('width', value + '%');
          parts = $element.find('.dashboard-nav-title.no-indicators');
          value = (((count - DASHBOARD_TITLE_MARGIN_LEFT - DASHBOARD_TITLE_NO_INDICATORS_MARGIN) / count) * 100).toFixed(2);
          parts.css('width', value + '%');
          parts = $element.find('.dashboard-nav-title-virtual-group.no-indicators');
          parts.css('width', '100%');
          if (count === HIDDED_WIDTH) {
            $scope.bar.css('visibility', 'hidden');
            $scope.links.css('visibility', 'hidden');
          }
        };

        function updateSidebarSize() {
          let width = !dashboardsNavState.isOpen() ? HIDDED_WIDTH : dashboardsNavState.navWidth();
          if (!uiChrome.isKibiNavbarVisible()) {
            width = 0;
          }
          $scope.resizeParts(width);
        }

        function updateGlobalNav() {
          $scope.isGlobalNavOpen = globalNavState.isOpen();
        }

        function updateDashboardsNav() {
          const isOpen = dashboardsNavState.isOpen();
          $scope.isDashboardsNavOpen = isOpen;

          const onGroupEditorOpen = dashboardsNavState.isGroupEditorOpen();
          $scope.isDashboardsNavGroupEditorOpen = onGroupEditorOpen;

          // Notify visualizations, e.g. the dashboard, that they should re-render.
          $rootScope.$broadcast('globalNav:update');

          updateSidebarSize();
        }

        updateGlobalNav();
        updateDashboardsNav();

        $scope.getLastNewDashboardName = (title) => {
          const regEx = /.*\s#([0-9]*)$/;
          let last = -1;
          dashboardGroups.getGroups().forEach(group => {
            if (!group.virtual && group.dashboards) {
              group.dashboards.forEach(dash => {
                if (dash.title.indexOf(title) === 0) {
                  const match = dash.title.match(regEx);
                  const matchNumber = match && match.length > 1 ? +match[1] : 0;
                  last = last < matchNumber ? matchNumber : last;
                }
              });
            } else if (group.virtual) {
              if (group.title.indexOf(title) === 0) {
                const match = group.title.match(regEx);
                const matchNumber = match && match.length > 1 ? +match[1] : 0;
                last = last < matchNumber ? matchNumber : last;
              }
            }
          });
          return last;
        };

        $scope.createDashboard = () => {
          const baseName = 'New Dashboard';
          const lastCopy = $scope.getLastNewDashboardName(baseName);
          const title = lastCopy < 0 ? baseName : baseName + ' #' + (lastCopy + 1);
          const options = {};
          newDashboardConfirmPromise(title, options)
            .then(resp => {
              let dash;
              $scope.isSaving = true;
              savedDashboards.get('')
                .then(savedDash => {
                  dash = savedDash;
                  dash.title = resp.title;
                  return savedDash.save();
                })
                // if user cancel the creating dashboard do nothing
                .then(function (id) {
                  if (!id) {
                    return Promise.reject();
                  }
                })
                .then(cache.invalidate)
                .then(() => {
                  // kibi: set appState if page is 'dashboard/new-dashboard/create'
                  if ($location.absUrl().includes('dashboard/new-dashboard/create')) {
                    const appState = getAppState();
                    appState.title = dash.title;
                    appState.description = dash.description;
                    appState.timeRestore = dash.timeRestore;
                    appState.panels = dash.panelsJSON ? JSON.parse(dash.panelsJSON) : [];
                    appState.options = dash.optionsJSON ? JSON.parse(dash.optionsJSON) : {};
                    appState.uiState = dash.uiStateJSON ? JSON.parse(dash.uiStateJSON) : {};
                    appState.query = { match_all: {} };
                    appState.filters = [];
                    appState.save();
                  };

                  $scope.isSaving = false;
                  notify.info('Dashboard was successfully created');
                  $rootScope.$broadcast('kibi:dashboardgroup:changed');
                  globalNavState.setOpen(false);
                  const groupsChangedEvent = function () {

                    const state = {
                      appState: {
                        viewMode: DashboardViewMode.EDIT
                      }
                    };
                    HashedItemStoreSingleton.setItem('kibi_appstate_param', JSON.stringify(state));

                    kbnUrl.change('dashboard/' + dash.id);
                    dashboardGroups.off('groupsChanged', groupsChangedEvent);
                  };
                  dashboardGroups.on('groupsChanged', groupsChangedEvent);
                })
                .catch (reason => {
                  $scope.isSaving = false;
                  notify.error(reason);
                });
            });
        };

        $scope.restoreCollapsedGroupState = () => {
          if (!dashboardsNavState.areCollapsedGroupsSet()) {
            dashboardGroups.getGroups().forEach(group => {
              group.collapsed = true;
            });
          } else {
            const collapsedGroups = dashboardsNavState.collapsedGroups();
            dashboardGroups.getGroups().forEach(group => {
              if (_.indexOf(collapsedGroups, group.id) >= 0) {
                group.collapsed = true;
              }
            });
          }
        };

        $scope.persistCollapsedGroupState = () => {
          const collapsedGroups = [];
          dashboardGroups.getGroups().forEach(group => {
            if (!group.virtual && group.collapsed) {
              collapsedGroups.push(group.id);
            }
          });
          dashboardsNavState.setCollapsedGroups(collapsedGroups);
        };

        $scope.getLastNewDashboardGroupName = (title) => {
          const regEx = /.*\s#([0-9]*)$/;
          let last = -1;
          dashboardGroups.getGroups().forEach(group => {
            if (!group.virtual && group.title.indexOf(title) === 0) {
              const match = group.title.match(regEx);
              const matchNumber = match && match.length > 1 ? +match[1] : 0;
              last = last < matchNumber ? matchNumber : last;
            }
          });
          return last;
        };

        $scope.newDashboardGroup = event => {
          event.preventDefault();
          $scope.isSaving = true;
          const baseName = 'New Group';
          const lastCopy = $scope.getLastNewDashboardGroupName(baseName);
          const title = lastCopy < 0 ? baseName : baseName + ' #' + (lastCopy + 1);
          dashboardGroups.newGroup(title).then((groupId) => {
            $scope.isSaving = false;
            if (groupId) {
              notify.info('New dashboard group was successfully created');
              $rootScope.$broadcast('kibi:dashboardgroup:changed', groupId);
              $timeout(() => {
                const groupElement = $element.find('#kibi-dashboard-group-' + groupId);
                if (groupElement.length > 0) {
                  const offset = groupElement[0].offsetTop + groupElement[0].scrollHeight;
                  dashboardsNavState.setScrollbarPos(offset);
                  $scope.links[0].scrollTop = dashboardsNavState.scrollbarPos();
                }
              }, 1000);
            }
          })
            .catch (reason => {
              $scope.isSaving = false;
              notify.error(reason);
            });
        };

        $scope.onClearFilter = () => {
          kibiState.resetFiltersQueriesTimes()
            .then(() => {
              const dashboardIds = _(dashboardGroups.getGroups())
                .filter(g => !g.collapsed || g.virtual)
                .map('dashboards')
                .flatten()
                .map('id')
                .tap(ids => {
                  _.each(ids, id => {
                    useVisualizationForCountOnDashboard.set(id, false);
                  });
                })
                .value();
              return dashboardGroups.updateMetadataOfDashboardIds(dashboardIds)
                .then(() => queryFilter.emit('update'));
            });
        };

        $scope.dragging = false;

        $scope.startSlide = event => {
          $document.on('mousemove touchmove', $scope.moveSlide);
          $document.on('mouseup touchend', $scope.stopSlide);
          $scope.dragging = true;
          event.preventDefault();
        };
        $scope.slider.on('mousedown touchstart', $scope.startSlide);

        $scope.moveSlide = event => {
          if ($scope.dragging) {
            const ev = event.originalEvent.changedTouches ? event.originalEvent.changedTouches[0] : event.originalEvent;
            let count = ev.pageX - $scope.bar[0].offsetLeft;
            count = count < (COLLAPSED_WIDTH - SLIDER_WIDTH) ? COLLAPSED_WIDTH : count;
            $scope.resizeParts(count);
          }
        };

        $scope.stopSlide = event => {
          $document.off('mousemove touchmove', $scope.moveSlide);
          $document.off('mouseup touchend', $scope.stopSlide);
          $rootScope.$broadcast('globalNav:update');
          $scope.dragging = false;
          dashboardsNavState.setNavWidth($scope.bar.width() - SLIDER_WIDTH);
          dashboardsNavState.setOpen(true);
        };

        $scope.$on('kibi:dashboardGroups:updated', function () {
          $timeout(() => {
            updateSidebarSize();
            $scope.restoreCollapsedGroupState();
          }, 500);
        });

        $scope.$on('kibi:dashboardGroup:expand', function () {
          $timeout(() => {
            updateSidebarSize();
          });
        });

        $timeout(() => {
          updateSidebarSize();
          $scope.links[0].scrollTop = dashboardsNavState.scrollbarPos();
        }, 200);
        $scope.restoreCollapsedGroupState();

        const scrollHandler = function (event) {
          dashboardsNavState.setScrollbarPos(event.target.scrollTop);
        };

        $scope.links.on('scroll', scrollHandler);

        const dash = kibiState.getDashboardOnView();
        if (dash) {
          $scope.dashboardLoaded = dash.id;
        }
        $scope.$watch(kibiState.getDashboardOnView, dash => {
          if (dash) {
            $scope.dashboardLoaded = dash.id;
            $timeout(() => {
              if ($scope.dashboardLoaded) {
                const dashboard = $element.find('.active');
                if (dashboard.length > 0) {
                  const offset = dashboard.offset().top - dashboardsNavState.scrollbarPos();
                  if (offset > $window.innerHeight) {
                    dashboardsNavState.setScrollbarPos(offset);
                    $scope.links[0].scrollTop = dashboardsNavState.scrollbarPos();
                  }
                }
              }
            }, 500);
          }
        });

        const globalNavStateOff = $rootScope.$on('globalNavState:change', () => {
          updateGlobalNav();
        });
        const dashboardsNavStateOff =  $rootScope.$on('dashboardsNavState:change', () => {
          updateDashboardsNav();
        });

        const resize = function () {
          const $container = angular.element($element.find('.links')[0]);
          const h = $element.height() - BOTTOM_BAR_HEIGHT;
          $container.height(Math.max(20, h));
        };
        resize();

        // Re-render if the window is resized
        const $windowElement = angular.element($window);
        $windowElement.on('resize', resize);


        $scope.$on('$destroy', () => {
          globalNavStateOff();
          dashboardsNavStateOff();

          $scope.links.off('scroll', scrollHandler);
          $scope.slider.off('mousedown touchstart', $scope.startSlide);
          $document.off('mousemove touchmove', $scope.moveSlide);
          $document.off('mouseup touchend', $scope.stopSlide);
          $windowElement.off('resize', resize);

          notify = null; // this allow to completely garbage collect the object from memory

          $timeout(() => {
            $document.find('.toaster-container .toaster').css('margin-left', 0);
          }, 1);
          $scope.persistCollapsedGroupState();
        });

      }
    };
  });
