import _ from 'lodash';
import React from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';
import { EuiSideNav } from '@elastic/eui';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';

function labelComparator(node1, node2) {
  const label1 = node1.entity.label;
  const label2 = node2.entity.label;
  return label1.localeCompare(label2, 'en', { sensitivity: 'base' });
};

function sortAlpha(a) {
  a.sort(labelComparator);
  for (let i = 0; i < a.length; i++) {
    const subNodes = a[i].nodes;
    if (subNodes && subNodes.length > 0) {
      sortAlpha(subNodes);
    }
  }
};

export class EntityNavigator extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      folded: false
    };
    if (this.props.selectedEntity) {
      this.state.selectedEntityId = this.props.selectedEntity.id;
    }
    if (this.props.parent && this.props.parent.folded) {
      this.state.folded = this.props.parent.folded;
    }
  }

  treeModel() {
    const rootSavedSearchItems = [];
    const rootVirtualEntityItems = [];

    _.each(this.props.entities, entity => {
      if (entity.parentId === null) {
        if (
          entity.type === EntityType.SAVED_SEARCH &&
          (this.props.entityTypeToShow === EntityType.SAVED_SEARCH || this.props.entityTypeToShow === 'ALL')
        ) {
          // here we are wrapping the entity into another object
          // so we could safely place nodes avoiding modification of entity itself
          rootSavedSearchItems.push({
            entity,
            nodes: []
          });
        } else if (
          entity.type === EntityType.VIRTUAL_ENTITY &&
          (this.props.entityTypeToShow === EntityType.VIRTUAL_ENTITY || this.props.entityTypeToShow === 'ALL')
        ) {
          rootVirtualEntityItems.push({
            entity,
            nodes: []
          });
        }
      }
    });

    if (this.props.showChildren === true) {
      _.each(this.props.entities, entity => {
        if (entity.parentId !== 'null' && entity.parentId !== null) {
          if (entity.type === EntityType.SAVED_SEARCH) {
            // find parent for now ok as we support only 1 level nesting
            for (let i = 0; i < rootSavedSearchItems.length; i++) {
              if (rootSavedSearchItems[i].entity.id === entity.parentId) {
                // check if already added
                const found = _.find(rootSavedSearchItems[i].nodes, node => {
                  return node.entity.id === entity.id;
                });
                if (!found) {
                  rootSavedSearchItems[i].nodes.push({
                    entity,
                    nodes: []
                  });
                }
              }
            }
          } else if (entity.type === EntityType.VIRTUAL_ENTITY) {
            // for now do nothing as virtual entities have no children
          }
        }
      });
    }

    sortAlpha(rootSavedSearchItems);
    sortAlpha(rootVirtualEntityItems);

    const data = [];
    if (rootSavedSearchItems.length > 0) {
      data.push({
        entity: {
          id: 'Searches',
          label: 'Searches'
        },
        nodes: rootSavedSearchItems
      });
    }
    if (rootVirtualEntityItems.length > 0) {
      data.push({
        entity: {
          id: 'Identifiers',
          label: 'Entity Identifiers',
        },
        nodes: rootVirtualEntityItems
      });
    }

    return data;
  }

  selectEntity(entityId) {
    const entity = _.find(this.props.entities, ent => ent.id === entityId);

    this.setState({ selectedEntityId: entityId });
    this.props.changeSelectedEntity({ entity: entity });
  }

  toggleParentFolded() {
    this.setState({ folded: !this.state.folded });
  }

  getExpandableSection() {
    const getEntityIcon = (entity) => {
      if (entity && entity.icon) {
        return (
          <span className={ 'entity-icon ' + entity.icon } style={{ color: entity.color }}/>
        );
      }
      return;
    };

    if (this.props.foldable) {
      const isFolded = this.state.folded;
      const foldableIcon = isFolded ? 'fa-chevron-circle-down' : 'fa-chevron-circle-up';
      const entity = this.props.selectedEntity;
      return (
        <div className={ 'siren-entity-folded' + (isFolded ? '' : ' bordered') }
          onClick={ () => this.toggleParentFolded() } >
          <div className='siren-entity-folded-node'>
            { getEntityIcon(entity) }
            <span className='entity-label'>
              { entity.label }
            </span>
          </div>
          <span className={ 'kuiIcon siren-entity-folded-expand ' + foldableIcon } />
          <div className="clear" />
        </div>
      );
    }
    return;
  }

  createItem(id, name, data = {}) {
    // NOTE: Duplicate `id` values will cause collisions.
    return {
      ...data,
      id,
      name
    };
  };

  getOntology(treeModel) {
    const getTreeItems = (tree, forceOpen, isHeaderElement) => {
      return _.reduce(tree, (array, elmt) => {
        const entity = elmt.entity;

        let subItems = null;
        for (let i = 0; i < elmt.nodes.length; i++) {
          if (!subItems) {
            subItems = [];
          }
          const subElmt = elmt.nodes[i];
          subItems.push(getTreeItems([subElmt], true)[0]);
        }

        const opts = {
          items: subItems,
          isSelected: this.state.selectedEntityId === entity.id
        };
        // Makes 'Searches' and 'Entity Identifiers' not clickable
        if (!isHeaderElement) {
          opts.onClick = () => this.selectEntity(entity.id);

          if (entity.type === EntityType.SAVED_SEARCH && !this.props.hideIndexPattern && !entity.parentId) {
            let indexPattern;
            if (entity._objects && entity._objects.indexPattern) {
              indexPattern = entity._objects.indexPattern.title;
            } else if (entity.indexPattern) {
              indexPattern = entity.indexPattern;
            }

            let className;
            if (entity.icon) {
              className = entity.icon + ' entity-icon entity-icon-spacer';
            } else {
              className = 'entity-icon-spacer icon-placeholder entity-icon far';
            }
            opts.icon = (
              <div data-test-subj='index-pattern-icon'>
                <div className='index-pattern-container'>
                  <div className='index-pattern'>
                    {indexPattern}
                  </div>
                </div>
                <i className={className} style={{ color: entity.color }}/>
              </div>
            );
          } else {
            if (entity.icon) {
              opts.icon = <div ><i className={ entity.icon + ' entity-icon' } style={{ color: entity.color }}/></div>;
            } else {
              opts.icon = <div><i className='entity-icon far' /></div>;
            }
          }
        }

        if (subItems && subItems.length > 0 && forceOpen) {
          opts.isOpen = forceOpen;
        }

        if (entity.hasAutoRelations) {
          opts.autorelationbg = '';
        }

        if (entity._objects && entity._objects.error) {
          opts.errored = '';
        }

        let label = entity.label;

        // Added to make building integration tests easier.
        opts['data-test-subj'] = label;

        if (entity.autoRelation) {
          opts['new-autorelation-entity'] = '';
          label = '*' + label;
        }

        array.push(this.createItem(entity.id, label, opts));

        return array;
      }, []);
    };

    if (treeModel && !(this.props.foldable && this.state.folded)) {
      if (treeModel.length === 0) {
        return (
          <div className="siren-entity-nav-no-data">
            There are no Index Pattern Searches or Entity Identifiers
          </div>
        );
      } else {
        const sideNav = getTreeItems(treeModel, true, true);
        const sideNavStyle = { width: '88%' };
        return (
          <EuiSideNav
            mobileTitle="Navigate Ontology"
            items={sideNav}
            style={sideNavStyle}
          />
        );
      }
    }
    return;
  }

  getUpdating(treeModel) {
    if (!treeModel) {
      return (
        <div className="siren-entity-nav-loader">
          <i className="kuiIcon fa-sync fa-spin"></i>
          Updating
        </div>
      );
    }
    return;
  }

  render() {
    // Derived data should be calculated at this point. Memoization should be implemented in
    // case we notice performance issues.
    const treeModel = this.treeModel();

    return (
      <div className="siren-entity-nav">
        { this.getExpandableSection() }
        { this.getOntology(treeModel) }
        { this.getUpdating(treeModel) }
      </div>
    );
  }
}

EntityNavigator.propTypes = {
  parent: PropTypes.object,
  foldable: PropTypes.bool,
  selectedEntity: PropTypes.object,
  entities: PropTypes.arrayOf(PropTypes.object).isRequired,
  entityTypeToShow: PropTypes.string,
  showChildren: PropTypes.bool,
  changeSelectedEntity: PropTypes.func,
  hideIndexPattern: PropTypes.bool
};

EntityNavigator.defaultProps = {
  foldable: false,
  entityTypeToShow: 'ALL',
  showChildren: false,
  hideIndexPattern: false
};

export function entityNavigator(container, $parent, foldable, selectedEntity, entities, entityTypeToShow,
  showChildren, changeSelectedEntity, hideIndexPattern) {
  const element = (
    <EntityNavigator parent={$parent}
      foldable={foldable}
      selectedEntity={selectedEntity}
      entities= {entities}
      entityTypeToShow={entityTypeToShow}
      showChildren={showChildren}
      changeSelectedEntity={changeSelectedEntity}
      hideIndexPattern={hideIndexPattern}
    >
    </EntityNavigator>
  );
  ReactDOM.render(element, container);
}
