import angular from 'angular';
import _ from 'lodash';
import { uiModules } from 'ui/modules';
import optionsTemplate from './siren_entity_options.html';
import './siren_entity_options.less';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { QueryBuilderFactory } from 'ui/kibi/helpers/query_builder';
import { previewFittedSizeHelper } from './helpers/preview_fitted_size_helper';
import { OntologyWrapperProvider } from 'ui/kibi/quick_relations/ontology_wrapper';

uiModules.get('apps/management')
  .directive('sirenEntityOptions',
    function ($compile, Private, ontologyModel, dataModel, kbnUrl, createNotifier, kibiState, indexPatterns, timefilter,
      $timeout, $document) {
      return {
        restrict: 'E',
        template: optionsTemplate,
        scope: {
          selectedTab: '=',
          entity: '=',
          savedSearch: '=?',
          indexPattern: '=?',
          errorRecoveredIndexPatternAttributes: '=?',
          selectedTab: '='
        },
        link: function ($scope) {
          const queryBuilder = Private(QueryBuilderFactory);
          ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
          dataModel = Private(OntologyWrapperProvider).forDataModel(dataModel);

          $scope.scriptedLabelError = false;
          $scope.EntityType = EntityType;
          $scope.matchAllQueryString = JSON.stringify({
            match_all: {}
          });
          $scope.emptyArrayString = '[]';

          const indexOptionsHelpers = Private(require('./helpers/index_options_helper'));

          function getSearchSourceJSON(savedSearch) {
            if (savedSearch && savedSearch.kibanaSavedObjectMeta && savedSearch.kibanaSavedObjectMeta.searchSourceJSON) {
              try {
                const searchSourceJSON = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
                if (searchSourceJSON.query) {
                  if (kibiState._isDefaultQuery(searchSourceJSON.query) || _.isEqual(searchSourceJSON.query, { match_all: {} })) {
                    delete searchSourceJSON.query;
                  }
                }

                if (searchSourceJSON.filter && searchSourceJSON.filter.length === 0) {
                  delete searchSourceJSON.filter;
                }
                return searchSourceJSON;
              } catch (e) {
                // do nothing
              }
            }
            return null;
          }

          function constructQuery(searchSourceJSON, indexPattern) {
            let timeFilterFromTimeAndPattern = null;
            if (timefilter.enabled) {
              const time = {
                mode: timefilter.time.mode,
                from: timefilter.time.from,
                to: timefilter.time.to
              };
              timeFilterFromTimeAndPattern = kibiState.turnTimeAndIndexPatternIntoFilter(time, indexPattern);
            }
            const filters = searchSourceJSON.filter ? searchSourceJSON.filter : [];
            const queries = searchSourceJSON.query ? [searchSourceJSON.query] : [];
            const query = queryBuilder(filters, queries, timeFilterFromTimeAndPattern);
            return query.query;
          }

          let addLabelPreviewPopupInProgress = false;
          function addLabelPreviewPopup() {
            if (!$scope.entity || !$scope.savedSearch || $scope.selectedTab !== 'info' || addLabelPreviewPopupInProgress) {
              return;
            }

            const searchSourceJSON = getSearchSourceJSON($scope.savedSearch);
            if (!searchSourceJSON) {
              return;
            }

            addLabelPreviewPopupInProgress = true;
            const result = document.getElementsByClassName('label-preview');
            const $el = angular.element(result);
            $el.qtip('destroy', true);
            const api = $el.qtip('api');
            // build html content
            return indexPatterns.get(searchSourceJSON.index)
              .then(function (indexPattern) {
                const query = constructQuery(searchSourceJSON, indexPattern);
                return indexOptionsHelpers.getInstanceLabelPreviewContent($scope.entity.instanceLabel, indexPattern.title, query)
                  .then((html) =>{
                    if (api) {
                      api.set('content.text', html);
                    } else {
                      $el.qtip({
                        content: {
                          text: html,
                        },
                        position: {
                          my: 'left center',
                          at: 'right center',
                          adjust: {
                            scroll: false
                          }
                        },
                        hide: {
                          event: 'unfocus click'
                        },
                        show: 'click',
                        events: {
                          show: (event, qtipApi) => {
                            // $timeout 0, to run in the "next tick" to be able to to get the tooltip dimensions after it is rendered
                            $timeout(() => {
                              previewFittedSizeHelper(qtipApi, $document, $el);
                            });
                          }
                        },
                        style: {
                          classes: 'qtip-light qtip-rounded qtip-shadow entity-label-preview'
                        }
                      });
                    }
                    $compile(html)($scope);
                  });
              })
              .then(() => {
                delete $scope.entity.$$scriptedLabelError;
                addLabelPreviewPopupInProgress = false;
              })
              .catch(err => {
                $scope.entity.$$scriptedLabelError = true;
                addLabelPreviewPopupInProgress = false;
              });
          };

          // Handle new line char
          if ($scope.entity.instanceLabel
            && $scope.entity.instanceLabel.value
            && $scope.entity.instanceLabel.value.includes('\n')) {
            $scope.entity.instanceLabel.value = $scope.entity.instanceLabel.value.replace(/\n/g, '\\n');
          }

          if ($scope.entity.parentId) {
            ontologyModel.getEntityMap()
              .then(entityMap => {
                const entity = entityMap[$scope.entity.parentId];
                $scope.parentEntityLabel = entity.label;
              });
          }

          $scope.$watch('selectedTab', addLabelPreviewPopup);
          timefilter.on('update', addLabelPreviewPopup);

          // Watching autorelated entities for changes - changes are implicitly
          // saved in session storage without users clicking 'save'. Explicit
          // click on 'save' button makes the entity permanent (saved on ES).
          let deregisterEntityWatcher;
          $scope.$watch('entity', function () {
            if (deregisterEntityWatcher) {
              deregisterEntityWatcher();
            }

            if (!$scope.entity.autoRelation) {
              return;
            }

            let initialized = false;
            deregisterEntityWatcher = $scope.$watchMulti([
              'entity.label', 'entity.icon', 'entity.color', 'entity.shortDescription'
            ], function () {
              if (initialized) {
                dataModel.updateAutoRelatedEntity($scope.entity);
              } else {
                initialized = true;
              }
            });
          });

          let deregisterOnLabelWatcher;
          $scope.$watch('savedSearch', savedSearch => {
            addLabelPreviewPopup();
            if (deregisterOnLabelWatcher) {
              deregisterOnLabelWatcher(); // prevent registering the same watcher multiple times
            }
            deregisterOnLabelWatcher = $scope.$watch('entity.instanceLabel.value', addLabelPreviewPopup);
          });

          $scope.$on('$destroy', () => {
            timefilter.off('update', addLabelPreviewPopup);
            if (deregisterOnLabelWatcher) {
              deregisterOnLabelWatcher();
            }
          });
        }
      };
    });
