import _ from 'lodash';
import angular from 'angular';
import Promise from 'bluebird';
import { uiModules } from 'ui/modules';
import EntityRelationsTemplate from './siren_entity_relations.html';
import 'plugins/investigate_core/ui/directives/siren_entity_select/siren_entity_select';
import 'plugins/investigate_core/ui/directives/siren_entity_relations/siren_entity_relations.less';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { FindRelationsProvider } from 'ui/kibi/quick_relations/find_relations';
import { OntologyWrapperProvider } from 'ui/kibi/quick_relations/ontology_wrapper';
import { RelationExplorerProvider } from 'ui/kibi/quick_relations/relation_explorer';
import { KibiSelectHelperFactory } from 'ui/kibi/directives/kibi_select_helper';
import { validateRelation } from './siren_validate_relation';
import uuid from 'uuid';

uiModules.get('apps/management')
  .directive(
    'sirenEntityRelations',
    function (Private, $route, $rootScope, $timeout, createNotifier, kbnUrl, confirmModalPromise, dataModel,
      ontologyModel, jdbcDatasources) {

      ontologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
      dataModel = Private(OntologyWrapperProvider).forDataModel(dataModel);
      const selectHelper = Private(KibiSelectHelperFactory);

      return {
        restrict: 'E',
        template: EntityRelationsTemplate,
        scope: false,
        // from parent scope we expect
        // entities
        // entity
        // entityRelations

        link: function ($scope) {
          let notify = createNotifier({ location: 'Entity relation' });

          $scope.EntityType = EntityType;
          $scope.typeErrorMessage = 'Source and target fields must have the same data type.';
          // Cache the first selectedMenuItem from the eid tree (top left -  not the dropdown in the config)
          let cachedSelectedMenuItem = $scope.selectedMenuItem || null;
          let relationLabelPairMap = {};
          const entityFieldTypeMap = new Map();
          const virtualIndexMap = new Map();
          const relationRangeFieldsTypeMap = new Map();

          const relationFormData = function (relation) {
            return [
              relation.domain.id, relation.domain.field,
              relation.range.id, relation.range.field,
              relation.directLabel, relation.inverseLabel
            ];
          };

          const relationEndpointsData = function (relation) {
            return [
              relation.domain.id, relation.domain.field,
              relation.range.id, relation.range.field
            ];
          };

          const toString = function (relations, dataFn) {
            return JSON.stringify(_.map(relations, dataFn));
          };

          const actualRelations = function () {
            return _.filter($scope.relations, rel => !rel.autoRelation);
          };

          const refreshEntityFormStatus = function () {
            const allRelationsString = toString($scope.relations, relationFormData);
            const actualRelationsString = toString(actualRelations(), relationFormData);

            $scope.entityForm[(
              $scope.pristineActualRelationsStr === allRelationsString
            ) ? '$setPristine' : '$setDirty']();

            $scope.actualRelationsDirty = $scope.pristineActualRelationsStr !== actualRelationsString;
          };

          const refreshRelations = function (entityRelations) {
            $scope.relations = [];

            if (entityRelations !== undefined) {
              // if there are equivalent relations hide them on UI
              const selfRelations = new Set();
              const relationsSubSet = [];

              _.each(entityRelations, rel => {
                if (rel.domain.id === rel.range.id) {
                  if (selfRelations.has(rel.inverseOf)) {
                    return;
                  } else {
                    selfRelations.add(rel.id);
                  }
                }
                relationsSubSet.push(rel);
              });

              // sort relations
              let sortedRelations = relationsSubSet;
              if (sortedRelations && sortedRelations.length && sortedRelations[0].domain.field) {
                //sort by id, as it is built with index/field
                sortedRelations = _.sortBy(sortedRelations, function (rel) {
                  return [!rel.autoRelation, rel.domain.id, rel.domain.field, rel.range.id, rel.range.field];
                });
              }
              $scope.relations = sortedRelations;
              $scope.pristineActualRelationsStr = toString(
                actualRelations(), relationFormData);

              ontologyModel.getUniqueRelationLabels()
                .then(uniqueRelationLabels => {
                  $scope.relationLabels = uniqueRelationLabels;
                });

              ontologyModel.getUniqueRelationLabelPairs()
                .then(uniqueRelationLabelPairs => {
                  // clean the map
                  relationLabelPairMap = {};

                  // construnc the map again
                  _.each(uniqueRelationLabelPairs, function (pair) {
                    relationLabelPairMap[pair.directLabel] = pair.inverseLabel;
                  });
                });

              jdbcDatasources.listVirtualIndices()
                .then(virtualIndicesList => {
                  _.each(virtualIndicesList, virtualIndex => {
                    virtualIndexMap.set(virtualIndex._id, virtualIndex._source.datasource);
                  });
                });
            }
          };

          $scope.aggregatableFilter = function (item) {
            return !!(item && !item.aggregatable);
          };

          const createFieldTypeMap = function () {
            selectHelper.getEntityFields($scope.entity.id)
              .then(fields => {
                _.each(fields, function (field) {
                  entityFieldTypeMap.set(field.value, field.esType);
                });
              });
          };

          const createRelationRangeFieldTypeMap = function (rel) {
            selectHelper.getEntityFields(rel.range.id)
              .then(fields => {
                const rangeFieldsTypeMap = new Map();
                _.each(fields, function (field) {
                  rangeFieldsTypeMap.set(field.value, field.esType);
                });
                relationRangeFieldsTypeMap.set(rel.id, rangeFieldsTypeMap);
              });
          };

          const refreshEntitiesMap = function (entities) {
            const tmpMap = {};

            if (entities !== undefined) {
              _.each(entities, (entity) => {
                tmpMap[entity.id] = entity.type;
              });
            }
            $scope.typeMap = tmpMap;
          };

          $scope.$watch('entityRelations', function (entityRelations) {
            refreshRelations(entityRelations);
            createFieldTypeMap();
          });

          // Note:
          // We need this new random id for each new added relation
          // so ng-repeat "track by relation.id" is not complaining about duplicates
          // when we keep adding new relations
          $scope.newRelationId = uuid.v1();
          $scope.$watch('relations.length', function () {
            $scope.newRelationId = 'relation:' + uuid.v1();
          });

          $scope.$watch('entities', function (entities) {
            refreshEntitiesMap(entities);
          });


          $scope.isSaveDisabled = function () {
            if ($scope.entityForm) {
              return $scope.entityForm.$pristine || $scope.entityForm.$invalid;
            }
          };

          // declare the deregister function
          let deregisterLocationChangeStartListener;

          // helper function ro re-register the listener
          function registerListener() {
            deregisterLocationChangeStartListener = $rootScope.$on('$locationChangeStart', confirmIfFormDirty);
          };

          // If the user attempts to close the browser or navigate to e.g. Timelion/Access Control/Sentinl
          window.onbeforeunload = function (e) {
            if ($scope.entityForm && $scope.actualRelationsDirty) {
              // This text needs to be set and returned from the function
              // but is never displayed for security reasons.
              // e.g. https://www.chromestatus.com/feature/5349061406228480
              const dialogText = "Not going to be rendered.";
              e.returnValue = dialogText;
              return dialogText;
            }
          };

          // Dedicated function for url changing to make it stubbable
          $scope.changeUrl = function (url) {
            window.location.href = url;
          };

          let handled = false;

          function confirmIfFormDirty(event, next, current) {
            if (!handled) {
              handled = true;
              // Check if staying on the same page but different entity within the page
              const regEx = /\/management\/siren\/datamodel\/(?:SAVED_SEARCH|VIRTUAL_ENTITY)\/([^\?\&\/]*)/;
              const tabCheck = current.match(regEx);
              const targetCheck = next.match(regEx);
              let tabName = '';
              if (tabCheck && tabCheck.length > 1) {
                tabName = tabCheck[1];
              } else {
                handled = false;
                return;
              }
              // If attempting to change entity
              if (targetCheck === null || targetCheck.length < 2 || targetCheck[1] !== tabName) {
                // If the user has changed some actual relation (autorelations don't count)
                if ($scope.entityForm && $scope.actualRelationsDirty) {
                  // prevent a digest cycle error by pushing the routeChangeHandling
                  // and the deregister of the $locationChangeStart listener to the next digest cycle
                  $timeout(() => {
                    deregisterLocationChangeStartListener();
                    const handleRouteChange = function () {
                      handled = false;
                      // Allow the navigation to take place
                      $scope.changeUrl(next);
                    };

                    confirmModalPromise(
                      'You have unsaved changes in the relational configuration. Are you sure you want to leave and lose the changes?',
                      {
                        confirmButtonText: 'confirm',
                        cancelButtonText: 'cancel'
                      }
                    )
                      .then(handleRouteChange)
                      .catch(error => {
                        if (error !== undefined) {
                          throw error;
                        }
                        // Navigation has been cancelled by the user in the modal
                        // If there is no cachedSelectedMenuItem, cache it.
                        if (!cachedSelectedMenuItem) {
                          cachedSelectedMenuItem = Object.assign({}, $scope.selectedMenuItem, { id: tabName });
                        } else {
                          // If the user selected a new EID on the eid tree on the left but cancelled the navigation
                          // the selectedMenuItem should be set back to the cached menu item
                          if ($scope.selectedMenuItem && cachedSelectedMenuItem.id !== $scope.selectedMenuItem.id) {
                            $scope.updateSelectedMenuItem(cachedSelectedMenuItem);
                          }
                        }
                        $timeout(() => {
                          handled = false;
                          // The user cancelled the navigation, so stay on the same page explicitly
                          $scope.changeUrl(current);
                          // We have deregistered the $locationChangeStart listener by now, so register a new one
                          registerListener();
                        }, 0);
                      });
                  }, 0);
                  // This is actually the initial prevention of navigation to allow the comfirm modal appear.
                  // event.preventDefault *should* work with a change to e.g. Dashboard/Discover but doesn't
                  // so we need to set the window.location.href as well
                  event.preventDefault();
                  $scope.changeUrl(current);
                } else {
                  handled = false;
                }
              } else {
                handled = false;
              }
            } else {
              // If there is already a function handling the routeChange, just stay on the same page and wait
              event.preventDefault();
              $scope.changeUrl(current);
            }
          }

          registerListener();

          const cleanRelationErrors = function (relations) {
            _.each(relations, rel => {
              delete rel.$$customError;
            });
            $scope.entityForm.$invalid = false;
          };

          $scope.setSelectedRangeEntity = function (relation, selectedRangeEntity) {
            relation.range = selectedRangeEntity;
          };

          $scope.$watchCollection(
            () => _($scope.relations)
              .partition('autoRelation')
              .map(relations => toString(relations, relationFormData))
              .value(),
            (newRelationsStrings, oldRelationsStrings) => {
              refreshEntityFormStatus();

              // Handle changes to relational endpoints
              const relationsEndpointsStr = toString(
                $scope.relations, relationEndpointsData);

              const relationsEndpointsChanged =
              relationsEndpointsStr !== $scope.relationsEndpointsStr;

              if (relationsEndpointsChanged) {
                $scope.relationsEndpointsStr = relationsEndpointsStr;

                cleanRelationErrors($scope.relations);
                _.each($scope.relations, rel => {
                  if (rel.range.id) {
                    createRelationRangeFieldTypeMap(rel);
                  }
                  validateRelation($scope.relations, rel, $scope.typeMap, entityFieldTypeMap,
                    $scope.typeErrorMessage, $scope.entityForm, virtualIndexMap, relationRangeFieldsTypeMap);
                });
              }

              // Handle autorelation changes
              if (!$scope.entityForm.$invalid &&
                newRelationsStrings[0] !== oldRelationsStrings[0]) {
                dataModel.updateEntityAutoRelations($scope.entity, $scope.relations);
              }

              if (relationsEndpointsChanged) {
                $scope.$emit('siren:relationsChanged', {
                  autoRelationCount: $scope.relations.filter(rel => rel.autoRelation).length,
                  relationCount: $scope.relations.length,
                  relationError: $scope.entityForm.$invalid
                });
              }
            }
          );

          const saveRelationsListener = function () {
            const skipNotification = true;
            return $scope.saveRelations(skipNotification)
              .then(() => $scope.reloadGraph())
              .catch(notify.error);
          };

          dataModel.registerBeforeSave(saveRelationsListener);

          $scope.$on('$destroy', function () {
            deregisterLocationChangeStartListener();
            dataModel.deregisterBeforeSave(saveRelationsListener);
            notify = null; // this allow to completely garbage collect the object from memory
          });

          const RelationError = {
            incomplete: 'Some of the relations are not complete, please check them again.',
            type: $scope.typeErrorMessage
          };

          /**
          * Checks if the relation has all the required fields.
          *
          * If Entity is of type VIRTUAL_ENTITY
          * the domain is fixed and it should NOT have a field
          * the range must be of type SAVED_SEARCH with a field and id
          *
          * If Entity is of type SAVED_SEARCH
          * the domain is fixed to but it must have a field
          * the range can be of either SAVED_SEARCH or VIRTUAL_ENTITY type
          */
          function getRelationError(rel) {
            const menuItem = $scope.entity;

            if (menuItem && menuItem.type === EntityType.SAVED_SEARCH) {
              if (!rel || !rel.domain || !rel.range || !rel.range.id || !rel.domain.field ||
                 (rel.range.type === EntityType.SAVED_SEARCH && !rel.range.field)) {
                return RelationError.incomplete;
              }

              if (!rel.directLabel || !rel.inverseLabel) {
                return RelationError.incomplete;
              }

              if (
                rel.range && rel.domain &&
                rel.range.field && entityFieldTypeMap &&
                relationRangeFieldsTypeMap.has(rel.id) &&
                entityFieldTypeMap.get(rel.domain.field) !== relationRangeFieldsTypeMap.get(rel.id).get(rel.range.field)
              ) {
                return RelationError.type;
              }
            } else if (menuItem && menuItem.type === EntityType.VIRTUAL_ENTITY) {
              if (!rel || !rel.range || !rel.range.id || (rel.range.type === EntityType.SAVED_SEARCH && !rel.range.field)) {
                return RelationError.incomplete;
              }
            } else {
              return RelationError.incomplete;
            }
          }

          $scope.saveRelations = function (skipNotification) {
            const id = $scope.entity.id;
            const relationError = _.find($scope.relations, getRelationError);
            if (!relationError) {
              return ontologyModel.getRelationMap()
                .then(relationsMap => {
                  const oldIdsSet = new Set();
                  const rels = _.reduce($scope.relations, (rels, rel) => {
                    if (relationsMap[rel.id]) {
                      rels.old.push(rel);
                      oldIdsSet.add(rel.id);
                      oldIdsSet.add(rel.inverseOf);
                    } else {
                      rels.new.push(rel);
                    }
                    return rels;
                  }, { new: [], old: [] });


                  rels.deleted = _.reduce(relationsMap, function (array, rel, relId) {
                    if (rel.domain.id === id && !oldIdsSet.has(relId)) {
                      array.push(relId);
                    }
                    return array;
                  }, []);

                  return Promise.all([
                    dataModel.updateRelations(rels.old),
                    dataModel.deleteRelations(rels.deleted),
                    dataModel.createRelations(rels.new)
                  ])
                    .then(() => {
                      if (skipNotification !== true) {
                        notify.info('Relations saved.');
                      }
                      $scope.actualRelationsDirty = false;
                    })
                    .catch(notify.error)
                    .finally(() => {
                      $scope.$emit('siren:reload-entity-relations');
                    });
                });
            } else {
              if (skipNotification !== true) {
                notify.warning(relationError);
              } else {
              // pass flag to do not show message about successful updating
                $scope.entity.invalidRelations = true;
              }
              return Promise.resolve();
            }
          };

          $scope.saveAutoRelation = function (rel) {
            if (!rel.autoRelation) { return Promise.resolve(); }

            const relationError = getRelationError(rel);
            if (relationError) {
              notify.warning(relationError);
              return Promise.resolve();
            }

            return Private(FindRelationsProvider).applyAutoRelations([ rel ], {
              showProgress: false
            })
              .then(() => {
                notify.info('Relation saved.');
                // kibi: make sure we reload relations page
                $route.updateParams({ selectedTab: 'relations' });
                $route.reload();
              })
              .catch(err => err && notify.error(err));
          };

          // this method automatically assigns inverseLabel when the user sets the directLabel if it is not set already or vice versa
          $scope.setOppositeLabel = function (relation, labelType) {
            const label = labelType === 'inverse' ? 'directLabel' : 'inverseLabel';
            if ($scope.relationLabels.indexOf(relation[label]) === -1) {
              $scope.relationLabels.push(relation[label]);
            }

            if (labelType === 'inverse' && !relation.inverseLabel && relationLabelPairMap[relation.directLabel]) {
              relation.inverseLabel = relationLabelPairMap[relation.directLabel];
            } else if (labelType === 'direct' && !relation.directLabel && relationLabelPairMap[relation.inverseLabel]) {
              relation.directLabel = relationLabelPairMap[relation.inverseLabel];
            }
          };

          // this method automatically refresh suggestion list during user input
          $scope.refreshSuggestions = function ($select) {
            const search = $select.search;
            let list = angular.copy($select.items);

            //remove last user input
            list = list.filter(function (item) {
              if ($scope.relationLabels.indexOf(item) !== -1 || search === item) {
                return true;
              } else {
                return false;
              }
            });

            if (!search) {
              //use the predefined list
              $select.items = list;
            } else {
              //manually add user input and set selection
              if ($scope.relationLabels.indexOf(search) === -1) {
                list = [search].concat(list);
              };
              $select.items = list;
              $select.selected = search;
            }
          };

          // advanced options
          $scope.edit = function (relId) {
            kbnUrl.change('/management/siren/relations/{{ entity }}/{{ id }}', {
              entity: encodeURIComponent($scope.entity.id),
              id: encodeURIComponent(relId)
            });
          };

          $scope.getAdvancedOptionsInfo = function (relation) {
            let info = 'Join Type: ';
            switch (relation.joinType) {
              case 'MERGE_JOIN':
                info += 'Distributed join using merge join algorithm';
                break;
              case 'HASH_JOIN':
                info += 'Distributed join using hash join algorithm';
                break;
              case 'BROADCAST_JOIN':
                info += 'Broadcast join';
                break;
              default:
                info += 'not set';
            }
            info += '\n';

            info += 'Task timeout: ';
            if (relation.timeout === 0) {
              info += 'not set';
            } else {
              info += relation.timeout;
            }
            return info;
          };

          $scope.explore = function (relation) {
            Private(RelationExplorerProvider)
              .spawnModal(relation.domain, relation.range);
          };
        }
      };
    })
  .directive('kibiRelationsSearchBar', () => {
    return {
      restrict: 'A',
      scope: true,
      link: function (scope, element, attrs) {

        scope.searchRelations = function () {
          const relations = _.get(scope, attrs.kibiRelationsSearchBarPath);
          const searchString = scope[attrs.ngModel];

          if (!searchString || searchString.length < 2) {
            relations.forEach((relation) => relation.$$hidden = false);
            return;
          }

          const search = function (obj, searchString) {
            let result;
            for (const key in obj) {
              if (obj.hasOwnProperty(key)) {
                if (typeof obj[key] === 'object' && obj[key] !== null || _.isArray(obj[key]) && obj[key].length) {
                  result = search(obj[key], searchString);
                  if (result) {
                    return result;
                  }
                }
                if (typeof obj[key] === 'string') {
                  const found = obj[key].match(new RegExp(searchString, 'gi'));
                  if (found && found.length) {
                    return true;
                  }
                }
              }
            }
            return result;
          };

          relations.forEach((relation) => {
            if (search(relation, searchString)) {
              relation.$$hidden = false;
            } else {
              relation.$$hidden = true;
            }
          });
        };
      }
    };
  });
