import { QueryBuilderFactory } from 'ui/kibi/helpers/query_builder';
import { DashboardHelperFactory } from 'ui/kibi/helpers/dashboard_helper';
import isJoinPruned from 'ui/kibi/helpers/is_join_pruned';
import _ from 'lodash';
import { SearchHelper } from 'ui/kibi/helpers/search_helper';
import { uiModules } from 'ui/modules';
import uiRoutes from 'ui/routes';
import { SimpleEmitter } from 'ui/utils/simple_emitter';
import { CacheProvider } from 'ui/kibi/helpers/cache_helper';
import { SpinnerStatus } from 'ui/kibi/spinners/spinner_status';

import { findMainCoatNodeSearchId } from 'ui/kibi/components/dashboards360/coat_tree';
import { filterHelper } from 'ui/kibi/components/dashboards360/filter_helper';

// kibi: these imports are needed for any third party plugin
import 'ui/filter_bar/join_explanation';
import 'ui/kibi/meta/kibi_meta';
// kibi: end

uiRoutes
  .addSetupWork(function loadDashboardGroups($injector, $location, createNotifier) {
  // kibi: init the service only when on dashboard page
  // we cannot use onDashboardPage() here because link is not active yet
    if ($location.absUrl().includes('app/kibana#/dashboard/')) {
      if ($injector.has('kibiState')) {
        const dashboardGroups = $injector.get('dashboardGroups');
        dashboardGroups.init().catch(err => {
          createNotifier({ location: 'Dashboard Groups - setup' }).warning(err);
        });
      }
    }
  });

uiModules
  .get('kibana')
  .service('dashboardGroups', function (createNotifier, $timeout, kibiState, Private, savedDashboards, timefilter, $rootScope,
    savedDashboardGroups, Promise, kbnIndex, joinExplanation, getAppState, kibiMeta, sessionId, coatDashboardMap,
    useVisualizationForCountOnDashboard) {
    const notify = createNotifier({ location: 'Dashboard groups' });
    const dashboardHelper = Private(DashboardHelperFactory);
    const queryBuilder = Private(QueryBuilderFactory);
    const searchHelper = new SearchHelper(kbnIndex, sessionId);
    let lastSelectDashboardEventTimer;
    const cache = Private(CacheProvider);

    const _getDashboardForGroup = function (groupTitle, dashboardHit) {
      return {
        id: dashboardHit.id,
        title: this._shortenDashboardName(groupTitle, dashboardHit.title),
        savedSearchId: findMainCoatNodeSearchId(dashboardHit.coatJSON),
        count: SpinnerStatus.UNDEFINED
      };
    };

    const _clearAllMeta = function (dashboard) {
      dashboard.count = SpinnerStatus.DONE;
      delete dashboard.isPruned;
      delete dashboard.filterIconMessage;
      delete dashboard.dirty;
    };

    class DashboardGroups extends SimpleEmitter {
      constructor() {
        super();
        this.groups = [];
        this._initialized = undefined;
      }

      init(timeout = 15 * 1000) { // default timeout set to 15sec after that user will see the warning
        if (this._initialized !== undefined) {
          return Promise.resolve();
        }
        this._initialized = false; // set to false to indicate it is in-progress
        return this.computeGroups('init')
          .then(groups => {
            this.groups = groups;

            // NOTE: It is important to wait until appState is fully ready before doing the init
            // this prevents situations where appState was not ready yet
            // causing dashboard meta to be computed incorectly due to missing filters or queries

            const appStateReady = new Promise((fulfill, reject) => {
              const STEP = 10;
              let timePassed = 0;
              let timer;
              const check = function () {
                const appState = getAppState();
                if (timer) {
                  $timeout.cancel(timer);
                }
                if (appState) {
                  // leaving this console log print to have an idea how long this takes on different systems
                  if (console.debug) {
                    console.debug('Got appState during dashboard_group service initailization after ' + timePassed + 'ms');
                  }
                  return fulfill();
                }
                timePassed += STEP;
                if (timePassed >= timeout) {
                  return reject(
                    new Error(
                      `Could not get appState after ${timeout} ms, during dashboard_group service initialization.` +
                  'Dashboard counts will not be shown.'
                    )
                  );
                }
                timer = $timeout(check, STEP);
              };
              check();
            });

            return appStateReady
              .then(() => {
                const dashboardIds = _(groups)
                  .filter(g => !g.collapsed || g.virtual)
                  .map('dashboards')
                  .flatten()
                  .map('id')
                  .value();

                return coatDashboardMap.updateCoatDashboardMap().then(() => {
                  return this.updateMetadataOfDashboardIds(dashboardIds)
                    .then(res => {
                      this._initialized = true;
                      return res;
                    });
                });
              });
          })
          .catch(err => {
            this._initialized = undefined;
            return Promise.reject(err);
          });
      }

      get isInitialized() {
        return this._initialized;
      }

      _dashboardMetadataCallback(
        dashboard, meta,
        filters, queries, time,
        dirty, dirtyDiff
      ) {
        if (!_.contains(Object.keys(meta), 'error')) {
          delete dashboard.error;
          dashboard.count = meta.hits.total;
        } else if (_.contains(Object.keys(meta), 'error') && meta.error.reason) {
          dashboard.error = meta.error.reason;
          return Promise.resolve();
        } else if (_.contains(Object.keys(meta), 'error') && meta.error.message) {
          dashboard.error = meta.error.message;
          return Promise.resolve();
        } else if (_.contains(Object.keys(meta), 'error') && meta.error.root_cause) {
        // we hit case where reason was null due to null pointer exception
          dashboard.error = JSON.stringify(meta.error.root_cause);
          return Promise.resolve();
        }
        dashboard.isPruned = isJoinPruned(meta);
        dashboard.dirty = dirty;

        return joinExplanation.constructFilterIconMessage(filters, queries, time, dirtyDiff)
          .then(filterIconMessage => {
            dashboard.filterIconMessage = filterIconMessage;
          })
          .catch(error => {
            dashboard.filterIconMessage = '';
            dashboard.error = JSON.stringify(error.message);
          });
      }

      _shortenDashboardName(groupTitle, dashboardTitle) {
        const g = groupTitle.toLowerCase();
        const d = dashboardTitle.toLowerCase();

        if (d.indexOf(g) === 0 && d.length > g.length) {
          return dashboardTitle.substring(groupTitle.length).replace(/^[\s-]{1,}/, '');  // replace any leading spaces or dashes
        }
        return dashboardTitle;
      }

      getGroups() {
        return this.groups;
      }

      setDashboardSelection(group, dashboard, state) {
        this.emit('dashboardSelected', group, dashboard, state);
      }

      setGroupSelection(group) {
        this.emit('groupSelected', group);
      }

      setGroupHighlight(dashboardId) {
      // here iterate over dashboardGroups remove all highlighted groups
      // then set the new highlight group
        _.each(this.getGroups(), function (group) {
          _.each(group.dashboards, function (dashboard) {
            if (dashboard.id === dashboardId) {
              dashboard.$$highlight = true;
            } else {
              dashboard.$$highlight = false;
            }
          });
        });
      }

      resetGroupHighlight() {
      // here iterate over dashboardGroups remove all highlighted groups
        _.each(this.getGroups(), function (group) {
          _.each(group.dashboards, function (dashboard) {
            dashboard.$$highlight = false;
          });
        });
      }

      renumberGroups() {
        let priority = 10;
        const saveActions = [];
        const groups = _.clone(_.sortBy(this.getGroups(), 'priority'));
        groups.forEach((group) => {
          group.priority = priority;
          priority += 10;
          if (group.virtual) {
            savedDashboards.get(group.id).then(savedDashboard => {
              savedDashboard.priority = group.priority;
              return savedDashboard.save();
            });
          } else {
            saveActions.push(savedDashboardGroups.get(group.id).then(savedGroup => {
              savedGroup.priority = group.priority;
              return savedGroup.save();
            }));
          }
        });
        saveActions.push(cache.invalidate);
        return Promise.all(saveActions);
      }

      newGroup(title = 'New group', iconCss = 'fa fa-folder') {
        return savedDashboardGroups.get().then(group => {
          let priority = 0;
          this.getGroups().forEach((group) => {
            priority = priority < group.priority ? group.priority : priority;
          });
          group.priority = priority + 10;
          group.title = title;
          group.iconCss = iconCss;
          return group.save();
        });
      }

      getDashboardsInGroup(groupId) {
        const group = _.find(this.getGroups(), 'id', groupId);
        return group.dashboards;
      }

      _getDashboardsMetadata(ids, forceCountsUpdate = false, suppressWarnings = false) {
        return savedDashboards.find()
          .then((resp) => {

            const dashboardIds = _(resp.hits)
              .filter(dashboardHit => findMainCoatNodeSearchId(dashboardHit.coatJSON) && _.contains(ids, dashboardHit.id))
              .map('id')
              .value();

            return kibiState.getStates(dashboardIds, suppressWarnings).then((results) => {
              // remap to an array
              const metadataPromises = [];
              for (const dashboardId in results) {
                if (results.hasOwnProperty(dashboardId)) {
                  let p;
                  const error = results[dashboardId].error;
                  if (error) {
                    p = Promise.resolve({
                      dashboardId: dashboardId,
                      filters: [],
                      queries: [],
                      time: null,
                      indices: [],
                      indexPattern: null,
                      error: error
                    });
                  } else {
                    const index = results[dashboardId].index;
                    const filters = results[dashboardId].filters;
                    const queries = results[dashboardId].queries;
                    const time = results[dashboardId].time;
                    const query = queryBuilder(filters, queries, time);
                    query.size = 0; // we do not need hits just a count

                    const dashboard = _(resp.hits)
                      .filter(dashboardHit => findMainCoatNodeSearchId(dashboardHit.coatJSON) && dashboardId === dashboardHit.id)
                      .value() [0];

                    p = kibiState.compareStateToSavedDashState(dashboard)
                      .then(resp => {
                        const dirty = !resp.stateEqual;
                        const dirtyDiff = resp.diff;

                        return kibiState.timeBasedIndices(index, dashboardId)
                          .then(indices => {
                            const optimizedQuery = searchHelper.optimize(indices, query, index);
                            return {
                              dashboardId,
                              query: optimizedQuery,
                              indexPattern: index,
                              filters,
                              queries,
                              time,
                              indices,
                              dirty,
                              dirtyDiff
                            };
                          })
                          .catch((error) => {
                            // If computing the indices failed because of an authorization error
                            // set indices to an empty array and mark the dashboard as forbidden.
                            if (error.status === 403 || error.statusCode === 403) {
                              const optimizedQuery = searchHelper.optimize([], query, index);
                              return {
                                dashboardId,
                                query: optimizedQuery,
                                indexPattern: index,
                                filters,
                                queries,
                                time,
                                indices: [],
                                forbidden: true
                              };
                            }
                            throw error;
                          });
                      });
                  }
                  metadataPromises.push(p);
                }
              }

              return Promise.all(metadataPromises);
            })
              .catch(notify.error);
          });
      }

      updateMetadataOfDashboardIds(ids, forceCountsUpdate = false, suppressWarnings = false) {
        const self = this;
        if (console.debug) {
          const msg = `DashboardGroups: requesting metadata update for following dashboards: ${JSON.stringify(ids)}`;
          console.debug(msg); // eslint-disable-line no-console
        }

        return this._getDashboardsMetadata(ids, forceCountsUpdate, suppressWarnings)
          .then(metadata => {

            const mapOfDashboardsRequestedButNoMetaFound = {};
            _.each(this.getGroups(), g => {
              _.each(g.dashboards, d => {
                if (ids.indexOf(d.id) !== -1 && !mapOfDashboardsRequestedButNoMetaFound[d.id]) {
                  mapOfDashboardsRequestedButNoMetaFound[d.id] = d;
                }
              });
            });

            const metaDefinitions = [];
            _.each(this.getGroups(), g => {
              _.each(g.dashboards, d => {
                const foundDashboardMetadata = _.find(metadata, 'dashboardId', d.id);
                if (foundDashboardMetadata) {

                  // remove every dashboard we found meta for
                  if (mapOfDashboardsRequestedButNoMetaFound[d.id]) {
                    delete mapOfDashboardsRequestedButNoMetaFound[d.id];
                  }

                  _clearAllMeta(d);

                  // if there was an error in metadata do not even try to fetch the counts
                  if (foundDashboardMetadata.error) {
                    d.count = SpinnerStatus.DONE;
                    self._dashboardMetadataCallback(d, { error: foundDashboardMetadata.error }).then(() => {
                      self.emit('dashboardMetadataUpdated', {
                        id: foundDashboardMetadata.dashboardId,
                        error: foundDashboardMetadata.error
                      });
                    });
                    return;
                  }

                  if (useVisualizationForCountOnDashboard.get(foundDashboardMetadata.dashboardId)) {
                    d.count = SpinnerStatus.FETCHING;
                  } else {
                    metaDefinitions.push({
                      definition: {
                        id: foundDashboardMetadata.dashboardId,
                        query: foundDashboardMetadata.query
                      },
                      callbackStart: function () {
                        if (console.debug) {
                          const msg = `kibiMeta service start to fetch metadata for dashboard: ${foundDashboardMetadata.dashboardId} ` +
                                  `using query: ${foundDashboardMetadata.query}`;
                          console.debug(msg);
                        }
                        d.count = SpinnerStatus.FETCHING;
                        self.emit('dashboardMetadataUpdated', { id: foundDashboardMetadata.dashboardId });
                      },
                      callback: function (error, meta) {
                        d.count = SpinnerStatus.DONE;
                        if (error) {
                          self._dashboardMetadataCallback(d, { error: error }).then(() => {
                            self.emit('dashboardMetadataUpdated', { id: foundDashboardMetadata.dashboardId, error: error });
                          });
                          return;
                        }
                        self._dashboardMetadataCallback(
                          d,
                          meta,
                          foundDashboardMetadata.filters,
                          foundDashboardMetadata.queries,
                          foundDashboardMetadata.time,
                          foundDashboardMetadata.dirty,
                          foundDashboardMetadata.dirtyDiff
                        ).then(() => {
                          self.emit('dashboardMetadataUpdated', { id: foundDashboardMetadata.dashboardId, meta: meta });
                        });
                      }
                    });
                  }
                }
              });
            });

            // clear meta for all dashboards requested but for which we did not found meta
            _.each(mapOfDashboardsRequestedButNoMetaFound, (dashboard, id) => {
              _clearAllMeta(dashboard);
            });

            kibiMeta.getMetaForDashboards(metaDefinitions);
          });
      }

      getGroup(dashboardId) {
        if (!dashboardId) {
          throw new Error('Missing dashboard Id');
        }
        return _.find(this.getGroups(), group => _.find(group.dashboards, 'id', dashboardId));
      }

      getIdsOfDashboardGroupsTheseDashboardsBelongTo(dashboardIds) {
        return _(this.getGroups())
        // do not consider groups that were created programmatically
          .reject('virtual')
        // keep only the groups which dashboards contains some of the ids passed in argument
          .filter(group => _.intersection(dashboardIds, _.map(group.dashboards, 'id')).length)
        // return the id of the groups
          .map('id')
          .value();
      }

      getTitlesOfDashboardGroupsTheseDashboardsBelongTo(dashboardIds) {
        return _(this.getGroups())
        // do not consider groups that were created programmatically
          .reject('virtual')
        // keep only the groups which dashboards contains some of the ids passed in argument
          .filter(group => _.intersection(dashboardIds, _.map(group.dashboards, 'id')).length)
        // return the title of the groups
          .map('title')
          .value();
      }

      selectDashboard(dashboardId) {
        $timeout.cancel(lastSelectDashboardEventTimer);

        const currentDashboardId = kibiState.getCurrentDashboardId();
        // only update meta if we stay on the same dashboard
        // in other cases the meta will be triggered by a watcher kibiState.getCurrentDashboardId
        const updateMeta = currentDashboardId === dashboardId ? true : false;

        // we need this timeout to allow ui event handler take place.
        lastSelectDashboardEventTimer = $timeout(() => {
        // save which one was selected for:
        // - iterate over dashboard groups remove the active group
        // - set the new active group and set the selected dashboard
          _.each(this.getGroups(), group => {
            group.active = false;
          });

          const activeGroup = _.find(this.getGroups(), { dashboards: [ { id: dashboardId } ] });
          activeGroup.active = true;
          activeGroup.selected = _.find(activeGroup.dashboards, 'id', dashboardId);
          return dashboardHelper.switchDashboard(
            dashboardId,
            !activeGroup.virtual ? activeGroup.id : null
          )
            .then(() => {
              if (updateMeta) {
                // Warnings suppressed when switching dashboards as they are already shown when the dashboards are first loaded
                this.updateMetadataOfDashboardIds([ dashboardId ], true, true);
              }
            });
        }, 0);
        return lastSelectDashboardEventTimer;
      }

      _getListOfDashboardsFromGroups(dashboardGroups) {
        const dashboardsInGroups = [];
        _.each(dashboardGroups, function (group) {
          if (group.dashboards) {
            _.each(group.dashboards, function (dashboard) {
              if (!_.find(dashboardsInGroups, 'id', dashboard.id)) {
                dashboardsInGroups.push(dashboard);
              }
            });
          }
        });
        return dashboardsInGroups;
      }

      /**
     * Copies dashboards groups from src to dest.
     * Modifies the dest object.
     */
      copy(src, dest) {
        if (!dest) {
          throw new Error('Dest object should be defined');
        }

        const _saveDashboardMeta = function (dash, fromDash) {
          if (fromDash) {
            _.assign(dash, {
              count: fromDash.count,
              isPruned: fromDash.isPruned,
              dirty: fromDash.dirty,
              filterIconMessage: fromDash.filterIconMessage
            });
          }
        };

        _.each(src, srcGroup => {
          const destGroup = _.find(dest, 'id', srcGroup.id);
          if (destGroup) {
            destGroup.virtual = srcGroup.virtual;
            destGroup.active = srcGroup.active;
            destGroup.hide = srcGroup.hide;
            destGroup.iconCss = srcGroup.iconCss;
            destGroup.iconUrl = srcGroup.iconUrl;
            destGroup.priority = srcGroup.priority;
            destGroup.title = srcGroup.title;

            // when copying selected reference we keep the count, filterIconMessage and isPruned properties
            // from the previously selected dashboard and all other dashboards in the group

            _saveDashboardMeta(srcGroup.selected, destGroup.selected);
            destGroup.selected = srcGroup.selected;

            // now for all the other dashboards in the group
            destGroup.dashboards = _.map(srcGroup.dashboards, srcDashboard => {
              const destDashboard = _.find(destGroup.dashboards, 'id', srcDashboard.id);

              if (destDashboard) {
                _saveDashboardMeta(srcDashboard, destDashboard);
              }
              return srcDashboard;
            });
          } else {
          // new group
            dest.push(srcGroup);
          }
        });
        for (let destIndex = dest.length - 1; destIndex >= 0; destIndex--) {
          const srcIndex = _.findIndex(src, { id: dest[destIndex].id });
          if (srcIndex === -1) {
            dest.splice(destIndex, 1);
          }
        }
      }

      getVisibleDashboardIds(dashboardIds) {
      // filter the given dashboardIds
      // to use only the visible dashboard
        return _(this.getGroups())
          .filter(g => !g.collapsed || g.virtual)
          .map('dashboards')
          .flatten()
          .filter(d => _.contains(dashboardIds, d.id))
          .map('id')
          .value();
      }

      setActiveGroupFromUrl() {
        const currentDashboardId = kibiState.getCurrentDashboardId();

        _.each(this.getGroups(), group => {
          group.active = false;
        });

        if (currentDashboardId) {
          const currentGroup = _.find(this.getGroups(), { dashboards: [ { id: currentDashboardId } ] });
          if (currentGroup) {
            currentGroup.active = true;
          }
        }
      }

      _computeGroupsFromSavedDashboardGroups(savedDashboardHits) {
        const self = this;

        // get all dashboard groups
        return savedDashboardGroups.find().then(function (respGroups) {
          if (!respGroups.hits) {
            return [];
          }

          const dashboardGroups1 = [];
          // first iterate over existing groups
          _.each(respGroups.hits, function (group) {

          // selected dashboard
            let selected;

            // set count to undefined
            _.each(group.dashboards, d => d.count = SpinnerStatus.UNDEFINED);

            // ignore empty or non existing dashboard objects inside a dashboard group
            const dashboards = _.reduce(group.dashboards, (filtered, dashboard) => {
              if (dashboard && dashboard.id) {
                const savedDashboardHit = _.find(savedDashboardHits, 'id', dashboard.id);
                if (savedDashboardHit) {
                  filtered.push(_getDashboardForGroup.call(self, group.title, savedDashboardHit));
                }
              }
              return filtered;
            }, []);

            // try to get the last selected one for this group
            if (dashboards.length > 0) {
              const lastSelectedId = kibiState.getSelectedDashboardId(group.id);
              selected = _.find(dashboards, 'id', lastSelectedId);
            }

            // nothing worked select the first one
            if (!selected && dashboards.length > 0) {
              selected = dashboards[0];
            }

            dashboardGroups1.push({
              id: group.id,
              title: group.title,
              hide: group.hide,
              iconCss: group.iconCss,
              iconUrl: group.iconUrl,
              priority: group.priority,
              dashboards: dashboards,
              selected: selected
            });

          }); // end of each

          return dashboardGroups1;
        });
      }

      _addAdditionalGroupsFromSavedDashboards(savedDashboardHits, dashboardGroups1) {
      // first create array of dashboards already used in dashboardGroups1
        const dashboardsInGroups = this._getListOfDashboardsFromGroups(dashboardGroups1);
        const highestGroup = _.max(dashboardGroups1, 'priority');
        let highestPriority = highestGroup && highestGroup.priority || 0;

        _.each(savedDashboardHits, dashboardHit => {
          let isInGroups = false;
          _.each(dashboardsInGroups, dashboard => {
            if (dashboard.id === dashboardHit.id) {
              // here add savedSearchId property to all already existing dashboard objects
              dashboard.savedSearchId = findMainCoatNodeSearchId(dashboardHit.coatJSON);
              isInGroups = true;
              return false;
            }
          });

          // so now we know that this dashboard is not in any group
          if (isInGroups === false) {
            // not in a group so add it as new group with single dashboard
            const groupId = dashboardHit.id;
            const groupTitle = dashboardHit.title;
            const onlyOneDashboard = _getDashboardForGroup.call(this, groupTitle, dashboardHit);

            dashboardGroups1.push({
              virtual: true,
              id: groupId,
              title: groupTitle,
              dashboards: [ onlyOneDashboard ],
              selected: onlyOneDashboard,
              priority: dashboardHit.priority ? dashboardHit.priority : ++highestPriority
            });
          }
        });

        // only here we can fulfill the promise
        return dashboardGroups1;
      }

      updateDashboardMetaData(dashId, data) {
        const group = this.getGroup(dashId);
        if (group) {
          return kibiState.getStates([dashId]).then((results) => {
            const filters = results[dashId].filters;
            const queries = results[dashId].queries;
            const time = results[dashId].time;
            const query = queryBuilder(filters, queries, time);

            const selectedDash = _.find(group.dashboards, d => d.id === dashId);
            return this._dashboardMetadataCallback(selectedDash, data, filters, query, time, !data.stateEqual, data.diff)
              .then(() => this.emit('dashboardMetadataUpdated', { id: dashId, meta: data }));
          });
        }
      }

      /*
     * Computes the dashboard groups array
     *
     *  [
          {
            title:
            priority:
            dashboards: [
              {
                id:
                title:
                onSelect:
                onOpenClose:
              },
              ...
            ]
            iconCss:
            iconUrl:
            selected: dashboard
          },
          ...
        ]
     *
     * groups in this array are used to render tabs
     *
     */
      computeGroups(reason) {
        if (console.debug) {
          console.debug('Dashboard Groups will be recomputed because: [' + reason + ']'); // eslint-disable-line no-console
        }
        // fetch all dashboards and pass it down
        return savedDashboards.find()
          .then(savedDashboards => {
            return this._computeGroupsFromSavedDashboardGroups(savedDashboards.hits)
              .then(dashboardGroups1 => this._addAdditionalGroupsFromSavedDashboards(savedDashboards.hits, dashboardGroups1))
              .then(groups => {
                this.groups = groups;
                this.emit('groupsChanged');
                return groups;
              });
          });
      }
    }

    return new DashboardGroups();
  });
