'use strict';

var _lodash = require('lodash');

var _boom = require('boom');

var _boom2 = _interopRequireDefault(_boom);

var _errors = require('request-promise/errors');

var _errors2 = _interopRequireDefault(_errors);

var _crypto_helper = require('./lib/crypto_helper');

var _crypto_helper2 = _interopRequireDefault(_crypto_helper);

var _datasources_schema = require('./lib/datasources_schema');

var _datasources_schema2 = _interopRequireDefault(_datasources_schema);

var _query_engine = require('./lib/query_engine');

var _query_engine2 = _interopRequireDefault(_query_engine);

var _index_helper = require('./lib/index_helper');

var _index_helper2 = _interopRequireDefault(_index_helper);

var _translate_to_es = require('./lib/translate_to_es');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * The Query engine plugin.
 *
 * The plugin exposes the following methods to other hapi plugins:
 *
 * - getQueryEngine: returns an instance of QueryEngine.
 * - getIndexHelper: returns an instance of IndexHelper.
 * - getCryptoHelper: returns an instance of CryptoHelper.
 * - translateToES: returns a translated query.
 */

module.exports = function (kibana) {
  let queryEngine;
  let indexHelper;

  const _validateQueryDefs = function (queryDefs) {
    if (queryDefs && queryDefs instanceof Array) {
      return true;
    }
    return false;
  };

  const queryEngineHandler = function (server, method, req, reply) {
    const params = req.query;
    let options = {};
    let queryDefs = [];

    if (params.options) {
      try {
        options = JSON.parse(params.options);
      } catch (e) {
        server.log(['error', 'query_engine'], 'Could not parse options [' + params.options + '] for method [' + method + '].');
      }
    }

    if (params.queryDefs) {
      try {
        queryDefs = JSON.parse(params.queryDefs);
      } catch (e) {
        server.log(['error', 'query_engine'], 'Could not parse queryDefs [' + params.queryDefs + '] for method [' + method + '].');
      }
    }

    if (_validateQueryDefs(queryDefs) === false) {
      return reply({
        query: '',
        error: 'queryDefs should be an Array of queryDef objects'
      });
    }

    const config = server.config();
    if (config.has('xpack.security.cookieName')) {
      options.credentials = req.state[config.get('xpack.security.cookieName')];
    }
    if (req.auth && req.auth.credentials && req.auth.credentials.proxyCredentials) {
      options.credentials = req.auth.credentials.proxyCredentials;
    }
    queryEngine[method](queryDefs, options).then(function (queries) {
      // check if error is returned, reply with error property
      let queryError = '';
      const errors = (0, _lodash.each)(queries, query => {
        if (query.error) {
          queryError = query.error;
        }
      });
      return reply({
        query: params,
        snippets: queries,
        error: queryError
      });
    }).catch(function (error) {
      let err;
      if (error instanceof Error) {
        err = _boom2.default.wrap(error, 400);
      } else {
        //When put additional data in badRequest() it's not be used. So we need to add error.message manually
        const msg = 'Failed to execute query on an external datasource' + (error.message ? ': ' + error.message : '');
        err = _boom2.default.badRequest(msg);
      }
      return reply(err);
    });
  };

  return new kibana.Plugin({
    // Is it ok to remove 'migrations'? If not we'll get a circular dependency with the migrations plugin
    // require: ['kibana','investigate_core','saved_objects_api', 'migrations'],
    require: ['kibana', 'investigate_core', 'saved_objects_api'],
    id: 'query_engine',

    uiExports: {
      injectDefaultVars: function () {
        return {
          kibiDatasourcesSchema: _datasources_schema2.default.toInjectedVar()
        };
      }
    },

    init: function (server, options) {
      const config = server.config();
      const datasourceCacheSize = config.get('investigate_core.datasource_cache_size');

      this.status.yellow('Initialising the query engine');
      queryEngine = new _query_engine2.default(server);
      queryEngine._init(datasourceCacheSize).then(data => {
        server.log(['info', 'query_engine'], data);
        this.status.green('Query engine initialized');
      }).catch(err => {
        server.log(['error', 'query_engine'], err);
        this.status.red('Query engine initialization failed');
      });

      server.expose('getQueryEngine', () => queryEngine);

      server.expose('getCryptoHelper', () => _crypto_helper2.default);

      indexHelper = new _index_helper2.default(server);
      server.expose('getIndexHelper', () => indexHelper);

      server.expose('translateToES', _translate_to_es.translateToES);

      server.route({
        method: 'GET',
        path: '/clearCache',
        handler: function (req, reply) {
          queryEngineHandler(server, 'clearCache', req, reply);
        }
      });

      server.route({
        method: 'GET',
        path: '/getQueriesHtml',
        handler: function (req, reply) {
          queryEngineHandler(server, 'getQueriesHtml', req, reply);
        }
      });

      server.route({
        method: 'GET',
        path: '/getQueriesData',
        handler: function (req, reply) {
          queryEngineHandler(server, 'getQueriesData', req, reply);
        }
      });

      server.route({
        method: 'GET',
        path: '/getIdsFromQueries',
        handler: function (req, reply) {
          queryEngineHandler(server, 'getIdsFromQueries', req, reply);
        }
      });

      server.route({
        method: 'POST',
        path: '/gremlin',
        handler: function (req, reply) {
          const config = server.config();
          let credentials = null;
          if (config.has('xpack.security.cookieName')) {
            const { username, password } = req.state[config.get('xpack.security.cookieName')];
            credentials = { username, password };
          }
          if (req.auth && req.auth.credentials && req.auth.credentials.proxyCredentials) {
            credentials = req.auth.credentials.proxyCredentials;
          }

          let sirenSessionId;
          if (server.plugins && server.plugins.siren_federate && server.plugins.siren_federate.federateClient) {
            const sessionHeader = server.plugins.siren_federate.federateClient.getFederateSessionHeader(req);
            if (sessionHeader) {
              sirenSessionId = sessionHeader.value;
            }
          }

          return queryEngine.gremlin(credentials, sirenSessionId, req.payload.params.options).then(reply).catch(_errors2.default.StatusCodeError, function (err) {
            reply(_boom2.default.create(err.statusCode, err.error.message || err.message, err.error.stack));
          }).catch(_errors2.default.RequestError, function (err) {
            if (err.error.code === 'ETIMEDOUT') {
              reply(_boom2.default.create(408, err.message, ''));
            } else if (err.error.code === 'ECONNREFUSED') {
              reply({ error: `Could not send request to Gremlin server, please check if it is running. Details: ${err.message}` });
            } else {
              reply({ error: `An error occurred while sending a gremlin query: ${err.message}` });
            }
          });
        }
      });

      /*
       * Handles query to the ontology schema backend (in the gremlin server).
       */
      server.route({
        method: 'POST',
        path: '/schemaQuery',
        handler: function (req, reply) {
          const config = server.config();
          const opts = {
            method: req.payload.method ? req.payload.method : 'POST',
            data: req.payload.data,
            url: config.get('investigate_core.gremlin_server.url')
          };
          queryEngine.schema(req.payload.path, opts).then(reply).catch(_errors2.default.StatusCodeError, function (err) {
            let message = err.message;
            let stack = err.stack;
            if (err.error) {
              message = err.error.message;
              stack = err.error.stack;
            }
            reply(_boom2.default.create(err.statusCode, message, stack));
          }).catch(_errors2.default.RequestError, function (err) {
            if (err.error.code === 'ETIMEDOUT') {
              reply(_boom2.default.create(408, err.message, ''));
            } else if (err.error.code === 'ECONNREFUSED') {
              reply({ error: `Could not send request to Gremlin server, please check if it is running. Details: ${err.message}` });
            } else {
              reply({ error: `An error occurred while sending a schema query: ${err.message}` });
            }
          });
        }
      });

      /*
       * Handles query to the ontology schema backend (in the gremlin server).
       * Note: left here for backward compability so the old migrations can run
       * there is no more ontology in the system
       */
      server.route({
        method: 'POST',
        path: '/schemaUpdate',
        handler: function (req, reply) {
          const config = server.config();
          const opts = {
            method: req.payload.method ? req.payload.method : 'POST',
            data: req.payload.data,
            url: config.get('investigate_core.gremlin_server.url')
          };

          return queryEngine.schema(req.payload.path, opts).then(returnedModel => {
            const ontologyDocId = 'default-ontology';
            const ontologyDocType = 'ontology-model';
            const savedObjectsClient = req.getSavedObjectsClient();
            return savedObjectsClient.get(ontologyDocType, ontologyDocId, req).then(doc => {
              const newProperties = doc.attributes;
              newProperties.model = returnedModel;
              return savedObjectsClient.update(ontologyDocType, ontologyDocId, newProperties, {}, req).then(reply);
            }).catch(err => {
              if (err.statusCode === 404 || err.status === 404) {
                // ontology-model not found. Create a new one.
                const doc = {
                  model: returnedModel,
                  version: 1
                };
                return savedObjectsClient.create(ontologyDocType, doc, { id: ontologyDocId }, req).then(reply).catch(reply);
              } else {
                throw err;
              }
            });
          }).catch(_errors2.default.StatusCodeError, function (err) {
            let message = err.message;
            let stack = err.stack;
            if (err.error) {
              message = err.error.message;
              stack = err.error.stack;
            }
            reply(_boom2.default.create(err.statusCode, message, stack));
          }).catch(_errors2.default.RequestError, function (err) {
            if (err.error.code === 'ETIMEDOUT') {
              reply(_boom2.default.create(408, err.message, ''));
            } else if (err.error.code === 'ECONNREFUSED') {
              reply({ error: `Could not send request to Gremlin server, please check if it is running. Details: ${err.message}` });
            } else {
              reply({ error: `An error occurred while sending a schema query: ${err.message}` });
            }
          }).catch(err => {
            if (err.name === "AuthorizationError") {
              reply(_boom2.default.create(403, 'Request denied by ACL.', ''));
            } else {
              reply(err);
            }
          });
        }
      });

      /*
       * Translate a query containing kibi-specific DSL into an Elasticsearch query
       */
      server.route({
        method: 'POST',
        path: '/translateToES',
        handler: function (req, reply) {
          // kibi: if query is a JSON, parse it to string
          let query;
          if (req.payload.query) {
            if (typeof req.payload.query !== 'object') {
              return reply(_boom2.default.wrap(new Error('Expected query to be a JSON object containing single query', 400)));
            }
            query = JSON.stringify(req.payload.query);
          } else if (req.payload.bulkQuery) {
            if (!(0, _lodash.isString)(req.payload.bulkQuery)) {
              return reply(_boom2.default.wrap(new Error('Expected bulkQuery to be a String containing a bulk elasticsearch query', 400)));
            }
            query = req.payload.bulkQuery;
          }
          (0, _translate_to_es.translateToES)(server, req, query).then(translatedQuery => reply({ translatedQuery })).catch(err => {
            let errStr;
            if (typeof err === 'object' && err.stack) {
              errStr = err.toString();
            } else {
              errStr = JSON.stringify(err, null, ' ');
            }
            reply(_boom2.default.wrap(new Error(errStr, 400)));
          });
        }
      });

      function gremlinStatusCodeErrorHandler(err, reply) {
        reply(_boom2.default.create(err.statusCode, err.error.message || err.message, err.error.stack));
      }

      function gremlinRequestErrorHandler(err, message, reply) {
        if (err.error.code === 'ETIMEDOUT') {
          reply(_boom2.default.create(408, err.message, ''));
        } else if (err.error.code === 'ECONNREFUSED') {
          reply({ error: `Could not send request to Gremlin server, please check if it is running. Details: ${err.message}` });
        } else {
          reply({ error: `An error occurred while ${message}: ${err.message}` });
        }
      }

      server.route({
        method: 'POST',
        path: '/gremlin/ping',
        handler: function (req, reply) {
          queryEngine.gremlinPing(req.payload.url).then(reply).catch(_errors2.default.StatusCodeError, err => gremlinStatusCodeErrorHandler(err, reply)).catch(_errors2.default.RequestError, err => gremlinRequestErrorHandler(err, 'sending a gremlin ping', reply));
        }
      });
      server.route({
        method: 'POST',
        path: '/gremlin/clearCache',
        handler: function (req, reply) {
          queryEngine.gremlinClearCache().then(reply).catch(_errors2.default.StatusCodeError, err => gremlinStatusCodeErrorHandler(err, reply)).catch(_errors2.default.RequestError, err => gremlinRequestErrorHandler(err, 'clearing the gremlin server cache', reply));
        }
      });
      server.route({
        method: 'POST',
        path: '/gremlin/reloadOntologyModel',
        handler: function (req, reply) {
          queryEngine.gremlinReloadOntologyModel().then(reply).catch(_errors2.default.StatusCodeError, err => gremlinStatusCodeErrorHandler(err, reply)).catch(_errors2.default.RequestError, err => gremlinRequestErrorHandler(err, 'reloading the ontology model', reply));
        }
      });
    }
  });
};
