'use strict';

var _logger = require('../logger');

var _logger2 = _interopRequireDefault(_logger);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _crypto = require('crypto');

var _crypto2 = _interopRequireDefault(_crypto);

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _url = require('url');

var _url2 = _interopRequireDefault(_url);

var _http = require('http');

var _http2 = _interopRequireDefault(_http);

var _handlebars = require('handlebars');

var _handlebars2 = _interopRequireDefault(_handlebars);

var _jade = require('jade');

var _jade2 = _interopRequireDefault(_jade);

var _kibiutils = require('kibiutils');

var _kibiutils2 = _interopRequireDefault(_kibiutils);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

_handlebars2.default.registerHelper('json', function (context) {
  return JSON.stringify(context);
});

_handlebars2.default.registerHelper('getVariableValue', function (binding, name, type, options) {
  if (binding[name] && type === binding[name].type) {
    return options.fn(binding[name]);
  } else {
    return options.inverse(binding[name]);
  }
});

function Query(server, queryDefinition, cache) {
  this.server = server;
  this.serverConfig = server.config();
  this.log = (0, _logger2.default)(server, 'abstract_query');
  this.callWithInternalUser = server.plugins.elasticsearch.getCluster('admin').callWithInternalUser;

  this.id = queryDefinition.id;

  const config = {
    id: queryDefinition.id,
    label: queryDefinition.label || '',
    description: queryDefinition.description || '',
    activationQuery: queryDefinition.activationQuery || '',
    resultQuery: queryDefinition.resultQuery || '',
    datasourceId: queryDefinition.datasourceId || null,
    datasource: queryDefinition.datasource,
    rest_params: queryDefinition.rest_params || [],
    rest_headers: queryDefinition.rest_headers || [],
    rest_variables: queryDefinition.rest_variables || [],
    rest_body: queryDefinition.rest_body || '',
    rest_method: queryDefinition.rest_method || 'GET',
    rest_path: queryDefinition.rest_path || '',
    rest_resp_status_code: queryDefinition.rest_resp_status_code || 200,
    activation_rules: queryDefinition.activation_rules || [],
    tags: queryDefinition.tags || [],
    entityWeight: queryDefinition.entityWeight || 0.3,
    queryPrefixes: queryDefinition.queryPrefixes || {}
  };

  this.config = config;
  this.config.prefixesString = _lodash2.default.map(this.config.queryPrefixes, function (value, key) {
    return 'prefix ' + key + ': <' + value + '>';
  }).join('\n');

  this.cache = cache;
}

Query.prototype._getUsername = function (options) {
  if (options && options.credentials && options.credentials.username) {
    return options.credentials.username;
  }
  return null;
};

Query.prototype.generateCacheKey = function (prefix, query, onlyValues, valueVariableName, username) {
  const hash = _crypto2.default.createHash('sha256');
  _lodash2.default.each(arguments, function (arg) {
    hash.update(arg && String(arg) || '-');
  });
  return hash.digest('hex');
};

Query.prototype._checkIfSelectedDocumentRequiredAndNotPresent = function (options) {
  const isEntityDependent = _kibiutils2.default.doesQueryDependOnEntity([this.config]);

  return isEntityDependent && (!options || !options.selectedDocuments || options.selectedDocuments.length === 0 || !options.selectedDocuments[0]);
};

Query.prototype._extractIdsFromSql = function (rows, idVariableName) {
  const ids = [];

  const dot = idVariableName.indexOf('.');
  if (dot !== -1) {
    idVariableName = idVariableName.substring(dot + 1);
  }
  _lodash2.default.each(rows, function (row) {
    if (row[idVariableName]) {
      ids.push(row[idVariableName]);
    } else if (row[idVariableName.toUpperCase()]) {
      ids.push(row[idVariableName.toUpperCase()]);
    } else if (row[idVariableName.toLowerCase()]) {
      ids.push(row[idVariableName.toLowerCase()]);
    }
  });
  return _lodash2.default.uniq(ids);
};

Query.prototype._fetchTemplate = function (templateId) {
  const self = this;

  if (self.cache) {
    const v = self.cache.get(templateId);
    if (v) {
      return _bluebird2.default.resolve(v);
    }
  }

  return self.callWithInternalUser('search', {
    index: self.serverConfig.get('kibana.index'),
    type: 'template',
    q: '_id:' + templateId
  }).then(function (result) {
    const template = result.hits.hits[0];
    if (self.cache) {
      self.cache.set(templateId, template._source);
    }
    return template._source;
  });
};

Query.prototype.getHtml = function (queryDef, options) {
  const that = this;

  // first run fetch results
  return that.fetchResults(options, null, queryDef.queryVariableName).then(function (data) {
    // here take the results and compile the result template

    // here if there is a prefix replace it in values when they are uris
    // this does not go to the fetch results function because
    // the results from that function should not be modified in any way
    try {
      data = that._postprocessResults(data);
    } catch (e) {
      that.log.error(e);
    }
    // here unique id
    data.id = _kibiutils2.default.getUuid4();

    // make sure that only picked not sensitive values goes in config
    // as it will be visible on the frontend
    const safeConfig = {};
    safeConfig.id = that.id;
    safeConfig.templateVars = queryDef.templateVars;
    safeConfig.open = queryDef.open;
    // now override the original config
    data.config = safeConfig;

    const templateId = queryDef.templateId || 'kibi-json-jade';
    // here fetch template via $http and cache it
    return that._fetchTemplate(templateId).then(function (template) {

      if (template.templateSource) {
        let html = 'Could not compile the template into html';
        if (template.templateEngine === 'handlebars') {
          const hbTemplate = _handlebars2.default.compile(template.templateSource);
          html = hbTemplate(data);
        } else if (template.templateEngine === 'jade') {
          const jadeFn = _jade2.default.compile(template.templateSource, { compileDebug: true, filename: templateId });
          html = jadeFn(data);
        } else {
          html = 'Unsupported template engine. Try handlebars or jade';
        }

        return _bluebird2.default.resolve({
          queryActivated: true,
          data: data,
          html: html
        });
      } else {
        return _bluebird2.default.reject('unknown template source');
      }
    }).catch(function (err) {
      // here DO NOT reject
      // as we want to still show the json data even if
      // template compiled with errors
      that.log.error(err);
      return _bluebird2.default.resolve({
        error: err,
        data: data
      });
    });
  }).catch(err => {
    // do not reject so that data from other successful queries can be displayed
    that.log.error(err);
    return _bluebird2.default.resolve({
      error: err,
      data: {
        config: {
          id: that.id,
          label: that.config.label
        }
      }
    });
  });
};

/**
 * Return a promise which when resolved should return true or false
 */
Query.prototype.checkIfItIsRelevant = function (options) {
  throw 'Must be implemented by subclass';
};

Query.prototype._extractIds = function (data) {
  throw 'Must be implemented by subclass';
};

Query.prototype.fetchResults = function (options, onlyIds, idVariableName) {
  throw 'Must be implemented by subclass';
};

Query.prototype._postprocessResults = function (data) {
  throw 'Must be implemented by subclass';
};

module.exports = Query;
