import _ from 'lodash';

import chrome from 'ui/chrome';
import Promise from 'bluebird';
import { uiModules } from 'ui/modules';
import { IndexPatternAuthorizationError } from 'ui/errors';

import { onVisualizePage } from 'ui/kibi/utils/on_page';

import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { KibiSequentialJoinVisHelperFactory } from 'ui/kibi/helpers/kibi_sequential_join_vis_helper';
import { RelationsHelperFactory }  from 'ui/kibi/helpers/relations_helper';
import { DelayExecutionHelperFactory } from 'ui/kibi/helpers/delay_execution_helper';
import { SearchHelper } from 'ui/kibi/helpers/search_helper';
import { SirenAutoJoinHelperProvider } from './siren_auto_join_helper';
import isJoinPruned from 'ui/kibi/helpers/is_join_pruned';
import treeHelper from './tree_helper';
import {
  isMemoryAllocationError,
  simplifyMemoryAllocationError,
  isIndexNotFoundError,
  simplifyIndexNotFoundError,
  isElasticsearchSecurityError,
  simplifyElasticsearchSecurityError
} from './join_errors_helper';
import 'ui/kibi/components/ontology_model/ontology_model_helper';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { findMainCoatNodeSearchId } from 'ui/kibi/components/dashboards360/coat_tree';

function controller($scope, $rootScope, Private, kbnIndex, config, kibiState, getAppState, globalState, createNotifier,
  savedDashboards, savedSearches, dashboardGroups, kibiMeta, timefilter, ontologyModel, ontologyModelHelper,
  sessionId, $sce, $element, useVisualizationForCountOnDashboard) {
  const DelayExecutionHelper = Private(DelayExecutionHelperFactory);
  const searchHelper = new SearchHelper(kbnIndex, sessionId);
  const edit = onVisualizePage();
  const sirenAutoJoinHelper = Private(SirenAutoJoinHelperProvider);

  const notify = createNotifier({
    location: 'Siren Automatic Relational filter'
  });
  const appState = getAppState();

  const relationsHelper = Private(RelationsHelperFactory);
  const sirenSequentialJoinVisHelper = Private(KibiSequentialJoinVisHelperFactory);
  const currentDashboardId = kibiState.getCurrentDashboardId();
  $scope.currentDashboardId = currentDashboardId;
  const queryFilter = Private(FilterBarQueryFilterProvider);

  const hideCount = function (button) {
    button.targetCount = ''; // hides count but do not triggers the spinner
  };

  const showSpinner = function (button) {
    delete button.targetCount; // set to undefined to show spinner
  };

  const approxStringLength = Math.round($element.width() / parseInt($element.css('font-size')) * 3 / 2);

  $scope.cutLabel = function (stringLength, button) {
    let label;
    if (button.labelParts.indexOf('({0})') === -1) {
      const labelStringLength = button.labelParts[0].length;
      const restStringLength = button.labelParts[2].length + 8;
      if (labelStringLength + restStringLength > stringLength) {
        label = button.labelParts[0].slice(0, stringLength - labelStringLength - restStringLength)
          + '... (' + button.labelParts[1] + ' ' + button.labelParts[2] + ')';
        button.tooltip = button.labelParts[0];
      } else {
        label = button.label;
      };
    } else {
      // for subButtons length can be different (2 or 3)
      const availableLength = button.labelParts.length === 3 ?
        stringLength - button.labelParts[1].length - 9
        : stringLength - 9;
      if (availableLength < button.labelParts[0].length) {
        label = button.labelParts.length === 3 ?
          button.labelParts[0].slice(0, availableLength - button.labelParts[0].length) + '...' + ' ' + button.labelParts[1]
            + ' ' + button.labelParts[2]
          : button.labelParts[0].slice(0, availableLength - button.labelParts[0].length) + '...' + ' ' + button.labelParts[1];
        button.tooltip = button.labelParts[0];
      } else {
        label = button.label;
      };
    };
    return label;
  };

  $scope.getButtonLabelAsHtml = function (button, addApproximate) {
    const labelString = $scope.cutLabel(approxStringLength, button);
    if (button.showSpinner && (button.targetCount === undefined || button.targetCount === '')) {
      let spinner = '...';
      if (button.targetCount === undefined) {
        spinner = '<span class="siren-spinner"><div class="cube1"></div><div class="cube2"></div></span>';
      }
      return $sce.trustAsHtml(labelString.replace('{0}', spinner));
    }

    let count = '...';
    if (button.targetCount !== undefined && button.targetCount !== '') {
      if (addApproximate) {
        count = '~' + button.targetCount;
      } else {
        count = button.targetCount;
      }
    }
    return labelString.replace('{0}', count);
  };

  $scope.btnCountsEnabled = function () {
    return config.get('siren:enableAllRelBtnCounts');
  };

  const buttonMetaCallback = function (button, meta) {
    if (button.forbidden) {
      hideCount(button);
      button.warning = 'Access to an index referred by this button is forbidden.';
      return;
    }
    if (meta.error) {
      hideCount(button);
      if (isElasticsearchSecurityError(meta.error)) {
        const error = simplifyElasticsearchSecurityError(meta.error);
        button.warning = error.warning;
        notify.error(meta.error);
      } else if (isIndexNotFoundError(meta.error)) {
        const error = simplifyIndexNotFoundError(meta.error);
        button.warning = error.warning;
        notify.error(meta.error);
      } else if (isMemoryAllocationError(meta.error)) {
        const error = simplifyMemoryAllocationError(meta.error);
        button.warning = error.warning;
        notify.warning(error);
      } else {
        button.warning = `Got an unexpected error while computing this button's count.`;
        notify.error(meta.error instanceof Error ? meta.error : JSON.stringify(meta.error, null, ' '));
      }
      return;
    }
    button.targetCount = meta.hits.total;
    button.warning = ''; // set to empty string to hide any previous warning
    if (isJoinPruned(meta)) {
      button.isPruned = true;
      button.warning = 'Notice: This is a sample of the results because join operation was pruned';
    }
  };

  let countReq = null;

  const updateCounts = function (results, scope) {
    const metaDefinitions = [];
    _.each(results, result => {

      hideCount(result.button);

      const definition = result.button;
      // only make sense if query is there and not == null
      // query is null when counts disabled in advanced settings
      if (definition.query) {
        metaDefinitions.push({
          definition: definition,
          callbackStart: function () {
            showSpinner(result.button);
          },
          callback: function (error, meta) {
            if (error) {
              buttonMetaCallback(result.button, { error });
            }
            if (meta) {
              buttonMetaCallback(result.button, meta);
            }

            if (scope && scope.multiSearchData) {
              const queryParts = result.button.query.split('\n');
              const stats = {
                index: result.button.targetIndexPatternId,
                type: result.button.targetIndexPatternType,
                meta: {
                  label: result.button.label
                },
                response: meta ? meta : error,
                query: JSON.parse(queryParts[1])
              };
              if (meta && isJoinPruned(meta)) {
                stats.pruned = true;
              }
              scope.multiSearchData.add(stats);
            }
          }
        });
      }
    });
    countReq = kibiMeta.getMetaForRelationalButtons(metaDefinitions);
  };

  const _disableButtons = function (buttons) {
    _.each(buttons, button => {
      button.disabled = true;
    });
  };

  const _addButtonQuery = function (buttons, dashboardId, updateOnClick = false) {
    if ($scope.multiSearchData) {
      $scope.multiSearchData.clear();
    }

    // to avoid making too many unnecessary http calls to
    //
    // 1 get dashboards metadata
    // 2 fetch field_stats while getting timeBasedIndices
    //
    // lets first collect all source and target dashboard ids from all buttons
    // and all indexPattern + dashboardIds
    // and fetch all required things before computing button queries
    const indexPatternButtons = [];
    const dashboardIds = [dashboardId];
    const timeBasedIndicesList = [];
    _.each(buttons, button => {
      if (button.type === EntityType.SAVED_SEARCH) {
        indexPatternButtons.push(button);

        // compute the target dashboard ids
        if (button.targetDashboardId && dashboardIds.indexOf(button.targetDashboardId) === -1) {
          dashboardIds.push(button.targetDashboardId);
        }

        // compute the target indexpattern + target dashboard map
        const sourceIndicesFound = _.find(timeBasedIndicesList, item => {
          return item.indexPatternId === button.sourceIndexPatternId &&
                 item.dashboardIds &&
                 item.dashboardIds.length === 1 &&
                 item.dashboardIds[0] === dashboardId;
        });
        if (!sourceIndicesFound) {
          timeBasedIndicesList.push({
            indexPatternId: button.sourceIndexPatternId,
            dashboardIds: [ dashboardId ]
          });
        }
        const targetIndicesFound = _.find(timeBasedIndicesList, item => {
          return item.indexPatternId === button.targetIndexPatternId &&
                 item.dashboardIds &&
                 item.dashboardIds.length === 1 &&
                 item.dashboardIds[0] === button.targetDashboardId;
        });
        if (!targetIndicesFound) {
          timeBasedIndicesList.push({
            indexPatternId: button.targetIndexPatternId,
            dashboardIds: [ button.targetDashboardId ]
          });
        }
      }
    });

    const dashboardStatesPromise = kibiState.getStates(dashboardIds);
    const timeBasedIndicesListPromise = kibiState.timeBasedIndicesMap(timeBasedIndicesList);

    return Promise.all([ dashboardStatesPromise, timeBasedIndicesListPromise ])
      .then(res => {
        const dashboardStates = res[0];
        const timeBasedIndicesOutputList = res[1];

        return Promise.all(_.map(indexPatternButtons, (button) => {

          const sourceIndicesItem = _.find(timeBasedIndicesOutputList, item => {
            return item.indexPatternId === button.sourceIndexPatternId &&
          item.dashboardIds[0] === dashboardId;
          });
          const sourceIndices = sourceIndicesItem.timeBasedIndices;

          const targetIndicesItem = _.find(timeBasedIndicesOutputList, item => {
            return item.indexPatternId === button.targetIndexPatternId &&
          item.dashboardIds[0] === button.targetDashboardId;
          });
          const targetIndices = targetIndicesItem.timeBasedIndices;

          return sirenSequentialJoinVisHelper.getJoinSequenceFilter(
            dashboardStates[dashboardId],
            sourceIndices,
            targetIndices,
            button,
            $scope.searchSource._siren
          )
            .then(joinSeqFilter => {
              button.joinSeqFilter = joinSeqFilter;
              button.disabled = false;
              if ($scope.btnCountsEnabled() || updateOnClick) {
                const query = sirenSequentialJoinVisHelper.buildCountQuery(dashboardStates[button.targetDashboardId], joinSeqFilter);
                button.query = searchHelper.optimize(targetIndices, query, button.targetIndexPatternId);
              } else {
                button.query = null; //set to null to indicate that counts should not be fetched
              }
              return { button, indices: targetIndices };
            })
            .catch((error) => {
              // If computing the indices failed because of an authorization error
              // set indices to an empty array and mark the button as forbidden.
              if (error instanceof IndexPatternAuthorizationError) {
                button.forbidden = true;
                button.disabled = true;
              }
              if ($scope.btnCountsEnabled() || updateOnClick) {
                const query = sirenSequentialJoinVisHelper.buildCountQuery(dashboardStates[button.targetDashboardId]);
                button.query = searchHelper.optimize([], query, button.targetIndexPatternId);
              }
              return { button, indices: [] };
            });
        }));

      }).catch(notify.error);
  };

  const delayExecutionHelper = new DelayExecutionHelper(
    (data, alreadyCollectedData) => {
      alreadyCollectedData.dashboardId = data.dashboardId;
      alreadyCollectedData.buttons = data.buttons;
    },
    (data) => {
      _addButtonQuery(data.buttons, data.dashboardId)
        .then(results => {
          updateCounts(results, $scope);
        });
    },
    750,
    DelayExecutionHelper.DELAY_STRATEGY.RESET_COUNTER_ON_NEW_EVENT
  );

  const _createCompatibleSavedSearchesMap = function (savedSearches) {
    const compatibleSavedSearchesMap = {};
    _.each(savedSearches, savedSearch => {
      const searchSource = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
      if (!compatibleSavedSearchesMap[searchSource.index]) {
        compatibleSavedSearchesMap[searchSource.index] = [];
      }
      compatibleSavedSearchesMap[searchSource.index].push(savedSearch);
    });
    return compatibleSavedSearchesMap;
  };

  const _createCompatibleDashboardsMap = function (savedDashboardHits) {
    const compatibleDashboardsMap = {};
    _.each(savedDashboardHits, savedDashboardHit => {
      const savedSearchId = findMainCoatNodeSearchId(savedDashboardHit.coatJSON);
      if (!compatibleDashboardsMap[savedSearchId]) {
        compatibleDashboardsMap[savedSearchId] = [];
      }
      compatibleDashboardsMap[savedSearchId].push(savedDashboardHit);
    });
    return compatibleDashboardsMap;
  };

  $scope.toggleNode = function (node) {
    if (node.showChildren === false) {
      // open children of these node
      node.showChildren = !node.showChildren;
      // close other opened siblings
      node.closeOpenSiblings();
    } else {
      node.showChildren = !node.showChildren;
    }
    // here grab visible buttons and request count update

    const buttons = sirenAutoJoinHelper.getVisibleVirtualEntitySubButtons($scope.tree, $scope.vis.params.layout);
    if (!edit) {
      _addButtonQuery(buttons, currentDashboardId)
        .then(results => {
          updateCounts(results, $scope);
        });
    }
  };

  // used to update the counts counts on filters created when user clicks on a button
  $scope.currentDashboardCount = null;
  const dashboardMetadataUpdatedHandler = function (updatedDashboard) {
    if (updatedDashboard && updatedDashboard.id === currentDashboardId && updatedDashboard.meta) {
      $scope.currentDashboardCount = updatedDashboard.meta.hits.total;
    }
  };

  dashboardGroups.on('dashboardMetadataUpdated', dashboardMetadataUpdatedHandler);

  const getButtonClickHandler = function (currentDashboard) {
    return function (button) {
      if (button.disabled) {
        return;
      }
      const count = $scope.currentDashboardCount || 'n/a';
      sirenAutoJoinHelper.updateFilterLabel(currentDashboard.title, button, count);
      // Disable the update of dashboard count in case the count is taken from the visualization before we switch
      useVisualizationForCountOnDashboard.set(currentDashboardId, false);
      sirenAutoJoinHelper.switchDashboard(button);
    };
  };

  // As buttons are shown on UI side as a tree
  // we compute a tree like structure where some nodes can be of following type
  // BUTTON
  // VIRTUAL_BUTTON
  // RELATION
  // DASHBOARD
  // Because the VIRTUAL_BUTTON children can be arranged for two ways
  // relation first OR dashboards first
  // each VIRTUAL_BUTTON tree node contain two properties
  // nodes AND alt_nodes
  // such structure will help to render and control the UI
  //
  // Exemple structure:
  //
  // {
  //   type: BUTTON
  //   id: 1,
  //   label: button 1
  //   showChildren: false
  //   useAltNodes: false
  //   button: {}
  // },
  // ...
  // for 'normal' layout
  // {
  //   type: VIRTUAL_BUTTON
  //   id: 2,
  //   label: virt button 1
  //   showChildren: true
  //   useAltNodes: false   // START from here try to use this property to render the alternative nodes subtree
  //   nodes: [
  //     {
  //       type: RELATION
  //       id: 3
  //       label: rel 1
  //       showChildren: false
  //       useAltNodes: false
  //       nodes: [
  //         // here all nodes are BUTTONS
  //      ]
  //     },
  //     ...
  //   ],
  //   altNodes: [
  //     {
  //       type: DASHBOARD
  //       id: 3
  //       label: dash 1
  //       showChildren: false
  //       useAltNodes: false
  //       nodes: [
  //         // here all nodes are BUTTONS
  //      ]
  //     },
  //     ...
  //   ]
  // }
  //

  // for 'light' layout
  // {
  //   type: VIRTUAL_BUTTON
  //   id: 2,
  //   label: virt button 1
  //   showChildren: true
  //   useAltNodes: false   // <-- there will be NO altNodes
  //   nodes: [
  //     {
  //       type: BUTTON
  //       id: 3
  //       label: rel 1 dash 1 // <-- HERE we skip one level and generate the buttons directly
  //       showChildren: false
  //       useAltNodes: false
  //       button: button
  //       nodes: [] <-- HERE no nodes
  //
  //     },
  //     ...
  //   ],
  //   altNodes: [] // <-- NO altNodes
  // }


  const constructTree = $scope.constructTree = function (indexPatternId) {
    return Promise.all([
      ontologyModel.getRelationList()
        .then(relations => {
          return ontologyModelHelper.addIndexPatternsToRelations(relations);
        }),
      savedDashboards.find(),
      savedSearches.find(),
      ontologyModel.getEntityList()
    ])
      .then(res => {
        const relations = res[0];
        const savedDashboards = res[1].hits;
        const savedSearches = res[2].hits;
        const entities = res[3];

        // here if (!edit) we grab the current dashboard on view and replace it in
        // savedDashboards to get the uptodate version before user hit save
        if (!edit) {
          const currentDashboard = kibiState.getDashboardOnView();
          if (currentDashboard) {
            const index = _.findIndex(savedDashboards, 'id', currentDashboard.id);
            if (index !== -1) {
              savedDashboards[index] = currentDashboard;
            }
          }
        }

        // build maps once to avoid doing the lookups inside the loop
        const compatibleSavedSearchesMap = _createCompatibleSavedSearchesMap(savedSearches);
        const compatibleDashboardsMap = _createCompatibleDashboardsMap(savedDashboards);
        const newButtons = sirenAutoJoinHelper.constructButtonDefinitions(
          relations,
          entities,
          compatibleSavedSearchesMap,
          compatibleDashboardsMap,
          $scope.vis.params.visibility
        );

        const buttonDefs = _.filter(
          newButtons,
          btn => relationsHelper.validateRelationIdWithRelations(btn.indexRelationId, relations)
        );

        const difference = newButtons.length - buttonDefs.length;
        if (!edit && difference === 1) {
          notify.warning(difference + ' button refers to a non existing relation');
        } else if (!edit && difference > 1) {
          notify.warning(difference + ' buttons refer to a non existing relation');
        }

        if (!edit) {
          const dashboardIds = [ currentDashboardId ];
          _.each(buttonDefs, button => {
            if (!_.contains(dashboardIds, button.targetDashboardId)) {
              dashboardIds.push(button.targetDashboardId);
            }
          });

          return kibiState._getDashboardAndSavedSearchMetas(dashboardIds, false)
            .then(metas => {
              return {
                metas,
                buttonDefs
              };
            })
            .then(({ metas, buttonDefs }) => {
              let currentDashboardIndex;
              const dashboardIdIndexPair = new Map();

              for (let i = 0; i < metas.length; i++) {
                if (metas[i].savedSearchMeta) {
                  dashboardIdIndexPair.set(metas[i].savedDash.id, metas[i].savedSearchMeta.index);
                  if (metas[i].savedDash.id === currentDashboardId) {
                    currentDashboardIndex = metas[i].savedSearchMeta.index;
                  }
                }
              }

              if (!currentDashboardIndex) {
                return [];
              }

              const buttons = sirenSequentialJoinVisHelper.constructButtonsArray(
                buttonDefs,
                relations,
                currentDashboardIndex,
                currentDashboardId,
                dashboardIdIndexPair
              );

              for (let i = 0; i < buttons.length; i++) {
              // disable buttons until buttons are ready
                buttons[i].disabled = true;
                // retain the buttons order
                buttons[i].btnIndex = i;
              }
              if (!buttons.length) {
                $scope.vis.error =
                `The relational filter visualization "${$scope.vis.title}" is not configured for this dashboard. ` +
                `No button has a source index matching the current dashboard index: ${currentDashboardIndex}.`;
              }

              const tree = sirenAutoJoinHelper.createTreeRoot();
              sirenAutoJoinHelper.createFirstLevelNodes(tree, buttons, $scope.btnCountsEnabled());
              if ($scope.vis.params.layout === 'normal') {
                sirenAutoJoinHelper.addNodesToTreeNormalLayout(
                  tree, relations,
                  compatibleSavedSearchesMap, compatibleDashboardsMap,
                  $scope.btnCountsEnabled(),
                  $scope.vis.params.visibility
                );
                treeHelper.addAlternativeNodesToTree(tree, $scope.btnCountsEnabled());
              } else if ($scope.vis.params.layout === 'light') {
                sirenAutoJoinHelper.addNodesToTreeLightLayout(
                  tree, relations,
                  compatibleSavedSearchesMap, compatibleDashboardsMap,
                  $scope.btnCountsEnabled(),
                  $scope.vis.params.visibility
                );
              }


              // here we take the current dashboard on view (!edit so we can assume the dash will be there)
              // we do this to be able to get unsaved changes to coatJSON and generate correct buttons
              const currentDashboard = kibiState.getDashboardOnView();
              if (currentDashboard) {
                sirenAutoJoinHelper.addButtonClickHandlerToTree(tree, getButtonClickHandler(currentDashboard));
              }
              $scope.tree = tree;

              function compareButtons(a,b) {
                if (a.type < b.type) {
                  return -1;
                }
                if (a.type > b.type) {
                  return 1;
                }
                if (a.label < b.label) {
                  return -1;
                }
                if (a.label > b.label) {
                  return 1;
                }
                return 0;
              }
              const sortNodes = function (nodes) {
                nodes.sort(compareButtons);
                _.each(nodes, function (node) {
                  if (node.nodes && node.nodes.length > 1) {
                    sortNodes(node.nodes);
                  }
                });
              };
              sortNodes($scope.tree.nodes);
              return tree;
            })
            .catch(notify.error);
        } else {
          const unFilteredButtons = sirenSequentialJoinVisHelper.constructButtonsArray(buttonDefs, relations);
          const filteredButtons = _.filter(unFilteredButtons, button => button.domainIndexPattern === indexPatternId);

          const tree = sirenAutoJoinHelper.createTreeRoot();
          sirenAutoJoinHelper.createFirstLevelNodes(tree, filteredButtons);
          $scope.tree = tree;
          return tree;
        }
      });
  };

  /*
   * Update counts in reaction to events.
   * Filter buttons by indexPatternId === domainIndexPattern (used in edit mode)
   *
   * As buttons are now shown as a tree we compute a tree of elelments
   */
  const updateButtons = function (reason, indexPatternId) {
    if (!kibiState.isSirenJoinPluginInstalled()) {
      notify.error(
        'This version of Siren Relational filter requires the Federate plugin for Elasticsearch. '
        + 'Please install it and restart Siren Investigate.'
      );
      return;
    }

    let getDashboard;
    if (edit) {
      getDashboard = Promise.resolve({});
    } else {
      // do NOT use savedDashboards.get as now we can access the reference to a dahs directly
      // to get even unsaved changes when user edits dashboard data model and did not save it yet
      getDashboard = Promise.resolve(kibiState.getDashboardOnView());
    }
    return getDashboard.then(currentDashboard => {
      if (currentDashboard === undefined) {
        return Promise.resolve();
      }
      if (!edit) {
        const mainSearchId = currentDashboard.getMainSavedSearchId();

        if (!mainSearchId && !$scope.vis.error) {
          $scope.vis.error = 'This component only works on dashboards which have a saved search set.';
          return;
        }

        if (mainSearchId && $scope.vis.error) {
          $scope.vis.error = null;
        }
      }

      if (console.debug) {
        console.debug(`Updating counts on the relational buttons because: ${reason}`);
      }
      const self = this;

      let promise;
      if (reason === 'savedDashboard:setMainSearchId') {
        promise = constructTree.call(self);
      } else if (!$scope.tree || !$scope.tree.nodes || !$scope.tree.nodes.length || edit) {
        promise = constructTree.call(self, indexPatternId);
      } else {
        promise = Promise.resolve($scope.tree);
      }

      return promise
        .then(tree => {
          if (!edit) {
          // NOTE:
          // There is no need to block here
          // do NOT add return here as this will block the rendering of the tree on the widget
          // until cardinality counts are fetched
            sirenAutoJoinHelper.updateTreeCardinalityCounts(tree);
          }
          return tree;
        })
        .then(tree => {
          if (!tree) {
            const tree = sirenAutoJoinHelper.createTreeRoot();
            return Promise.resolve(tree);
          } else if (edit) {
            return Promise.resolve(tree);
          }
          const buttonsToUpdate = sirenAutoJoinHelper.getAllVisibleButtons(tree, $scope.vis.params.layout);
          // NOTE:
          // important to disable buttons while adding them to delayExecutionHelper
          // to prevent users clicking them when they still contain the wrong "previous" query
          _disableButtons(buttonsToUpdate);
          delayExecutionHelper.addEventData({
            buttons: buttonsToUpdate,
            dashboardId: currentDashboardId
          });
          return tree;
        })
        .then(tree => $scope.tree = tree);
    })
      .catch(notify.error);
  };

  $scope.getCurrentDashboardBtnCounts = function () {
    const virtualEntityButtons = sirenAutoJoinHelper.getVisibleVirtualEntitySubButtons($scope.tree, $scope.vis.params.layout);
    const allButtons = $scope.buttons.concat(virtualEntityButtons);
    _addButtonQuery(allButtons, currentDashboardId, true) // TODO take care about this true parameter
      .then(results => {
        updateCounts(results, $scope);
      });
  };

  const sirenDashboardChangedOff = $rootScope.$on('kibi:dashboard:changed', updateButtons.bind(this, 'kibi:dashboard:changed'));
  let editFilterButtonsOff;
  let editIndexPatternId;
  if (edit) {
    editFilterButtonsOff = $rootScope.$on('siren:auto-join-params:filter:indexpattern', (event, indexPatternId) => {
      updateButtons('siren:auto-join-params:filter:indexpattern', indexPatternId);
      editIndexPatternId = indexPatternId;
    });
  }

  // NOTE:
  // this vis should not update counts on changes in kibiState
  // Do NOT add $scope.$listen(appState, 'save_with_changes'
  // The kibiState is updated only when you navigate between dashboards,
  // when this happens a new dashboard is loaded and the counts will be updated anyway

  $scope.$listen(appState, 'save_with_changes', function (diff) {
    if (diff.indexOf('query') === -1) {
      return;
    }
    updateButtons.call(this, 'AppState changes');
  });

  $scope.$listen(queryFilter, 'update', function () {
    updateButtons.call(this, 'filters change');
  });

  $scope.$listen(globalState, 'save_with_changes', function (diff) {
    const currentDashboard = kibiState.getDashboardOnView();
    if (!currentDashboard) {
      return;
    }

    if (diff.indexOf('filters') !== -1) {
      // the pinned filters changed, update counts on all selected dashboards
      updateButtons.call(this, 'GlobalState pinned filters change');
    } else if (diff.indexOf('time') !== -1) {
      updateButtons.call(this, 'GlobalState time changed');
    } else if (diff.indexOf('refreshInterval') !== -1) {
      // force the count update to refresh all tabs count
      updateButtons.call(this, 'GlobalState refreshInterval changed');
    }
  });

  const removeSetMainSavedSearchIdHandler = $rootScope.$on('savedDashboard:setMainSearchId', (event, dashId, searchId) => {
    const currentDashboard = kibiState.getDashboardOnView();
    if (!currentDashboard) {
      return;
    }
    if (currentDashboard.id !== dashId) {
      return;
    }
    updateButtons('savedDashboard:setMainSearchId');
  });


  // when autoupdate is on we detect the refresh here
  const removeAutorefreshHandler = $rootScope.$on('courier:searchRefresh', (event) => {
    if ((timefilter.refreshInterval.display !== 'Off')
        && (timefilter.refreshInterval.pause === false)) {
      const currentDashboard = kibiState.getDashboardOnView();
      if (!currentDashboard) {
        return;
      }

      updateButtons('courier:searchRefresh');
    }
  });

  let editUpdateForVisibilityOff;
  if (edit) {
    editFilterButtonsOff = $scope.$watch('vis.params.visibility', (newVal, oldVal) => {
      if (newVal && !_.isEqual(newVal, oldVal)) {
        updateButtons('visibility', editIndexPatternId);
      }
    }, true);
  }

  $scope.$on('$destroy', function () {
    delayExecutionHelper.destroy();
    sirenDashboardChangedOff();
    removeAutorefreshHandler();
    removeSetMainSavedSearchIdHandler();
    if (editFilterButtonsOff) {
      editFilterButtonsOff();
    }
    if (editUpdateForVisibilityOff) {
      editUpdateForVisibilityOff();
    }
    if (countReq && !countReq.aborted) {
      countReq.abort();
    }
    dashboardGroups.off('dashboardMetadataUpdated', dashboardMetadataUpdatedHandler);
    kibiMeta.flushRelationalButtonsFromQueue();
  });

  $scope.hoverIn = function (button) {
    dashboardGroups.setGroupHighlight(button.targetDashboardId);
  };

  $scope.hoverOut = function () {
    dashboardGroups.resetGroupHighlight();
  };

  // init if not in edit mode
  // in edit mode an event will be filred from the configuration panel to init buttons.
  if (!edit) {
    updateButtons('init');
  }
};

uiModules
  .get('kibana/siren_auto_join_vis', ['kibana', 'ui.tree', 'ngSanitize'])
  .controller('SirenAutoJoinVisController', controller);
