import _ from 'lodash';

import template from 'plugins/siren_auto_join_vis/siren_auto_join_vis_params.html';
import { uiModules } from 'ui/modules';
import 'ui/kibi/components/ontology_model/ontology_model_helper';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { findMainCoatNodeSearchId } from 'ui/kibi/components/dashboards360/coat_tree';

uiModules
  .get('kibana/siren_auto_join_vis')
  .directive('sirenAutoJoinVisParams', function ($rootScope, createNotifier, ontologyModel, ontologyModelHelper, savedDashboards,
    savedSearches, indexPatterns, chrome) {
    return {
      restrict: 'E',
      template,
      link: function ($scope) {
        $scope.EntityType = EntityType;
        $scope.buttons = [];

        const notify = createNotifier({
          location: 'Siren Automatic Relational filter params'
        });

        // Updates the buttons to show according to the selected indexpattern.
        $scope.updateFilteredButtons = function () {
          $scope.filteredButtons = _.filter($scope.buttons, (button) => {
            return button.domainIndexPattern === $scope.filterSavedSearch.indexPattern || !button.domainIndexPattern;
          });
          $rootScope.$broadcast('siren:auto-join-params:filter:indexpattern', $scope.filterSavedSearch.indexPattern);
        };

        /**
         * This functions show the available target dashboards for a given EID relation
         */
        $scope.toggleTargetDashboards = function (button) {
          if (!button.compatibleDashboard) {
            ontologyModel.getRelationMap(button.indexRelationId)
              .then(relationMap => {
                const relation = relationMap[button.indexRelationId];
                let entity = relation.domain;
                if (relation.range.type === EntityType.VIRTUAL_ENTITY) {
                  entity = relation.range;
                }
                return Promise.all([
                  ontologyModelHelper.getDashboardsByEntity(entity),
                  ontologyModel.getRelationsByDomain(entity.id),
                  ontologyModel.getEntityList()
                ])
                  .then(([dashboards, targetRelations, entities]) => {
                    button.compatibleDashboard = {};
                    _.each(targetRelations, (targetRel) => {
                      const filteredDashboards = _.filter(dashboards, (dashboard) => {
                        // TODO compare root nodes savedSearch ids
                        return dashboard.savedSearchId !== relation.domain.id && targetRel.range.id === dashboard.savedSearchId;
                      });
                      const compatibleDashboards = _.map(filteredDashboards, (dashboard) => {
                        return dashboard.title;
                      });
                      const targetEntity = _.find(entities, entity => entity.id === targetRel.range.id);
                      if (relation.inverseOf !== targetRel.id &&
                        targetEntity &&
                        relation.domain.indexPattern !== targetEntity.indexPattern) {
                        if (button.compatibleDashboard[targetRel.directLabel]) {
                          compatibleDashboards.forEach(dashboard => {
                            if (button.compatibleDashboard[targetRel.directLabel].indexOf(dashboard) === -1) {
                              button.compatibleDashboard[targetRel.directLabel].push(dashboard);
                            };
                          });
                        } else {
                          button.compatibleDashboard[targetRel.directLabel] = compatibleDashboards;
                        };
                      };
                    });
                    const visibility = $scope.vis.params.visibility[button.id];
                    if (visibility) {
                      updateCompatibleDashboardsVisibility(button);
                      if ((visibility.button !== undefined && _.isEmpty(visibility.relation)) || visibility.button === false) {
                        $scope.vis.params.visibility[button.id] = newVisibilityForVirtualButtonChilds(button, visibility.button);
                      };
                    };
                    button.showTargetDashboards = !button.showTargetDashboards;
                  });
              });
          } else {
            button.showTargetDashboards = !button.showTargetDashboards;
          }
        };

        $scope.resetVisibility = function () {
          $scope.vis.params.visibility = {};
        };

        $scope.getButtonVisibilityClass = function (button) {
          const visibility = $scope.vis.params.visibility;
          if (!button.tooltip) {
            button.tooltip = {};
          }
          if (visibility[button.id] === undefined || visibility[button.id].button === undefined) {
            button.tooltip.root = 'Default visibility: visible';
            return 'fa fa-eye button-default';
          } else if (visibility[button.id].button) {
            button.tooltip.root = 'Visible';
            return 'fa fa-eye button-set';
          } else {
            button.tooltip.root = 'Not visible';
            return 'fa fa-eye-slash button-set';
          }
        };

        $scope.getRelationVisibilityClass = function (button, relName) {
          const visibility = $scope.vis.params.visibility;
          if (!button.tooltip) {
            button.tooltip = { relation: {} };
          } else if (!button.tooltip.relation) {
            button.tooltip.relation = {};
            button.tooltip.relation[relName] = {};
          }

          let css;
          if (visibility[button.id] && visibility[button.id].relation && visibility[button.id].relation[relName]
          && (visibility[button.id].relation[relName].toggle === true || visibility[button.id].relation[relName].toggle === false)) {
            css = 'button-set';
          } else if (button.tooltip.relation[relName]) {
            button.tooltip.relation[relName].tooltip =  'Default visibility: visible';
            css = 'button-default';
          }

          if (visibility[button.id] && visibility[button.id].relation && visibility[button.id].relation[relName]
          && visibility[button.id].relation[relName].toggle === false && button.tooltip.relation[relName]) {
            button.tooltip.relation[relName].tooltip =  'Not visible';
            css += ' fa-eye-slash';
          } else {
            if (css === 'button-set' && button.tooltip.relation[relName]) {
              button.tooltip.relation[relName].tooltip =  'Visible';
            }
            css += ' fa-eye';
          }

          return css;
        };

        $scope.getDashboardVisibilityClass = function (button, relName, dashboardName) {
          const visibility = $scope.vis.params.visibility;
          if (!button.tooltip) {
            button.tooltip = { relation: {} };
          } else if (!button.tooltip.relation) {
            button.tooltip.relation = {};
            button.tooltip.relation[relName] = { dashboard: {} };
          } else if (!button.tooltip.relation[relName]) {
            button.tooltip.relation[relName] = { dashboard: {} };
          } else if (!button.tooltip.relation[relName].dashboard) {
            button.tooltip.relation[relName].dashboard = {};
          }

          let css;
          if (visibility[button.id] && visibility[button.id].relation && visibility[button.id].relation[relName]
          && visibility[button.id].relation[relName].dashboard
          && (visibility[button.id].relation[relName].dashboard[dashboardName] === true
            || visibility[button.id].relation[relName].dashboard[dashboardName] === false)) {
            css = 'button-set';
          } else {
            button.tooltip.relation[relName].dashboard[dashboardName] =  'Default visibility: visible';
            css = 'button-default';
          }

          if (visibility[button.id] && visibility[button.id].relation && visibility[button.id].relation[relName]
          && visibility[button.id].relation[relName].dashboard
          && visibility[button.id].relation[relName].dashboard[dashboardName] === false) {
            button.tooltip.relation[relName].dashboard[dashboardName] =  'Not visible';
            css += ' fa-eye-slash';
          } else {
            if (css === 'button-set') {
              button.tooltip.relation[relName].dashboard[dashboardName] =  'Visible';
            }
            css += ' fa-eye';
          }

          return css;
        };

        $scope.toggleButtonVisibility = function (button) {
          let visibility;
          if ($scope.vis.params.visibility[button.id]) {
            visibility = $scope.vis.params.visibility[button.id];
          } else {
            visibility = {};
          }
          // default state
          if (_.isEmpty(visibility)) {
            visibility.button = false;
          } else if (visibility.button) {
            visibility.button = false;
          } else {
            visibility.button = true;
          }

          if (button.type === EntityType.VIRTUAL_ENTITY) {
            $scope.vis.params.visibility[button.id] = newVisibilityForVirtualButtonChilds(button, visibility.button);
          } else {
            $scope.vis.params.visibility[button.id] = visibility;
          };
        };

        $scope.toggleRelationVisibility = function (button, relationName) {
          let visibilityWasUndefined = false;
          let visibility = $scope.vis.params.visibility[button.id];
          if (!$scope.vis.params.visibility[button.id] || !$scope.vis.params.visibility[button.id].relation[relationName]) {
            visibility = newVisibilityForVirtualButtonChilds(button, true);
            visibilityWasUndefined = true;
          };
          visibility.relation[relationName].toggle = !visibility.relation[relationName].toggle;
          if (visibilityWasUndefined) {
            $scope.vis.params.visibility[button.id] = visibility;
          };
          switchVisibilityForRelationChilds(button, relationName, visibility.relation[relationName].toggle);
          return visibility;
        };

        function dashboardsForRelationAreHidden(button, relationName) {
          let noVisibleDashboards = true;
          const visibilityRule = $scope.vis.params.visibility[button.id];
          _.each(visibilityRule.relation[relationName].dashboard, (dashVisibility, dashTitle) => {
            if (dashVisibility) {
              noVisibleDashboards = false;
              return false;
            }
          });
          return noVisibleDashboards;
        }

        function allDashboardsAreHidden(button) {
          let noVisibleDashboards = true;
          const visibilityRule = $scope.vis.params.visibility[button.id];
          _.each(visibilityRule.relation, (relValue, relName) => {
            _.each(relValue.dashboard, (dashVisibility, dashTitle) => {
              if (dashVisibility) {
                noVisibleDashboards = false;
                return false;
              };
            });
            if (!noVisibleDashboards) {
              return false;
            };
          });
          return noVisibleDashboards;
        };

        function newVisibilityForVirtualButtonChilds(button, visibility) {
          const visibilityForButton = {
            button: visibility
          };
          const createdRelation = {};
          _.each(button.compatibleDashboard, (dashboards, relName) => {
            createdRelation[relName] = {
              dashboard: {},
              toggle: visibility
            };
            dashboards.forEach(dashboardName => {
              createdRelation[relName].dashboard[dashboardName] = visibility;
            });
          });
          visibilityForButton.relation = createdRelation;
          return visibilityForButton;
        };

        function switchVisibilityForRelationChilds(button, relationName, visibility) {
          const currentVisibility = $scope.vis.params.visibility[button.id];
          const updatedVisibility = {};
          _.each(currentVisibility.relation[relationName].dashboard, (dashVisibility, dashTitle) => {
            updatedVisibility[dashTitle] = visibility;
          });
          currentVisibility.relation[relationName].dashboard = updatedVisibility;
          if (allDashboardsAreHidden(button)) {
            currentVisibility.button = false;
          } else {
            currentVisibility.button = true;
          };
        };

        function updateCompatibleDashboardsVisibility(button) {
          let relationsDashboardsMap = $scope.vis.params.visibility[button.id].relation;
          if (!relationsDashboardsMap) {
            relationsDashboardsMap = {};
          };
          const dashboardsToAdd = Object.keys(button.compatibleDashboard);
          _.each(relationsDashboardsMap, (dashboards, relation) => {
            if (button.compatibleDashboard[relation]) {
              const relationsDashboardsMapForRelation = Object.keys(dashboards.dashboard);
              const dashboardsFromButton = button.compatibleDashboard[relation];
              relationsDashboardsMapForRelation.forEach(dashboardTitle => {
                if (dashboardsFromButton.indexOf(dashboardTitle) === -1) {
                  delete relationsDashboardsMap[relation].dashboard[dashboardTitle];
                };
              });
              dashboardsFromButton.forEach(dashboardTitle => {
                if (relationsDashboardsMapForRelation.indexOf(dashboardTitle) === -1) {
                  relationsDashboardsMap[relation].dashboard[dashboardTitle] = true;
                };
              });
              dashboardsToAdd.splice(dashboardsToAdd.indexOf(relation), 1);
            } else {
              delete relationsDashboardsMap[relation];
            };
          });
          dashboardsToAdd.forEach(relation => {
            relationsDashboardsMap[relation] = {
              dashboard: {},
              toggle: true
            };
            button.compatibleDashboard[relation].forEach(dashboardTitle => {
              relationsDashboardsMap[relation].dashboard[dashboardTitle] = true;
            });
          });
        };

        function getLastDashboardIdFromLastUrl(url) {
          const dashIdByRegex = url.match(/dashboard:[A-Za-z0-9-]+/);
          return dashIdByRegex[0];
        };

        $scope.toggleDashboardVisibility = function (button, relationName, dashboardName) {
          let visibilityWasUndefined = false;
          let visibility = $scope.vis.params.visibility[button.id];
          if (!visibility || !visibility.relation[relationName]) {
            visibility = newVisibilityForVirtualButtonChilds(button, true);
            visibilityWasUndefined = true;
          }
          visibility.relation[relationName].dashboard[dashboardName] = !visibility.relation[relationName].dashboard[dashboardName];

          if (visibilityWasUndefined) {
            $scope.vis.params.visibility[button.id] = visibility;
          };
          if (allDashboardsAreHidden(button)) {
            visibility.button = false;
            _.each(visibility.relation, (property, key) => {
              property.toggle = false;
            });
          };
          if (dashboardsForRelationAreHidden(button, relationName)) {
            visibility.relation[relationName].toggle = false;
          } else {
            visibility.relation[relationName].toggle = true;
            visibility.button = true;
          };
          return visibility;
        };

        // Init the config panel
        Promise.all([
          ontologyModel.getRelationList(),
          ontologyModel.getEntityList()
        ])
          .then(([relations, entities]) => {
            const availableSavedSearches = _.filter(entities, entity => {
              return entity.type === EntityType.SAVED_SEARCH && !entity.parentId;
            });
            availableSavedSearches.sort((a, b) => {
              return (a.label.toLowerCase() > b.label.toLowerCase());
            });
            $scope.availableSavedSearches = availableSavedSearches;

            const lastDashboardLink = _.find(chrome.getNavLinks(), link => link.id === 'kibana:dashboard');
            if (lastDashboardLink) {
              const dashIdByRegex = getLastDashboardIdFromLastUrl(lastDashboardLink.lastSubUrl);
              return Promise.resolve(savedDashboards.get(dashIdByRegex))
                .then((dashboard) => {
                  if (dashboard && dashboard.savedSearchId) {
                    const selectedEntity = _.find(availableSavedSearches, search => dashboard.savedSearchId === search.id);
                    $scope.filterSavedSearch = selectedEntity;
                  };
                  return relations;
                });
            } else {
              return relations;
            };

          })
          .then(relations => {

            // autoselect the first one if it was not defined before
            if (!$scope.filterSavedSearch) {
              $scope.filterSavedSearch = $scope.availableSavedSearches[0];
            };

            // check if we have to add buttons for new relations with EIDs
            const relationsWithNoButton = _.filter(relations, (rel) => {
              return rel.domain.type === EntityType.SAVED_SEARCH;
            });

            return Promise.all([
              savedDashboards.find(),
              savedSearches.find()
            ])
              .then(([savedDashboards, savedSearches]) => {
                _.each(relationsWithNoButton, (rel) => {
                  const button = {
                    indexRelationId: rel.id,
                    domainIndexPattern: rel.domain.indexPattern,
                    status: 'default'
                  };
                  if (rel.range.type === EntityType.VIRTUAL_ENTITY) {
                    button.type = EntityType.VIRTUAL_ENTITY;
                    button.id = rel.id + '-ve-' + rel.range.id;
                    button.label = rel.directLabel + ' (' + rel.range.id + ')';
                    $scope.buttons.push(button);
                  } else if (rel.range.type === EntityType.SAVED_SEARCH) {
                    button.type = EntityType.SAVED_SEARCH;

                    const compatibleSavedSearches = _.filter(savedSearches.hits, (savedSearch) => {
                      const searchSource = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
                      return searchSource.index === rel.range.indexPattern;
                    });

                    _.each(compatibleSavedSearches, (compatibleSavedSearch) => {
                      const compatibleDashboards = _.filter(savedDashboards.hits, (savedDashboard) => {
                        const savedSearchId = findMainCoatNodeSearchId(savedDashboard.coatJSON);
                        return savedSearchId === compatibleSavedSearch.id;
                      });
                      _.each(compatibleDashboards, (compatibleDashboard) => {
                        const clonedButton = _.clone(button);
                        clonedButton.label = rel.directLabel + ' (' + compatibleDashboard.title + ')';
                        clonedButton.id = rel.id + '-ip-' + compatibleDashboard.title;
                        $scope.buttons.push(clonedButton);
                      });
                    });
                  }
                });
              })
              .then($scope.updateFilteredButtons);
          });

      }
    };
  });
