'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FederateClient = undefined;

var _fs = require('fs');

var _http = require('http');

var _http2 = _interopRequireDefault(_http);

var _https = require('https');

var _https2 = _interopRequireDefault(_https);

var _url = require('url');

var _url2 = _interopRequireDefault(_url);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _licensing = require('../../federate_resolver/licensing');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function readFile(file) {
  return (0, _fs.readFileSync)(file, 'utf8');
}

function getRequestOptions(config, isHttps, parsedEsURL) {
  const username = config.get('elasticsearch.username');
  const password = config.get('elasticsearch.password');

  const options = {
    headers: {
      'Content-Type': 'application/json'
    },
    timeout: config.get('elasticsearch.requestTimeout'),
    hostname: parsedEsURL.hostname,
    port: parsedEsURL.port
  };
  if (username && password) {
    options.auth = `${username}:${password}`;
  }

  if (!isHttps) {
    return options;
  }
  const verificationMode = config.get('elasticsearch.ssl.verificationMode');
  switch (verificationMode) {
    case 'none':
      options.rejectUnauthorized = false;
      break;
    case 'certificate':
      options.rejectUnauthorized = true;

      // by default, NodeJS is checking the server identify
      options.checkServerIdentity = _lodash2.default.noop;
      break;
    case 'full':
      options.rejectUnauthorized = true;
      break;
    default:
      throw new Error(`Unknown ssl verificationMode: ${verificationMode}`);
  }

  if (_lodash2.default.size(config.get('elasticsearch.ssl.certificateAuthorities'))) {
    options.ca = config.get('elasticsearch.ssl.certificateAuthorities').map(readFile);
  }

  // Add client certificate and key if required by elasticsearch
  if (config.get('elasticsearch.ssl.certificate') && config.get('elasticsearch.ssl.key')) {
    options.cert = readFile(config.get('elasticsearch.ssl.certificate'));
    options.key = readFile(config.get('elasticsearch.ssl.key'));
    options.passphrase = config.get('elasticsearch.ssl.keyPassphrase');
  }
  return options;
}

function executeRequest(options, isHttps) {
  return new Promise((resolve, reject) => {
    const httpClient = isHttps ? _https2.default : _http2.default;
    const request = httpClient.request(options, function (response) {
      if (response.statusCode !== 200) {
        reject(new Error(`Responded with status code ${response.statusCode}`));
      }
      const chunks = [];
      response.on('data', function (chunk) {
        chunks.push(chunk);
      });
      response.on('end', function (data) {
        const ret = Buffer.concat(chunks).toString('utf8');
        resolve(ret);
      });
    });
    request.end();
    request.on('error', reject);
  });
}

//
// Client to be used by other plugins when making a direct calls to specific internal federate API from backend side
// Uses the internal elasticsearch user credentials
// e.g. when making a call to _siren/sessions to revoke a session on logout
// Should never be used to proxy calls made from frontend in such case use the es_admin path which is handled by create_kibi_proxy.js
//
class FederateClient {
  constructor(server) {
    const config = server.config();
    const parsedEsURL = _url2.default.parse(config.get('elasticsearch.url'));
    this.server = server;
    this.isHttps = /^https/.test(parsedEsURL.protocol);
    this.options = getRequestOptions(config, this.isHttps, parsedEsURL);
    this.federateResolver = server.plugins.federate_resolver.getFederateResolver();
    this.licenseParser = new _licensing.LicenseParser(this.federateResolver);
  }

  getFederateSessionHeader(req) {
    if (req.state && req.state.sirenSID) {
      const header = {
        name: 'X-Federate-Session-Id'
      };
      // NOTE:
      // This has to be dynamic check as during this plugin initialization access control plugin might not yet be available
      // and we do not want to depend on access control
      const isAccessControlEnabled = !!_lodash2.default.get(this.server, 'plugins.investigate_access_control');
      if (isAccessControlEnabled && req.auth && req.auth.credentials && req.auth.credentials._username) {
        header.value = req.auth.credentials._username + req.state.sirenSID;
      } else {
        header.value = req.state.sirenSID;
      }
      return header;
    }
    return null;
  }

  async removeFederateSession(req) {
    const header = this.getFederateSessionHeader(req);
    if (this.federateResolver.checkApiAvailability('SESSION_API_V1') && header) {
      const options = _lodash2.default.cloneDeep(this.options);
      options.path = `/_siren/sessions/${header.value}`;
      options.method = 'DELETE';
      options.headers[header.name] = header.value;
      return executeRequest(options, this.isHttps);
    } else {
      return Promise.resolve({});
    }
  }

  async getSearchguardLicense(req) {
    const options = _lodash2.default.cloneDeep(this.options);
    options.path = '/_searchguard/license';
    options.method = 'GET';
    try {
      const resp = await executeRequest(options, this.isHttps);
      const parsedResp = JSON.parse(resp);
      return parsedResp.sg_license ? parsedResp.sg_license : { error: 'No sg_license property in the response' };
    } catch (error) {
      return { error };
    }
  }

  async getLicense() {
    const options = _lodash2.default.cloneDeep(this.options);
    options.path = '/_siren/license';
    options.method = 'GET';
    try {
      const resp = await executeRequest(options, this.isHttps);
      return this.licenseParser.ensureLatest(JSON.parse(resp));
    } catch (error) {
      return { error };
    }
  }

}
exports.FederateClient = FederateClient;
