'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _bluebird = require('bluebird');

var _boom = require('boom');

var _boom2 = _interopRequireDefault(_boom);

var _path = require('path');

var _utils = require('./../utils');

var _ui_exports = require('./ui_exports');

var _ui_exports2 = _interopRequireDefault(_ui_exports);

var _ui_bundle = require('./ui_bundle');

var _ui_bundle2 = _interopRequireDefault(_ui_bundle);

var _ui_bundle_collection = require('./ui_bundle_collection');

var _ui_bundle_collection2 = _interopRequireDefault(_ui_bundle_collection);

var _ui_bundler_env = require('./ui_bundler_env');

var _ui_bundler_env2 = _interopRequireDefault(_ui_bundler_env);

var _ui_i18n = require('./ui_i18n');

var _ui_settings = require('./ui_settings');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

exports.default = async (kbnServer, server, config) => {
  const uiExports = kbnServer.uiExports = new _ui_exports2.default({
    urlBasePath: config.get('server.basePath')
  });

  await kbnServer.mixin(_ui_settings.uiSettingsMixin);

  const uiI18n = kbnServer.uiI18n = new _ui_i18n.UiI18n(config.get('i18n.defaultLocale'));
  uiI18n.addUiExportConsumer(uiExports);

  const bundlerEnv = new _ui_bundler_env2.default((0, _utils.fromRoot)(config.get('optimize.bundleDir')));
  bundlerEnv.addContext('env', config.get('env.name'));
  bundlerEnv.addContext('sourceMaps', config.get('optimize.sourceMaps'));
  bundlerEnv.addContext('kbnVersion', config.get('pkg.version'));
  // kibi: added to manage kibi version
  bundlerEnv.addContext('kibiVersion', config.get('pkg.kibiVersion'));
  // kibi: added to display the version of kibana this kibi is based on
  bundlerEnv.addContext('kibiKibanaAnnouncement', config.get('pkg.kibiKibanaAnnouncement'));
  bundlerEnv.addContext('buildNum', config.get('pkg.buildNum'));
  uiExports.addConsumer(bundlerEnv);

  for (const plugin of kbnServer.plugins) {
    uiExports.consumePlugin(plugin);
  }

  const bundles = kbnServer.bundles = new _ui_bundle_collection2.default(bundlerEnv, config.get('optimize.bundleFilter'));

  for (const app of uiExports.getAllApps()) {
    bundles.addApp(app);
  }

  for (const gen of uiExports.getBundleProviders()) {
    const bundle = await gen(_ui_bundle2.default, bundlerEnv, uiExports.getAllApps(), kbnServer.plugins);
    if (bundle) bundles.add(bundle);
  }

  // render all views from the ui/views directory
  server.setupViews((0, _path.resolve)(__dirname, 'views'));

  server.route({
    path: '/app/{id}',
    method: 'GET',
    async handler(req, reply) {
      const id = req.params.id;

      // kibi: block access to specific apps
      // we exclude kibana as then nothing would work at all
      if (server.plugins.investigate_access_control && config.get('investigate_access_control.acl.enabled') && id !== 'kibana') {
        const user = req.auth.credentials;
        const result = await server.plugins.investigate_access_control.isAllowed(user, ['view'], 'app:' + id);
        if (result === false) {
          return reply(`Access to ${id} denied by ACL.<br/><a href="../">Go back</a>`);
        }
      }

      const app = uiExports.apps.byId[id];
      if (!app) return reply(_boom2.default.notFound('Unknown app ' + id));

      try {
        if (kbnServer.status.isGreen()) {
          await reply.renderApp(app);
        } else {
          await reply.renderStatusPage();
        }
      } catch (err) {
        reply(_boom2.default.wrap(err));
      }
    }
  });

  async function getKibanaPayload({ app, request, includeUserProvidedConfig, injectedVarsOverrides }) {
    const uiSettings = request.getUiSettingsService();
    const translations = await uiI18n.getTranslationsForRequest(request);

    return {
      app: app,
      nav: uiExports.navLinks.inOrder,
      version: kbnServer.version,
      branch: kbnServer.branch,
      kibiVersion: kbnServer.kibiVersion, // kibi: added to manage kibi version
      kibiKibanaAnnouncement: kbnServer.kibiKibanaAnnouncement, // kibi: added to manage kibi announcement
      buildNum: config.get('pkg.buildNum'),
      buildSha: config.get('pkg.buildSha'),
      buildTimestamp: config.get('pkg.buildTimestamp'),
      basePath: config.get('server.basePath'),
      serverName: config.get('server.name'),
      devMode: config.get('env.dev'),
      translations: translations,
      uiSettings: await (0, _bluebird.props)({
        defaults: uiSettings.getDefaults(),
        user: includeUserProvidedConfig && uiSettings.getUserProvided(request) // kibi: pass the request
      }),
      vars: await (0, _bluebird.reduce)(uiExports.injectedVarsReplacers, async (acc, replacer) => await replacer(acc, request, server), (0, _lodash.defaults)(injectedVarsOverrides, (await app.getInjectedVars()) || {}, uiExports.defaultInjectedVars))
    };
  }

  async function renderApp({ app, reply, includeUserProvidedConfig = true, injectedVarsOverrides = {} }) {
    try {
      const request = reply.request;
      const translations = await uiI18n.getTranslationsForRequest(request);

      return reply.view(app.templateName, {
        app,
        kibanaPayload: await getKibanaPayload({
          app,
          request,
          includeUserProvidedConfig,
          injectedVarsOverrides
        }),
        bundlePath: `${config.get('server.basePath')}/bundles`,
        i18n: key => _lodash._.get(translations, key, '')
      });
    } catch (err) {
      reply(err);
    }
  }

  server.decorate('reply', 'renderApp', function (app, injectedVarsOverrides) {
    return renderApp({
      app,
      reply: this,
      includeUserProvidedConfig: true,
      injectedVarsOverrides
    });
  });

  server.decorate('reply', 'renderAppWithDefaultConfig', function (app) {
    return renderApp({
      app,
      reply: this,
      includeUserProvidedConfig: false
    });
  });
};

module.exports = exports['default'];
