import React from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';
import {
  EuiFieldSearch,
  EuiButton,
  EuiBasicTable,
  EuiIcon,
  EuiLink,
  EuiFlexGroup,
  EuiFlexItem
} from '@elastic/eui';
import _ from 'lodash';
import rison from 'rison-node';

// kibi: imports
import { onDashboardPage } from 'ui/kibi/utils/on_page';
// kibi: end

export class ObjectFinder extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      loading: false,
      //paging
      pageIndex: 0,
      pageOfItems: [],
      //sort
      sortField: 'name',
      sortDirection: 'asc',
      //items
      totalItemCount: 0,
      hits: [],
      //filter
      currentFilter: '',
      prevSearch: null,
      // kibi: variable to store the checkbox state
      investigate: {
        onDashboardPage: onDashboardPage(),
        basedOnSameSavedSearch: false,
        _prevBasedOnSameSavedSearch: false
      }
    };
  }

  onFilterChange = (event) => {
    this.setState({ currentFilter: event.target.value });
  }

  onTableChange = ({ page = {}, sort = {} }) => {
    const {
      index: pageIndex
    } = page;

    const {
      field: sortField,
      direction: sortDirection,
    } = sort;

    const orderedHits = (sortDirection === 'asc') ? _.sortBy(this.state.hits, 'title') : _.sortBy(this.state.hits, 'title').reverse();
    const pageSize = this.props.perPage;
    const pageOfItems = orderedHits.splice(pageIndex * pageSize, pageSize);
    this.setState({
      pageIndex,
      sortField,
      sortDirection,
      pageOfItems
    });
  };

  onCheckChange = (event) => {
    this.setState({
      investigate: {
        onDashboardPage: this.state.investigate.onDashboardPage,
        basedOnSameSavedSearch: event.target.checked,
        _prevBasedOnSameSavedSearch: this.state.investigate._prevBasedOnSameSavedSearch
      }
    });
  }

  /**
   * Passed the hit objects and will determine if the
   * hit should have a url in the UI, returns it if so
   * @return {string|null} - the url or nothing
   */
  makeUrl = (hit) => {
    if (this.props.userMakeUrl) {
      return this.props.userMakeUrl(hit);
    }

    if (!this.props.userOnChoose) {
      return hit.url;
    }

    return '#';
  };

  /**
   * Called when a hit object is clicked, can override the
   * url behavior if necessary.
   */
  onChoose = (event, hit) => {
    const nativeEvent = event.nativeEvent;
    if (this.props.userOnChoose) {
      this.props.userOnChoose(hit, nativeEvent);
    }

    const url = this.makeUrl(hit);
    if (!url || url === '#' || url.charAt(0) !== '#') return;

    nativeEvent.preventDefault();

    // we want the '/path', not '#/path'
    this.props.kbnUrl.change(url.substr(1));
  };

  manageObjects = (type) => {
    this.props.location.url('/management/siren/objects?_a=' + rison.encode({ tab: type }));
  };

  filterResults = () => {
    if (!this.props.service) return;
    if (!this.props.properties) return;

    // track the filter that we use for this search,
    // but ensure that we don't search for the same
    // thing twice. This is called from multiple places
    // and needs to be smart about when it actually searches
    const filter = this.state.currentFilter;
    // kibi: allow to filter visualizations based on the savedsearch associated to a dashboard
    const basedOnSameSavedSearch = this.state.investigate.basedOnSameSavedSearch;
    if (this.state.prevSearch === filter && this.state.investigate._prevBasedOnSameSavedSearch === basedOnSameSavedSearch) {
      return;
    }
    // kibi: end

    this.state.prevSearch = filter;
    this.state.investigate._prevBasedOnSameSavedSearch = basedOnSameSavedSearch;

    this.state.loading = true;
    this.props.service.find(filter)
      .then((hits) => {
      // ensure that we don't display old results
      // as we can't really cancel requests

      // kibi: filter visualization based on the same saved search as the dashboard if requested
        const filtered = _.filter(hits.hits, (hit) => {
          if (this.props.savedSearchId && basedOnSameSavedSearch) {
            return hit.savedSearchId === this.props.savedSearchId;
          } else {
            return true;
          }
        });

        let orderedHits = [];
        if (this.state.currentFilter === filter) {
          orderedHits = (this.state.sortDirection === 'asc') ? _.sortBy(filtered, 'title') : _.sortBy(filtered, 'title').reverse();
        } else {
          orderedHits = filtered.slice(0);
        }
        const pageOfItems = orderedHits.splice(this.state.pageIndex, this.state.pageIndex + this.props.perPage);
        this.setState({ totalItemCount: filtered.length, loading: false, pageOfItems: pageOfItems, hits: filtered });
      });
  }

  getRowProps = (item) => {
    const props = {
      onClick: (event) => {
        this.state.selectedItem = item.id;
        this.onChoose(event, item);
      },
    };
    if (this.state.selectedItem === item.id) {
      props.className = 'euiTableRow-selected';
    }
    return props;
  };

  render() {
    const getIcon = (hit) => {
      if (hit.icon) {
        return <span aria-hidden='true' className={ 'finder-type fa ' + hit.icon } data-test-subj="index-pattern-icon"></span>;
      } else if (hit.type && hit.type.image) {
        return <img className='kuiStatusText__icon kuiIcon' aria-hidden='true' src={ hit.type.image } data-test-subj="index-pattern-icon"/>;
      }
      return <span/>;
    };

    const getSavedSearchOptionDiv = () => {
      if (this.state.investigate.onDashboardPage && this.props.type === 'visualizations') {
        if (this.props.savedSearchId) {
          return <div className="fullWidth">
            <input type='checkbox'
              onChange={this.onCheckChange}
              data-test-subj= {"input-checkbox-" + this.props.savedSearchTitle }/>
            <label className='based-saved-search'>
                    Show only visualizations linked to the saved search {this.props.savedSearchTitle}.
            </label>
          </div>;
        } else {
          return <div className="fullWidth">
            <input type="checkbox" disabled/>
            <label className='based-saved-search'>
                    To be able to filter by saved search, set a saved search when saving this dashboard.
            </label>
          </div>;
        }
      }
      return <span/>;
    };

    const getManageButton = () => {
      if (this.props.onAddNew) {
        return  <EuiButton onClick={ this.props.onAddNew } data-test-subj= {"add-new-" + this.props.properties.name + "-btn"}>
                  Add new { this.props.properties.noun }
        </EuiButton>;
      } else if (!this.props.useLocalManagement) {
        return  <EuiButton onClick={ () => { this.manageObjects(this.props.properties.name); } }
          data-test-subj={"manage-" + this.props.properties.name + "-btn"}>
                  Manage { this.props.properties.noun }
        </EuiButton>;
      }
    };

    const columns = [];
    const column = {
      field: 'title',
      name: 'Name',
      sortable: true,
      truncateText: true,
      hideForMobile: true,
    };
    column.render = (name, item) => (
      <div>
        {getIcon(item)}
        <EuiLink target='_blank' data-test-subj={name}> {name} </EuiLink>
      </div>
    );
    columns.push(column);

    const pagination = {
      pageIndex: this.state.pageIndex,
      pageSize: this.props.perPage,
      totalItemCount: this.state.totalItemCount,
      pageSizeOptions: [this.props.perPag],
      hidePerPageOptions: true
    };

    const sorting = {
      sort: {
        field: this.state.sortField,
        direction: this.state.sortDirection,
      },
    };

    this.filterResults();

    return (
      <div>
        <div>
          { getSavedSearchOptionDiv() }
        </div>

        <div>
          <div>
            <EuiFlexGroup>
              <EuiFlexItem>
                <EuiFieldSearch
                  fullWidth
                  placeholder="Search..."
                  onChange={ this.onFilterChange }
                  incremental={true}
                  data-test-subj="input-saved-visualization-field"
                />
              </EuiFlexItem>
              <EuiFlexItem grow={false}>{ getManageButton() }</EuiFlexItem>
            </EuiFlexGroup>
          </div>
          <EuiBasicTable
            loading={ this.state.loading }
            items={ this.state.pageOfItems }
            columns={ columns }
            pagination={ pagination }
            sorting={ sorting }
            onChange={ this.onTableChange }
            rowProps={ this.getRowProps }
          />
        </div>

      </div>
    );
  }
}

ObjectFinder.propTypes = {
  service: PropTypes.object.isRequired,
  properties: PropTypes.object,
  savedSearchId: PropTypes.string,
  savedSearchTitle: PropTypes.string,
  userMakeUrl: PropTypes.func,
  userOnChoose: PropTypes.func,
  type: PropTypes.string.isRequired,
  perPage: PropTypes.number.isRequired,
  onAddNew: PropTypes.func,
  kbnUrl: PropTypes.object,
  location: PropTypes.object
};

export function objectFinder(container, service, properties, savedSearchId, savedSearchTitle,
  userMakeUrl, userOnChoose, type, perPage, onAddNew, useLocalManagement, kbnUrl, location) {
  const element = (
    <ObjectFinder service={service}
      properties={properties}
      savedSearchId={savedSearchId}
      savedSearchTitle={savedSearchTitle}
      userMakeUrl={userMakeUrl}
      userOnChoose={userOnChoose}
      type={type}
      perPage={perPage}
      onAddNew={onAddNew}
      useLocalManagement={useLocalManagement}
      kbnUrl={kbnUrl}
      location={location}
    >
    </ObjectFinder>
  );
  ReactDOM.render(element, container);
}
