import _ from 'lodash';
import html from 'ui/doc_table/doc_table.html';
import { getSort } from 'ui/doc_table/lib/get_sort';
import 'ui/doc_table/doc_table.less';
import 'ui/directives/truncated';
import 'ui/directives/infinite_scroll';
import 'ui/doc_table/components/table_header';
import 'ui/doc_table/components/table_row';
import { uiModules } from 'ui/modules';

import { getLimitedSearchResultsMessage } from './doc_table_strings';

// kibi: imports
import { ExportAsCsvProvider } from 'plugins/kibi_data_table_vis/actions/csv_export';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { onDashboardPage } from 'ui/kibi/utils/on_page';
import { filterHelper } from 'ui/kibi/components/dashboards360/filter_helper';
import { coatHelper } from 'ui/kibi/components/dashboards360/coat_helper';
// kibi: end

uiModules.get('kibana')
  .directive('docTable', function (Private, courier, config, createNotifier, getAppState, pagerFactory, $filter, $timeout,
    savedVisualizations, timefilter, savedDashboards, kibiState, $rootScope) {
    return {
      restrict: 'E',
      template: html,
      scope: {
        sorting: '=',
        columns: '=',
        hits: '=?', // You really want either hits & indexPattern, OR searchSource
        indexPattern: '=?',
        searchSource: '=?',
        infiniteScroll: '=?',
        filter: '=?',
        filters: '=?',
        onAddColumn: '=?',
        onChangeSortOrder: '=?',
        onMoveColumn: '=?',
        onRemoveColumn: '=?',
        title: '@?', // kibi: added by us to pass the visualization title
        savedSearchId: '=?', // kibi: added to use sirenDataExport

        // kibi:
        // added cellClickHandlers and customColumnAliases
        // to make them available to the scope of kibiTableRow and kibiTableHeader
        cellClickHandlers: '=',
        customColumnAliases: '=?',
        customColumnWordWrap: '=?',
        // kibi: increase the number of results that are retrieved
        increaseSample: '@?',
        // kibi: export hits as CSV
        csv: '@?',
        // kibi: resize hit count per page
        pageSize: '=?',
        // kibi: custom view
        templateId: '=?',
        showCustomView: '=?',
        customView: '=?',
        customViewerMode: '=?',
        // kibi: end
        // kibi: enable top paginator and min width for the columns, plus, make time field column optional
        enableTopPaginator: '=?',
        customColumnMinWidth: '=?',
        disableTimeField: '=?',
        searchEngineLook: '=?'
        // kibi: end
      },
      link: function ($scope, $element) {
        const notify = createNotifier({ location: 'Doc table' });
        $scope.limit = 50;
        $scope.persist = {
          sorting: $scope.sorting,
          columns: $scope.columns
        };

        // kibi: add exportAsCsv to the scope
        $scope.exportAsCsv = Private(ExportAsCsvProvider).exportAsCsv;
        const queryFilter = Private(FilterBarQueryFilterProvider);
        $scope.mfilters = [];

        const $state = getAppState();

        $scope.handleTimeQueryOrFilterChange = function () {
          return savedDashboards.get(kibiState.getCurrentDashboardId())
            .then(function (dashboard) {
              return kibiState.compareStateToSavedDashState(dashboard)
                .then(function (resp) {
                  if (resp.stateEqual) {
                    $scope.shouldShowTable = false;
                    // kibi: let kibi_data_table_vis_controller.js know the value for custom view header
                    $scope.$emit('shouldShowTable:changed', false);
                  } else {
                    $scope.shouldShowTable = true;
                    // kibi: let kibi_data_table_vis_controller.js know the value for custom view header
                    $scope.$emit('shouldShowTable:changed', true);
                  }
                });
            });
        };

        $scope.$watch('searchEngineLook', function (searchEngineLook) {
          $scope.shouldShowTable = !searchEngineLook;
        });

        if ($scope.searchEngineLook) {
          $scope.shouldShowTable = false;
          if (onDashboardPage()) {
            $scope.handleTimeQueryOrFilterChange();
            $rootScope.$listen($state, 'save_with_changes', function () {
              return $scope.handleTimeQueryOrFilterChange();
            });
            $scope.$listen(timefilter, 'fetch', () => {
              return $scope.handleTimeQueryOrFilterChange();
            });
          }
        } else {
          $scope.shouldShowTable = true;
        };

        const createFilter = function (alias) {
          const filterQuery = {
            meta: {},
            query: {
              bool: {
                should: []
              }
            }
          };

          filterHelper.addSirenPropertyToFilterMeta(filterQuery, $scope.searchSource._siren);

          if (alias) {
            filterQuery.meta.alias = alias;
          };

          _.each($scope.mfilters, (row) => {
            filterQuery.query.bool.should.push({
              bool: {
                must: [
                  {
                    term: {
                      _uid: `${row._type}#${row._id}`
                    },
                  },
                  {
                    term: {
                      _index: row._index
                    }
                  }
                ]
              }
            });
          });

          queryFilter.addFilters(filterQuery);
          $scope.mfiltersEnabled.enabled = false;
          $scope.mfilters = [];
        };

        $scope.addFilters = () => {
          // kibi: if it is a visualization use saved search, if it is a saved search use title
          // we assume that the $scope.searchSource._siren will always be there

          // TODO: use $scope.searchSource._siren to grab all neccessary info

          if ($scope.searchSource && $scope.searchSource.vis && $scope.searchSource.vis.id) {
            return savedVisualizations.get($scope.searchSource.vis.id)
              .then((savedVis) => {
                const alias = $scope.mfilters.length + ' entities from ' + savedVis.savedSearch.title;
                createFilter(alias);
              }).catch(err => {
                createFilter();
              });
          } else if ($scope.title) {
            const alias = $scope.mfilters.length + ' entities from ' + $scope.title;
            createFilter(alias);
          } else {
            createFilter();
          }
        };

        // kibi: increase the number of results retrieved
        $scope.size = $scope.pageSize || parseInt(config.get('discover:sampleSize'));
        $scope.hasNextPage = function () {
          return $scope.increaseSample ? ($scope.pager.endItem !== $scope.totalHitCount) : $scope.pager.hasNextPage;
        };
        // kibi: end

        const prereq = (function () {
          const fns = [];

          return function register(fn) {
            fns.push(fn);

            return function () {
              fn.apply(this, arguments);

              if (fns.length) {
                _.pull(fns, fn);
                if (!fns.length) {
                  $scope.$root.$broadcast('ready:vis');
                }
              }
            };
          };
        }());
        const limitTo = $filter('limitTo');
        const calculateItemsOnPage = () => {
          $scope.pager.setTotalItems($scope.hits.length);
          $scope.pageOfItems = limitTo($scope.hits, $scope.pager.pageSize, $scope.pager.startIndex);
        };

        $scope.limitedResultsWarning = getLimitedSearchResultsMessage(config.get('discover:sampleSize'));

        $scope.addRows = function () {
          $scope.limit += 50;
        };

        // This exists to fix the problem of an empty initial column list not playing nice with watchCollection.
        $scope.$watch('columns', function (columns) {
          if (columns.length !== 0) return;

          $scope.columns.push('_source');
          if ($state) $state.replace();
        });

        $scope.$watchCollection('columns', function (columns, oldColumns) {
          if (oldColumns.length === 1 && oldColumns[0] === '_source' && $scope.columns.length > 1) {
            _.pull($scope.columns, '_source');
          }

          if ($scope.columns.length === 0) $scope.columns.push('_source');
        });

        // Kibi: filters and query flags set on appState changes.
        // Attached to the scope for testing purposes
        $scope.filtersQueryOrTimeChanged = false;
        $scope.sortChanged = false;
        $scope.queryChanged = false;
        $scope.coatChanged = false;
        $scope._oldQuery;

        // Kibi: catch query changes
        const appStateHandler = function (diff) {
          const checkQuery = _.indexOf(diff, 'query');
          const checkFilters = _.indexOf(diff, 'filters');
          if (checkQuery !== -1 || checkFilters !== -1) {
            $scope.filtersQueryOrTimeChanged = true;
          }
        };

        let appStateHandlerOff;
        $scope.$watch(getAppState, (appState) => {
          if (appState) {
            if (appStateHandlerOff) {
              // kibi: remove the old handler before attaching the new one
              appStateHandlerOff();
            }
            appState.on('save_with_changes', appStateHandler);
            appStateHandlerOff = function () {
              appState.off('save_with_changes', appStateHandler);
            };
          }
        });

        $scope.$listen(timefilter, 'fetch', () => {
          $scope.filtersQueryOrTimeChanged = true;
        });

        const isPageSizeChanged = function () {
          return $scope.pageSize && $scope.pager && $scope.pageSize !== $scope.pager.pageSize;
        };

        const refreshTable = prereq(function () {
          if (!$scope.searchSource) return;

          $scope.indexPattern = $scope.searchSource.get('index');

          // kibi: use '$scope.size' for the 'searchSource.size'
          $scope.searchSource.size($scope.size);
          $scope.searchSource.sort(getSort($scope.sorting, $scope.indexPattern));

          // Set the watcher after initialization
          $scope.$watchCollection('sorting', function (newSort, oldSort) {
            // Don't react if sort values didn't really change
            // kibi: check $scope.searchSource too
            if (!$scope.searchSource || newSort === oldSort) return;
            $scope.sortChanged = true;
            $scope.searchSource.sort(getSort(newSort, $scope.indexPattern));
            $scope.searchSource.fetchQueued();
          });

          $scope.$on('$destroy', function () {
            // kibi: catch query changes
            if (appStateHandlerOff) {
              appStateHandlerOff();
            }
            if ($scope.searchSource) {
              $scope.searchSource.destroy();
            }
          });

          // TODO: we need to have some way to clean up result requests
          $scope.searchSource.onResults().then(function onResults(resp) {
            // kibi: delete the error on searchSource if any
            delete $scope.searchSource.error;

            // Reset infinite scroll limit
            $scope.limit = 50;

            // Abort if something changed
            if ($scope.searchSource !== $scope.searchSource) return;

            // kibi: here detect that there was a change in the query or sorting from the previous request if yes force refresh the hits
            if ($scope.searchSource.history.length > 0 && $scope.searchSource.history[0].fetchParams) {
              const presentQuery = _.cloneDeep($scope.searchSource.history[0].fetchParams.body.query);
              $scope.queryChanged = $scope._oldQuery && !_.isEqual(presentQuery, $scope._oldQuery);
              $scope._oldQuery = presentQuery;
            }

            if (
              $scope.hits &&
              $scope.hits.length > 0 &&
              !$scope.filtersQueryOrTimeChanged &&
              !$scope.sortChanged &&
              !$scope.queryChangeds &&
              !$scope.coatChanged
            ) {
              // kibi: we should check all rows, they can be modified or additional row can be added live
              for (let i = 0; i < resp.hits.hits.length; i++) {
                let found = false;
                for (let j = 0; j < $scope.hits.length; j++) {
                  if ($scope.hits[j]._id === resp.hits.hits[i]._id) {
                    found = true;
                    $scope.hits[j] = resp.hits.hits[i];
                    break;
                  }
                }
                if (!found) {
                  $scope.hits.push(resp.hits.hits[i]);
                }
              }
            } else {
              $scope.hits = resp.hits.hits;
            }

            // kibi: start the page
            let startingPage = 1;
            if ($scope.increaseSample && $scope.pager && ($scope.totalHitCount === resp.hits.total) &&
              !$scope.filtersQueryOrTimeChanged && !$scope.queryChanged && !isPageSizeChanged()
            ) {
              startingPage = $scope.pager.currentPage;
            }

            // We limit the number of returned results, but we want to show the actual number of hits, not
            // just how many we retrieved.
            $scope.totalHitCount = resp.hits.total;
            // kibi: ontology controller listen total hit count and shows it on data tab
            $scope.$emit('totalHitsChanged', $scope.totalHitCount);
            // kibi: use '$scope.pageSize' instead of harcoded number
            $scope.pager = pagerFactory.create($scope.hits.length, $scope.pageSize || parseInt(config.get('discover:sampleSize')),
              startingPage);
            calculateItemsOnPage();

            // Kibi: reset the flag
            $scope.filtersQueryOrTimeChanged = false;
            $scope.sortChanged = false;
            $scope.queryChanged = false;
            $scope.coatChanged = false;

            return $scope.searchSource.onResults().then(onResults);
          }).catch(notify.fatal);

          $scope.searchSource
            // kibi: notify the user what to do if more results cannot be retrieved
            .onError((error) => {
              if (error.message) {
                const matches = error.message.match(/from \+ size must be less than or equal to: \[(\d+)]/);
                if (matches) {
                  const message = `Can't retrieve more than ${matches[1]} results.` +
                    'Please check the index.max_result_window Elasticsearch index setting.';
                  const expError = new Error(message);
                  expError.stack = message;
                  return notify.error(expError);
                }
                // in kibi
                // notify if it is NOT a missing index error
                if (_.get(error, 'resp.error.type') === 'index_not_found_exception') {
                  $scope.searchSource.error =
                    (error.resp.error.reason && error.resp.error['resource.id']) ?
                      error.resp.error.reason + ' ' + error.resp.error['resource.id'] :
                      'Index not found';
                } else {
                  return notify.error(error);
                }
              }
            })
            // kibi: end
            .catch(notify.fatal);
        });
        $scope.$watch('searchSource', refreshTable);
        $scope.$watch('pageSize', function (pageSize) {
          if (pageSize !== undefined) {
            $scope.size = $scope.pageSize;
            $scope.searchSource.size($scope.size);
            courier.fetch();
          }
        });

        $scope.pageOfItems = [];
        $scope.onPageNext = () => {
          const _onPageNext = function () {
            $scope.pager.nextPage();
            calculateItemsOnPage();
          };

          // kibi: fetch more results if possible
          if (!$scope.pager.hasNextPage && $scope.increaseSample) {
            if ($scope.size < $scope.totalHitCount) {
              const newSize = $scope.size * 2;
              if (newSize >= $scope.totalHitCount) {
                $scope.size = $scope.totalHitCount;
              } else {
                $scope.size = newSize;
              }
              $scope.searchSource.from($scope.hits.length);
              $scope.searchSource.size($scope.size);
              return courier.fetch()
                .then(() => {
                  $scope.searchSource.from(0);
                  _onPageNext();
                });
            }
          }
          _onPageNext();
        };

        $scope.onPagePrevious = () => {
          $scope.pager.previousPage();
          calculateItemsOnPage();
        };

        $scope.shouldShowLimitedResultsWarning = () => (
          // kibi: do not show warning if the increaseSample option is enabled
          !$scope.increaseSample && !$scope.pager.hasNextPage && $scope.pager.totalItems < $scope.totalHitCount
        );


        $scope.alignFilterButtons = function () {
          const filterButtons = $element[0].querySelector('#filter-buttons');
          if (!filterButtons) return;
          // kibi: 'get vis-container'
          const parent = $element.parent().parent().parent();
          const right = parent.innerWidth() - 270;
          const fixedHeader = $element[0].querySelector('#fixed-header');
          const relativeHeader = $element[0].querySelector('#relative-header');
          filterButtons.style.left = right + 'px';

          if (fixedHeader.getBoundingClientRect().y > relativeHeader.getBoundingClientRect().y) {
            filterButtons.style.top = fixedHeader.offsetTop + 'px';
          } else {
            filterButtons.style.top = '';
          }
        };

        $scope.$on('visResized', function () {
          $scope.alignFilterButtons();
        });

        $scope.$watch('showCustomView', function () {
          if ($scope.showCustomView && $scope.mfiltersShow) {
            $element[0].querySelector('#filter-buttons').style.display = 'none';
          } else if (!$scope.showCustomView && $scope.mfiltersShow) {
            $element[0].querySelector('#filter-buttons').style.display = '';
            $scope.alignFilterButtons();
          }
        });

        $scope.$watch(() => {
          if (!$scope.searchSource._siren) return;

          const node = coatHelper.findItemByVisIdAndPanelIndex($scope.searchSource._siren.coat.items,
            $scope.searchSource._siren.vis.id,  $scope.searchSource._siren.vis.panelIndex);

          return node ? node.id : null;
        }, function (newVal, oldVal) {
          if (newVal !== oldVal) {
            $scope.coatChanged = true;
          }
        });

        $scope.$on('visScrolled', function (event, visTopOffset) {
          if (!$scope.mfiltersShow) {
            return;
          }
          $scope.alignFilterButtons();
        });

        $scope.mfiltersEnabled = {
          enabled: false
        };
        $scope.$watchCollection('mfiltersEnabled', function (mfiltersEnabled) {
          $scope.mfiltersShow = mfiltersEnabled.enabled;
          $scope.alignFilterButtons();
        });

        $scope.cancelFiltering = function () {
          $element[0].querySelector('#filter-buttons').style.visibility = 'invisible';
          $scope.mfiltersEnabled.enabled = false;
          $scope.mfilters = [];
        };
      }
    };
  });
