import 'ui/filters/short_dots';
import { IndexPatternMissingIndices } from 'ui/errors';
import { IndexPatternProvider } from 'ui/index_patterns/_index_pattern';
import { IndexPatternsPatternCacheProvider } from 'ui/index_patterns/_pattern_cache';
import { IndexPatternsGetProvider } from 'ui/index_patterns/_get';
import { IndexPatternsIntervalsProvider } from 'ui/index_patterns/_intervals';
import { IndexPatternsExcludeIndicesProvider } from 'ui/kibi/index_patterns/_exclude_indices';
import { FieldsFetcherProvider } from './fields_fetcher_provider';
import { RegistryFieldFormatsProvider } from 'ui/registry/field_formats';
import { uiModules } from 'ui/modules';
import { flatten, uniq } from 'lodash';
import { NoAccessToFieldStats } from '../errors';
import { SavedObjectsClientProvider } from 'ui/saved_objects';

const module = uiModules.get('kibana/index_patterns');

export { IndexPatternsApiClientProvider } from './index_patterns_api_client_provider';

export function IndexPatternsProvider(Notifier, Private, config) {
  const self = this;

  const IndexPattern = Private(IndexPatternProvider);
  const patternCache = Private(IndexPatternsPatternCacheProvider);
  const getProvider = Private(IndexPatternsGetProvider);
  const exclusions = Private(IndexPatternsExcludeIndicesProvider);
  const savedObjectsClient = Private(SavedObjectsClientProvider);

  self.get = function (id) {
    if (!id) return self.make();

    const cache = patternCache.get(id);
    return cache || patternCache.set(id, self.make(id));
  };

  self.make = function (id) {
    self.getIds.clearCache();
    return (new IndexPattern(id)).init();
  };

  self.delete = function (pattern) {
    self.getIds.clearCache();
    return pattern.destroy();
  };

  self.errors = {
    MissingIndices: IndexPatternMissingIndices
  };

  self.getTimeFieldOptions = function (indexPattern, nameIsPattern = false) {
    // Checking each index pattern separately so we can ensure they all resolve to actual indices
    const includedIndices = indexPattern.split(',').filter(index => !index.startsWith('-'));
    const excludedIndices = indexPattern.split(',').filter(index => index.startsWith('-'));
    const excludedIndicesSuffix = excludedIndices.length ? ',' + excludedIndices.join(',') : '';

    return Promise.all(includedIndices.map(indexPattern => {
      if (!indexPattern) {
        return [];
      }

      const indexPatternWithExcl = indexPattern.includes('*') ? exclusions.excludeIndices([indexPattern], true).join(',') : indexPattern;

      let promise;
      if (nameIsPattern) {
        promise = self.fieldsFetcher.fetchForTimePattern(indexPatternWithExcl + excludedIndicesSuffix);
      } else {
        promise = self.fieldsFetcher.fetchForWildcard(indexPatternWithExcl + excludedIndicesSuffix);
      }
      return promise
        .catch(err => {
          if (err instanceof IndexPatternMissingIndices) {
            err.failedIndexPattern = indexPattern;
          }
          throw err;
        });
      // kibi: end
    }))
      .then(flatten)
      .then(fields => {
        const dateFields = uniq(
          fields
            .filter(field => field.type === 'date')
            .map(field => field.name)
        );

        if (dateFields.length === 0) {
          return {
            options: [
              {
                display: "The indices which match this index pattern don't contain any time fields."
              }
            ]
          };
        }

        return {
          options: [
            {
              display: "I don't want to use the Time Filter"
            },
            ...dateFields.map(field => ({
              display: field,
              fieldName: field
            }))
          ]
        };
      })
      .catch(err => {
        if (err instanceof IndexPatternMissingIndices) {
          return { error: `Unable to fetch mapping for '${err.failedIndexPattern}'. Do you have indices matching the pattern?` };
        } else if (err instanceof NoAccessToFieldStats) {
          return { error: 'You have no permission to fetch field stats for this index pattern.' };
        } else if (err.status < 500) {
          return { error: err.message };
        }
        // kibi: end
        throw err;
      });
  };

  self.pickDefaultTimeFieldOption = function (timeFieldOptions) {
    // options that represent a time field
    const fieldOptions = timeFieldOptions.filter(option => !!option.fieldName);
    // options like "I don't want the time filter" or "There are no date fields"
    const nonFieldOptions = timeFieldOptions.filter(option => !option.fieldName);

    // if there are multiple field or non-field options then we can't select a default, the user must choose
    if (fieldOptions.length > 1 || nonFieldOptions.length > 1) {
      return;
    } else {
      return fieldOptions[0] || nonFieldOptions[0];
    }
  };

  // kibi: added find method
  self.find = function () {
    return savedObjectsClient.find({
      type: 'index-pattern',
      perPage: 10000
    });
  };

  // kibi: addedd to pass down to the server the index exclusion list
  let excludedIndices = [];
  const indices = config.get('siren:excludedIndices');
  if (indices && indices.length) {
    excludedIndices = indices.split(',');
  }
  self.cache = patternCache;
  self.getIds = getProvider('id', excludedIndices);
  self.getTitles = getProvider('attributes.title');
  self.intervals = Private(IndexPatternsIntervalsProvider);
  self.fieldsFetcher = Private(FieldsFetcherProvider);
  self.fieldFormats = Private(RegistryFieldFormatsProvider);
  self.IndexPattern = IndexPattern;
}

module.service('indexPatterns', Private => Private(IndexPatternsProvider));
