import { uiModules } from 'ui/modules';

import _ from 'lodash';


/*
 * Arrow icon for sorting a list of values, currently supporting only a single sorter
 * at any given time.
 */

uiModules
  .get('kibana')
  .directive('sortIcon', function ($timeout) {
    return {
      restrict: 'AE',
      scope: {
        rows: '=',
        sortBy: '=',
        onSort: '&'
      },

      transclude: true,
      template:
`<span ng-click="onClick()">
  <span ng-transclude />
  <i class="fa" ng-class="sortBy.iconClass()" />
</span>`,

      link($scope, element) {
        $scope.onClick = function onClick() {
          $scope.rows = $scope.sortBy.toggle($scope.rows);

          if ($scope.onSort) { $timeout($scope.onSort); }
          $scope.$emit('sort-icon.click');
        };
      }
    };
  });

function sort(list) {
  const activeSorters = this.activeSorters;
  if (activeSorters.length === 0) { return list; }

  const sorters = _.map(activeSorters, 'sorter');
  const orders = _.map(activeSorters, 'order');

  return _.sortByOrder(list, sorters, orders);
}

export function sortContext(sorters, opts = {}) {
  _.defaults(opts, { controls: {} });

  const sortOrders = ['asc', 'desc', 'none'];
  const iconClasses = { asc: 'fa-sort-up', desc: 'fa-sort-down', none: 'fa-sort' };

  const controls = _.assign(opts.controls, {
    activeSorters: [],
    sort,
    opts
  });

  function setOrder(order, list) {
    const index = _.findIndex(
      this.controls.activeSorters, sorter => sorter.key === this.key);

    if (index >= 0) {
      this.controls.activeSorters.splice(index, 1);
    }

    if (order !== 'none') {
      this.controls.activeSorters.unshift(this);
    }

    this.order = order;
    return list && this.controls.sort(list);
  }

  function toggle(list) {
    const idx = sortOrders.indexOf(this.order);
    return this.setOrder(sortOrders[+!idx], list);
  }

  function iconClass() {
    return iconClasses[(this.controls.activeSorters[0] === this) ? this.order : 'none'];
  }

  return _.mapValues(sorters, (sorter, key) => ({
    key,
    controls,
    sorter: _.iteratee(sorter),
    order: 'none',
    iconClass,
    setOrder,
    toggle
  }));
};

