const DELAY_STRATEGY = {
  RESET_COUNTER_ON_NEW_EVENT: 'RESET_COUNTER_ON_NEW_EVENT',
  DO_NOT_RESET_COUNTER_ON_NEW_EVENT: 'DO_NOT_RESET_COUNTER_ON_NEW_EVENT'
};

const ERRORS = {
  CANCELLED: 'Cancelled by DelayExecutionHelper'
};

/*
 * Triggers the executeCallback callback after delay counted from the first call to
 * addEventData method. After triggering executeCallback the alreadyCollectedData are cleared and
 * helper is ready to accept new calls to addEventData.
 * If a new call is made before the timeout depends on the policy either:
 * - RESET_COUNTER_ON_NEW_EVENT (default)
 *   an internal timer is canceled and a new one with _timeoutTime = delayTime is created
 *
 * - DO_NOT_RESET_COUNTER_ON_NEW_EVENT
 *   an internal timer is canceled and a new one with reduced _timeoutTime = delayTime - "time elapsed from the previous call" is created
 *
 * Constructor
 * DelayExecutionHelper(collectDataCallback, executeCallback, delayTime, delayStrategy)
 * Where
 * - collectDataCallback - callback function to preproces the new data before adding them to internal alreadyCollectedData structure
 *   collectDataCallback(newData, alreadyCollectedData)
 *
 * - executeCallback - callback function executed after delay with all alreadyCollectedData data
 *   executeCallback(alreadyCollectedData)
 *
 * - delayTime - delay time in ms
 * - delayStrategy - strategy of the delay
 *
 * Methods:
 * - addEventData - Adds data to internal *alreadyCollectedData* structure always returns a Promise which:
 *  resolves to a value returned by executeCallback function
 *  rejects with an error thrown from executeCallback function
 *  Where executeCallback can return a Promise or value or throw an error
 *
 * - cancel - cancels current timeout cycle, do not clear the data
 * - destroy - cancel any pending timeout and cleans the data queue object to free resources
 *
 * NOTE:
 * this class should never clone or do any other modification to passed data objects
 * object which are passed in and object retunned must be the same (equal by reference)
 *
 */
export function DelayExecutionHelperFactory(Promise, $timeout) {
  class DelayExecutionHelper {
    static get DELAY_STRATEGY() {
      return DELAY_STRATEGY;
    }

    static get ERRORS() {
      return ERRORS;
    }

    constructor(collectDataCallback, executeCallback, delayTime, delayStrategy = DELAY_STRATEGY.RESET_COUNTER_ON_NEW_EVENT) {
      this.collectDataCallback = collectDataCallback;
      this.executeCallback = executeCallback;
      this.delayTime = delayTime;
      this.delayStrategy = delayStrategy;

      this._data = {};
      this._timeout;
      this._timeoutTime = delayTime;

      this._dataQueue = [{}];
      this._dataQueueIndex = 0;
    }

    _getNow() {
      const d = new Date();
      return d.getTime();
    }

    _getData() {
      return this._dataQueue[this._dataQueueIndex];
    }

    _advanceQueue() {
      this._dataQueue.push({});
      this._dataQueueIndex++;
    }

    addEventData(data) {
      // cancel any previous timeout
      if (this._timeout) {
        this.cancel();
        if (this.delayStrategy === DELAY_STRATEGY.RESET_COUNTER_ON_NEW_EVENT) {
          this._timeoutTime = this.delayTime;
        } else {
          //DO_NOT_RESET_COUNTER_ON_NEW_EVENT
          this._timeoutTime = this.delayTime - (this._getNow() - this._startTime);
        }
      }
      this.collectDataCallback(data, this._getData());
      this._startTime = this._getNow();

      return new Promise((fulfill, reject) => {

        this._timeout = $timeout(() => {
          try {
            const ret = this.executeCallback(this._getData());
            this._advanceQueue();
            fulfill(ret);
          } catch (e) {
            reject(e);
          }
        }, this._timeoutTime);

        this._timeout.catch(err => {
          if (err === 'canceled') {
            reject(new Error(ERRORS.CANCELLED)); // wrap to always return a proper error object
          } else {
            reject(err);
          }
        });

      });
    }

    cancel() {
      if (this._timeout) {
        $timeout.cancel(this._timeout);
        this._timeout = null;
      }
    }

    destroy() {
      this.cancel();
      this._dataQueue = [{}];
      this._dataQueueIndex = 0;
    }

  }

  return DelayExecutionHelper;
};
