import _ from 'lodash';
import uniqFilters from 'ui/filter_bar/lib/uniq_filters';
import { DecorateQueryProvider } from 'ui/courier/data_source/_decorate_query';
import { migrateFilter } from 'ui/courier/data_source/_migrate_filter';


/**
* Create a filter that can be reversed for filters with negate set
* @param {boolean} reverse This will reverse the filter. If true then
*                          anything where negate is set will come
*                          through otherwise it will filter out
* @returns {function}
*/
const filterNegate = function (reverse) {
  return function (filter) {
    if (_.isUndefined(filter.meta) || _.isUndefined(filter.meta.negate)) return !reverse;
    return filter.meta && filter.meta.negate === reverse;
  };
};

/**
* Translate a filter into a query to support es 3+
* @param  {Object} filter - The filter to translate
* @return {Object} the query version of that filter
*/
const translateToQuery = function (filter) {
  if (!filter) return;

  if (filter.query) {
    return filter.query;
  }

  if (filter.range) {
    return {
      range: filter.range
    };
  }

  return filter;
};

/**
* Clean out any invalid attributes from the filters
* @param {object} filter The filter to clean
* @returns {object}
*/
const cleanFilter = function (filter) {
  return _.omit(filter, ['meta', '$state']);
};

export function QueryBuilderFactory(Private) {
  const decorateQuery = Private(DecorateQueryProvider);

  /*
  * The parameter savedSearch should be a reference to a SavedSearch
    * instance, not a SavedSearch id
    */
  return function queryBuilder(filters = [], queries, time) {
    const request = {};

    // logic copied from src/ui/public/courier/data_source/_abstract.js
    _.each(filters, function (filter) {
      if (filter.query) {
        decorateQuery(filter.query);
      }
    });

    request.query = {
      bool: {
        must: (
          filters
            .filter(filterNegate(false))
            .map(translateToQuery)
            .map(cleanFilter)
            .map(migrateFilter)
        ),
        must_not: (
          filters
            .filter(filterNegate(true))
            .map(translateToQuery)
            .map(cleanFilter)
            .map(migrateFilter)
        )
      }
    };

    _.each(queries, (q) => {
      decorateQuery(q);
      request.query.bool.must.push(q);
    });

    // add time
    if (time) {
      request.query.bool.must.push(time);
    }
    return request;
  };
};
