import html2canvas from 'html2canvas';
import domtoimage from 'dom-to-image';
import canvg from 'canvg';
import d3 from 'd3';

export class SirenExportVisualizationHelper {
  constructor(vis, $element) {
    this.$element = $element;
    this.vis = vis;

    // some visualizations can be used only as control elements
    // and there is no reason to handle it as valid to export visualization
    const visualizationTypes = {
      vectorVis: [
        'pie',
        'line',
        'histogram',
        'area',
        'bubble_diagram_vis',
        'heatmap'
      ],
      vectorVisWithExtraContent: [
        'scatterplot_vis',
        'horizontal_bar_chart_vis',
        'goal',
        'gauge',
        'metric',
        'tagcloud',
      ],
      htmlVis: [
        // tables currently is not included in visualization export
        // as it can be exported by CSV/JSON method
        // 'table',
        // 'kibi-data-table',
        'markdown'
      ],
      multiVis: [
        'multi_chart_vis'
      ],
      graphVis: [
        'graph_browser_vis'
      ],
      canvasVis: [
        'timelion'
      ],
      mapVis: [
        'kibi_timeline',
        'region_map',
        'tile_map',
        'topic_clustering_vis',
        'parallel_lines_chart_vis',
        'enhanced_tilemap'
      ]
    };

    this.validToExport = () => {
      // for now - do not add export if MS Edge
      // TO-DO Remove when MS Edge will use Chromium - issue #8400
      if (window.navigator.userAgent.indexOf('Edge') !== -1) {
        return false;
      };

      if (this.$element.find('div#no-results-found-box')[0]) {
        return false;
      };

      const type = this.vis && this.vis.type ? this.vis.type.name : undefined;
      const supportedVis = visualizationTypes.vectorVis.concat(
        visualizationTypes.vectorVisWithExtraContent,
        visualizationTypes.htmlVis,
        visualizationTypes.multiVis,
        visualizationTypes.graphVis,
        visualizationTypes.canvasVis,
        visualizationTypes.mapVis
      );

      return supportedVis.indexOf(type) !== -1 ? true : false;
    };

    this.getVis = (passedOptions) => {
      // TO-DO use options as selector
      // select the method of rendering based on visualization type
      const options = passedOptions ? passedOptions : {};
      const exportOptions = {
        method: options.method || this.methods.DOWNLOAD_AS_PNG,
        cutByPanel: options.cutByPanel || false,
        src: options.src,
        title: options.title !== undefined ? options.title : true
      };
      if (visualizationTypes.vectorVis.indexOf(this.vis.type.name) !== -1) {
        return this.getVectorVis(this.vis, this.$element, exportOptions);
      } else if (visualizationTypes.vectorVisWithExtraContent.indexOf(this.vis.type.name) !== -1) {
        return this.getVectorVisWithExtraContent(this.vis, this.$element, exportOptions);
      } else if (visualizationTypes.multiVis.indexOf(this.vis.type.name) !== -1) {
        return this.getMultiVis(this.vis, this.$element, exportOptions);
      } else if (visualizationTypes.mapVis.indexOf(this.vis.type.name) !== -1) {
        return this.getRenderedDomWithVis(this.vis, this.$element.find('.vis-container').eq(0), exportOptions);
      } else if (visualizationTypes.graphVis.indexOf(this.vis.type.name) !== -1) {
        return this.getGraphVis(this.vis, exportOptions);
      } else if (visualizationTypes.canvasVis.indexOf(this.vis.type.name) !== -1) {
        return this.getCanvasVis(this.vis, this.$element, exportOptions);
      } else {
        return this.getHtmlVis(this.vis, this.$element.find('.vis-container').eq(0), exportOptions);
      }
    };
  };

  methods = {
    CONVERT_TO_PNG: 'convertToPng',
    DOWNLOAD_AS_PNG: 'downloadAsPng'
  };

  prepareContainer(vis) {
    const container = document.createElement('div');
    container.innerHTML = '<div style="margin-bottom: 10px">' + vis.title + '</div>';
    return container;
  };

  vectorizeVis($element, search) {
    // convert d3 to regular SVG
    const $svg = $element.find(search);
    const html = d3.select($svg[0])
      .attr('xmlns', 'http://www.w3.org/2000/svg')
      .node();
    if (html) {
      const svgParentHtml = html.parentNode.innerHTML;
      // remove extra content which can block rendering if it exists
      const svgHtml = svgParentHtml.substring(svgParentHtml.indexOf('<svg'), svgParentHtml.indexOf('</svg>') + 6);
      const img = document.createElement('img');
      img.src = 'data:image/svg+xml;base64,' + btoa(svgHtml);
      img.style.width = $svg.width() + 'px';
      img.style.height = $svg.height() + 'px';
      return img;
    } else {
      return null;
    };
  };

  restoreStylesOnClonedVisualization($cloned, $original) {
    // some styles can be lost in conversion process
    // apply styles from original visualization
    const $originalDomains = $original.find('.domain');
    const $clonedDomains = $cloned.find('.domain');
    $clonedDomains.each((domainIndex) => {
      const $originalElement = $originalDomains.eq(domainIndex);
      $clonedDomains.eq(domainIndex)
        .css('fill', $originalElement.css('fill'))
        .css('stroke', $originalElement.css('stroke'));
    });
    const $originalTexts = $original.find('text');
    const $clonedTexts = $cloned.find('text');
    $clonedTexts.each((textIndex) => {
      const $originalElement = $originalTexts.eq(textIndex);
      $clonedTexts.eq(textIndex)
        .css('fill', $originalElement.css('fill'))
        .css('stroke', $originalElement.css('stroke'))
        .css('font-size', $originalElement.css('font-size'))
        .css('font-weight', $originalElement.css('font-weight'));
    });
    const $originalRects = $original.find('rect');
    const $clonedRects = $cloned.find('rect');
    $clonedRects.each((rectIndex) => {
      const $originalElement = $originalRects.eq(rectIndex);
      $clonedRects.eq(rectIndex)
        .css('fill', $originalElement.css('fill'))
        .css('stroke', $originalElement.css('stroke'));
    });
  };

  prepareAxisToProcessing($svgAxis, appliedPosition) {
    // if visualization has the axes we should prepare them and convert to canvas
    // because current version of html2canvas can't inherit CSS rules for the basic SVG elements
    // 'appliedPosition' defines additional CSS rules for Axis
    const canvasForSvg = document.createElement('canvas');
    canvasForSvg.setAttribute('width', $svgAxis.width() + 'px');
    canvasForSvg.setAttribute('height', $svgAxis.height() + 'px');
    if (appliedPosition === 'left' || appliedPosition === 'right') {
      canvasForSvg.style.display = 'block';
      canvasForSvg.style.float = 'left';
    };
    const $clone = $svgAxis.clone();
    this.restoreStylesOnClonedVisualization($clone, $svgAxis);
    if ($clone.children().eq(0).prop('tagName') !== 'text') {
      // it is an Axis's line SVG box
      // move the first and the last tick to prevent cropping if it can happen
      const FIRST_CROPPED_TICK = appliedPosition === 'left' || appliedPosition === 'right' ?
        'translate(0,' + $svgAxis.height() + ')' : 'translate(' + $svgAxis.width() + ',0)';
      const LAST_CROPPED_TICK = 'translate(0,0)';
      const $allTicks = $clone.children().find('.tick');
      if ($allTicks.eq(0).attr('transform') === FIRST_CROPPED_TICK) {
        const firstCroppedElementText = $allTicks.eq(0).find('text');
        if (appliedPosition === 'left' || appliedPosition === 'right') {
          firstCroppedElementText.attr('y', -5);
        } else if (appliedPosition === 'top' || appliedPosition === 'bottom') {
          firstCroppedElementText.attr('x', 5);
        }
      };
      if ($allTicks.eq($allTicks.length - 1).attr('transform') === LAST_CROPPED_TICK) {
        const $lastCroppedElementText = $allTicks.eq($allTicks.length - 1).find('text');
        if (appliedPosition === 'left' || appliedPosition === 'right') {
          $lastCroppedElementText.attr('y', 5);
        } else if (appliedPosition === 'top' || appliedPosition === 'bottom') {
          $lastCroppedElementText.attr('x', -5);
        }
      };
    };
    canvg(canvasForSvg, '<svg>' + $clone.html() + '</svg>');
    return canvasForSvg;
  };

  wrapLegend($element, search) {
    // return the legend if the legend exists
    // and define the floating of the legend
    const $legendElement = $element.find(search).eq(0);

    let legendFloatted = false;

    if ($legendElement.width() <= $legendElement.height()) {
      legendFloatted = true;
    };

    return {
      legendElement: $legendElement,
      legendFloatted: legendFloatted
    };
  };

  executeDownloadingAsLink(vis, canvas) {
    const dataURL = canvas
      .toDataURL('image/png')
      .replace('image/png', 'image/octet-stream');
    const link = document.createElement('a');
    link.href = dataURL;
    link.download = vis.title ? vis.title + '.png' : 'exported-visualization.png';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  getVisualizationAsPng(vis, canvas) {
    const src = canvas
      .toDataURL('image/png')
      .replace('image/png', 'image/octet-stream');
    return {
      title: vis.title,
      imgSrc: src,
      width: canvas.width,
      height: canvas.height
    };
  };

  processExportedVisualization(vis, element, method) {
    // temporary append the final rendering of element to prepare it for conversion
    // and hide it by CSS
    document.body.appendChild(element);
    element.style.zIndex = -1;
    element.style.position = 'fixed';
    element.style.background = '#ffffff';
    element.style.padding = '20px';
    element.style.width = 'fit-content';
    return html2canvas(element, { logging: false }).then((canvas) => {
      // remove the final rendering after conversion and simulate user click
      // to download rendered graphics
      document.body.removeChild(element);
      if (method === this.methods.CONVERT_TO_PNG) {
        return this.getVisualizationAsPng(vis, canvas);
      } else {
        return this.executeDownloadingAsLink(vis, canvas);
      };
    });
  };

  validVectorSelector($element) {
    // find valid existed selector
    // to correctly grab SVG element
    let validSelector;
    const availableSelectors = [
      '.svg svg',
      '.chart svg',
      '.visualize-chart svg',
      'svg'
    ];
    for (let i = 0; i < availableSelectors.length; i++) {
      if ($element.find(availableSelectors[i])[0]) {
        validSelector = availableSelectors[i];
        break;
      };
    };
    return validSelector;
  };

  // build each element separately if visualization is based on SVG
  // to grab just elements that we need
  getVectorVis(vis, $element, options) {
    // for visualizations with more than one SVG elements
    const container = options.title ? this.prepareContainer(vis) : document.createElement('div');
    const visualizationWrap = document.createElement('div');
    const visualization = document.createElement('div');
    visualization.style.float = 'left';
    const legendContent = this.wrapLegend($element, '.legend-ul');
    const $legendElement = legendContent.legendElement;
    const search = this.validVectorSelector($element);
    const vectorizedVis = this.vectorizeVis($element, search);
    if (vectorizedVis) {
      visualization.appendChild(vectorizedVis);
    };
    let visualizationWrapWidth = 0;
    // add xAxes if it is exists
    const $xInfo = $element.find('.x-axis-div-wrapper');
    if ($xInfo[0]) {
      const $svgAxes = $xInfo.find('svg');
      $svgAxes.each(svgIndex => {
        visualization.appendChild(this.prepareAxisToProcessing($svgAxes.eq(svgIndex), 'bottom'));
      });
      visualizationWrapWidth += $xInfo.width();
    };
    // add left yAxis if it is exists
    const $yInfo = $element.find('.y-axis-div-wrapper');
    if ($yInfo.eq(0)[0]) {
      const $svgAxes = $yInfo.eq(0).find('svg');
      $svgAxes.each(svgIndex => {
        visualizationWrap.appendChild(this.prepareAxisToProcessing($svgAxes.eq(svgIndex), 'left'));
      });
      visualizationWrapWidth += $yInfo.eq(0).width();
    };
    visualizationWrap.appendChild(visualization);
    // add right yAxis if it is exists
    if ($yInfo.eq(1)[0]) {
      const $svgAxes = $yInfo.eq(1).find('svg');
      $svgAxes.each(svgIndex => {
        visualizationWrap.appendChild(this.prepareAxisToProcessing($svgAxes.eq(svgIndex), 'right'));
      });
      visualizationWrapWidth += $yInfo.eq(1).width();
    };
    // add custom width if axes exist
    if (visualizationWrapWidth > 0) {
      visualizationWrap.style.width = visualizationWrapWidth + 'px';
    };
    container.appendChild(visualizationWrap);
    if ($legendElement && vectorizedVis) {
      if (legendContent.legendFloatted) {
        $legendElement.css('float', 'left');
        visualizationWrap.style.float = 'left';
      } else {
        $legendElement.css('width', vectorizedVis.style.width);
        const $legendElements = $legendElement.find('li');
        $legendElements.each((liIndex) => {
          $legendElements.eq(liIndex)
            .css('display', 'inline-block');
        });
      };
      visualization.style.width = $element.find(search).eq(0).width() + 'px';
      $legendElement.css('list-style-type', 'none');
      $legendElement.clone().attr('id', 'exported-legend').appendTo(container);
    };

    return this.processExportedVisualization(vis, container, options.method);
  };

  getVectorVisWithExtraContent(vis, $element, options) {
    // for visualizations with one SVG element
    const container = options.title ? this.prepareContainer(vis) : document.createElement('div');
    const validSelector = this.validVectorSelector($element);
    const $visualizationWraps = $element.find(validSelector);
    $visualizationWraps.each(visWrapIndex => {
      const canvasForSvg = document.createElement('canvas');
      canvasForSvg.setAttribute('width', $visualizationWraps.eq(visWrapIndex).width() + 'px');
      canvasForSvg.setAttribute('height', $visualizationWraps.eq(visWrapIndex).height() + 'px');
      const $visualizationWrapClone = $visualizationWraps.eq(visWrapIndex).clone();
      this.restoreStylesOnClonedVisualization($visualizationWrapClone, $visualizationWraps);
      canvg(canvasForSvg, '<svg>' + $visualizationWrapClone.html() + '</svg>');
      container.appendChild(canvasForSvg);
      const legendContent = this.wrapLegend($element, '.legend-ul');
      const $legendElement = legendContent.legendElement;
      if ($legendElement) {
        if (legendContent.legendFloatted) {
          canvasForSvg.style.float = 'left';
        };
        if (visWrapIndex === $visualizationWraps.length - 1) {
          if (legendContent.legendFloatted) {
            $legendElement.css('float', 'left');
          } else {
            $legendElement.css('width', $visualizationWraps.eq(visWrapIndex).width() + 'px');
            const $legendElements = $legendElement.find('li');
            $legendElements.each((liIndex) => {
              $legendElements.eq(liIndex)
                .css('display', 'inline-block');
            });
          };
          canvasForSvg.style.width = $element.find(validSelector).eq(0).width() + 'px';
          $legendElement.css('list-style-type', 'none');
          $legendElement.clone().attr('id', 'exported-legend').appendTo(container);
        };
      };
    });

    return this.processExportedVisualization(vis, container, options.method);
  };

  getHtmlVis(vis, $element, options) {
    const container = options.title ? this.prepareContainer(vis) : document.createElement('div');
    const $visualizationWrap = $element;
    const $visualizationWrapClone = $visualizationWrap.clone();
    $visualizationWrapClone.css('height', $visualizationWrap.height() + 'px');
    $visualizationWrapClone.css('width', $visualizationWrap.width() + 'px');
    $visualizationWrapClone.appendTo(container);

    return this.processExportedVisualization(vis, container, options.method);
  };

  getMultiVis(vis, $element, options) {
    if ($element.find('.svg svg')[0] || $element.find('.chart svg')[0]) {
      return this.getVectorVis(vis, $element, options);
    } else {
      return this.getHtmlVis(vis, $element.find('.visualize-chart .content').eq(0), options);
    };
  };

  getRenderedDomWithVis(vis, $element, options) {
    const container = options.title ? this.prepareContainer(vis) : document.createElement('div');
    const $mapLegend = $element.find('.leaflet-bar');
    if ($mapLegend[0]) {
      $mapLegend.css('display', 'none');
    };
    return domtoimage.toPng($element[0]).then((dataUrl) => {
      if ($mapLegend[0]) {
        $mapLegend.css('display', '');
      };
      const imgContainer = document.createElement('div');
      imgContainer.style.overflow = 'hidden';
      imgContainer.style.width = $element.width() + 'px';
      imgContainer.style.height = $element.height() + 'px';
      const img = document.createElement('img');
      img.src = dataUrl;
      imgContainer.appendChild(img);
      container.appendChild(imgContainer);

      return this.processExportedVisualization(vis, container, options.method);
    });
  };

  getGraphVis(vis, options) {
    const container = this.prepareContainer(vis);
    const imgContainer = document.createElement('div');
    const img = document.createElement('img');
    img.src = options.src;
    imgContainer.appendChild(img);
    container.appendChild(imgContainer);

    return this.processExportedVisualization(vis, container, options.method);
  };

  getCanvasVis(vis, $element, options) {
    const container = options.title ? this.prepareContainer(vis) : document.createElement('div');
    const $visualizationCanvas = $element.find('canvas');
    if ($visualizationCanvas[0]) {
      const dataURL = $visualizationCanvas[0]
        .toDataURL('image/png')
        .replace('image/png', 'image/octet-stream');
      const img = document.createElement('img');
      img.src = dataURL;
      img.style.height = $visualizationCanvas.height() + 'px';
      img.style.width = $visualizationCanvas.width() + 'px';
      const imgContainer = document.createElement('div');
      imgContainer.style.position = 'relative';
      imgContainer.appendChild(img);
      const $axis = $element.find('.flot-text');
      if ($axis[0]) {
        $axis.clone().appendTo(imgContainer);
      };
      container.appendChild(imgContainer);
    };

    return this.processExportedVisualization(vis, container, options.method);
  };
};
