
/*eslint import/namespace: 0*/
import * as cc from './chartcommons';
import { valueFilter, rangeFilter } from './renderdata';
import { arrayMoveValue, truncateTextAtWidth } from './utils';
import * as d3 from './d3';
import _ from 'lodash';

import '../styles/parallel_chart.less';


// Consts

const paddingY = 0;
const textLineHeight = 24;
const fontSize = textLineHeight / 2;
const fontUpShift = .75 * fontSize;
const topToolbarHeight = 2 * textLineHeight;
const botToolbarHeight = textLineHeight;
const topToolbarTextY = -1.25 * textLineHeight;
const padTop = paddingY + topToolbarHeight;
const padBottom = paddingY + botToolbarHeight;

const splineCP = .5;
const borderAxisSplit = 2;

const mouseFilterTolerance = 10;
const filterRangeWidth = 20;
const filterRangeWidthHalf = filterRangeWidth / 2;
const filterRangeHandlesHeight = 5;
const filterValueRadius = 5;
const filterIconsPad = 15;

const labelPositionsByFilter = {
  none: { top: [fontUpShift], bottom: [fontUpShift] },
  range: { top: [fontUpShift], bottom: [fontUpShift] },
  value: { top: [fontUpShift + filterValueRadius], bottom: [] }
};

const sortIcons = { none: '\uf0dc', asc: '\uf0de', desc: '\uf0dd' };


// Local Functions

// Geometry

function xToColumnIndex(ctx, x) {
  return Math.min(Math.max(
    Math.floor(ctx.xScale.invert(x)),
    0), ctx.columns.length - 2);
}

function xToColumnPos(ctx, c, x) {
  let xScale = ctx.xScale;

  let x0 = xScale(c);
  let x1 = xScale(c + 1);

  return (x - x0) / (x1 - x0);
}

function xToClosestColumnIndex(ctx, x) {
  const c = Math.floor(ctx.xScale.invert(x));

  return Math.min(Math.max(
    xToColumnPos(ctx, c, x) <= .5 ? c : c + 1,
    0), ctx.columns.length - 1);
}

function valueToY(ctx, val) {
  // Y scales inverted due to svg y axis top->bot
  return ctx.refSizeY * (1 - val);
}

function yToValue(ctx, y) {
  return 1 - y / ctx.refSizeY;
}


function cubic(t, v0, v1) {
  /* eslint-disable  camelcase */
  const one_t = 1 - t;
  const v0_1 = one_t * v0 + t * v1;

  return one_t * (one_t * v0 + t * v0_1) + t * (one_t * v0_1 + t * v1);
}

function cubicX(t) {
  /* eslint-disable  camelcase, space-unary-ops */
  let one_t = 1 - t;

  let c1 = splineCP;
  let c2 = 1 - splineCP;

  let mid = (one_t * c1 + t * c2);

  return (one_t * (one_t * (/*one_t * 0*/ + t * c1) + t * mid)
    + t * (one_t * mid + t * (one_t * c2 + t /** 1*/)));
  /* eslint-enable  camelcase, space-unary-ops */
}

function cubicXDerivative(t) {
  /* eslint-disable  camelcase */
  let one_t = 1 - t;

  let c1 = splineCP;
  let c2 = 1 - splineCP;
  let c2_c1 = c2 - c1;

  return 3 * (one_t * (one_t * (c1 /*- 0*/) + t * c2_c1) +
    t * (one_t * c2_c1 + t * (1 - c2)));
  /* eslint-enable  camelcase */
}

function newtonCubicX(x, t) {
  return t - (cubicX(t) - x) / cubicXDerivative(t);
}

function closestSpline(ctx, posX, posY) {
  const c = xToColumnIndex(ctx, posX);

  let x = xToColumnPos(ctx, c, posX);
  let t = x;

  t = newtonCubicX(x, t);
  t = newtonCubicX(x, t);
  t = newtonCubicX(x, t);

  /* eslint-disable  camelcase */
  const one_t = 1 - t;

  const yScale0 = ctx.yScales[c];
  const yScale1 = ctx.yScales[c + 1];

  const l0 = ctx.visibleToAll[c];
  const l1 = ctx.visibleToAll[c + 1];

  let min = Infinity;
  const line = _.reduce(ctx.filtered.records[0], function (memo, line) {
    const v0 = line[l0];
    const v1 = line[l1];

    if (v0 === null || v1 === null) { return memo; }

    const y0 = yScale0(v0);
    const y1 = yScale1(v1);

    const y0_1 = one_t * y0 + t * y1;
    const y = one_t * (one_t * y0 + t * y0_1) + t * (one_t * y0_1 + t * y1);

    const val =  Math.abs(y - posY);
    return (val <= min) ? (min = val, line) : memo;
  }, null);

  return line;
  /* eslint-enable  camelcase */
}

function xToLineY(ctx, line, posX) {
  const c = xToColumnIndex(ctx, posX);

  let x = xToColumnPos(ctx, c, posX);
  let t = x;

  t = newtonCubicX(x, t);
  t = newtonCubicX(x, t);
  t = newtonCubicX(x, t);

  const yScale0 = ctx.yScales[c];
  const yScale1 = ctx.yScales[c + 1];

  const l0 = ctx.visibleToAll[c];
  const l1 = ctx.visibleToAll[c + 1];

  return cubic(t, yScale0(line[l0]), yScale1(line[l1]));
}


// Interaction

function updateHover(ctx) {
  let mPos = cc.rawMousePos(ctx, d3.event);
  if (!cc.rectContains(ctx.hoverRect, mPos)) { return cc.resetHover(ctx); }

  mPos = cc.movedInsideRect(ctx.contentRect, mPos);

  const { hover } = ctx.view;

  const c = xToClosestColumnIndex(ctx, mPos[0]);
  const line = closestSpline(ctx, mPos[0], mPos[1]);

  if (c === hover.c && line === hover.line) { return; }

  hover.c = c;
  hover.line = line;
  hover.event = d3.event;

  ctx.viewChanges.hover = true;

  cc.refresh(ctx);
}


function filterDefineFilter(ctx) {
  const mPos = d3.mouse(ctx.svg);

  if (d3.event.button || !cc.rectContains(ctx.hoverRect, mPos)) {
    return false;
  }

  const c = xToClosestColumnIndex(ctx, mPos[0]);
  const dx = mPos[0] - ctx.xScale(c);

  return Math.abs(dx) < filterRangeWidthHalf;
}

function filterDefineStart(ctx) {
  const mPos = [d3.event.x, d3.event.y];

  ctx.filterDrag = {
    mPos,
    c: xToClosestColumnIndex(ctx, mPos[0]),
    v: yToValue(ctx, mPos[1])
  };

  toggleValueFilter(ctx);
}

function toggleValueFilter(ctx) {
  const hoverLine = ctx.view.hover.line;
  if (!hoverLine) { return; }

  ctx.viewChanges.colFilters = true;

  const c = ctx.filterDrag.c;
  const colId = ctx.columns[c].id;

  const filter = ctx.view.colFilters[colId];
  if (filter && filter.type === 'value') {
    delete ctx.view.colFilters[colId];
    return cc.refresh(ctx);
  }

  const l = ctx.visibleToAll[c];

  ctx.view.colFilters[colId] = valueFilter(ctx, colId, hoverLine[l]);

  cc.resetHover(ctx);
}

function filterDefineMove(ctx) {
  const mPos = [d3.event.x, d3.event.y];

  const c = ctx.filterDrag.c;
  const colId = ctx.columns[c].id;
  const filter = ctx.view.colFilters[colId];

  if (!(filter && filter.type === 'range') &&
      Math.abs(mPos[1] - ctx.filterDrag.mPos[1]) < mouseFilterTolerance) {

    return;
  }

  const v = yToValue(ctx, mPos[1]);
  const v0 = Math.min(v, ctx.filterDrag.v);
  const v1 = Math.max(v, ctx.filterDrag.v);

  ctx.view.colFilters[colId] = rangeFilter(ctx, colId, v0, v1);
  ctx.viewChanges.colFilters = true;

  cc.refresh(ctx);
}

function filterRemove(ctx, c) {
  cc.preventClick(d3.select(ctx.svg));

  delete ctx.view.colFilters[ctx.columns[c].id];

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}

function filterDragStart(ctx, c) {
  const filter = ctx.view.colFilters[ctx.columns[c].id];

  ctx.filterDrag = {
    c,
    v0: filter.v0,
    v1: filter.v1,
    vPress: yToValue(ctx, d3.event.y)
  };

  cc.resetHover(ctx);
  cc.refresh(ctx);
}

function filterDragMove(ctx) {
  const { c, v0, v1, vPress } = ctx.filterDrag;
  const drag = yToValue(ctx, d3.event.y) - vPress;
  const colId = ctx.columns[c].id;

  ctx.view.colFilters[colId] = rangeFilter(ctx, colId, v0 + drag, v1 + drag);
  ctx.viewChanges.colFilters = true;

  cc.refresh(ctx);
}

function filterTopStart(ctx, c) {
  const filter = ctx.view.colFilters[ctx.columns[c].id];

  ctx.filterDrag = {
    c,
    v0: filter.v0,
    v1: filter.v1,
    vPress: yToValue(ctx, d3.event.y)
  };

  cc.resetHover(ctx);
}

function filterTopMove(ctx) {
  const { c, v0, v1, vPress } = ctx.filterDrag;
  const drag = yToValue(ctx, d3.event.y) - vPress;
  const colId = ctx.columns[c].id;

  ctx.view.colFilters[colId] = rangeFilter(
      ctx, colId, v0, Math.max(v0, v1 + drag));

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}


function filterBotStart(ctx, c) {
  const filter = ctx.view.colFilters[ctx.columns[c].id];

  ctx.filterDrag = {
    c,
    v0: filter.v0,
    v1: filter.v1,
    vPress: yToValue(ctx, d3.event.y)
  };

  cc.resetHover(ctx);
}

function filterBotMove(ctx) {
  const { c, v0, v1, vPress } = ctx.filterDrag;
  const drag = yToValue(ctx, d3.event.y) - vPress;
  const colId = ctx.columns[c].id;

  ctx.view.colFilters[colId] = rangeFilter(
    ctx, colId, Math.min(v0 + drag, v1), v1);

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}


function colDragFilter(ctx) {
  return (d3.event.button === 0) && !ctx.colDrag;
}

function colDragStart(ctx, c) {
  ctx.colDrag = { dstIdx: c, columnsMap: _.range(ctx.columns.length) };
  renderOverlayGroup(ctx);
}

function colDragMove(ctx, c) {
  const { columns } = ctx;

  let dstIdx = xToClosestColumnIndex(ctx, d3.event.x);
  if (dstIdx === ctx.colDrag.dstIdx) { return; }

  const columnsMap = _.range(columns.length);
  arrayMoveValue(columnsMap, c, dstIdx);

  ctx.colDrag.dstIdx = dstIdx;
  ctx.colDrag.columnsMap = columnsMap;

  renderOverlayGroup(ctx);
}

function colDragEnd(ctx, c) {
  let dstIdx = ctx.colDrag.dstIdx;

  if (dstIdx === c) {
    ctx.colDrag = null;
    return renderOverlayGroup(ctx);
  }

  ctx.onMoveColumn(c, dstIdx);
}

function colHeadTipShow(ctx, c) {
  const { viewRect, viewBox } = ctx;
  const group = d3.select(ctx.svg).select('.plg-overlay');

  const tipBox = cc.materialize(group, 'foreignObject', 'plg-column-tip-box', [0]).enter;
  if (tipBox.empty()) { return; }

  tipBox
    .attr('y', 0).attr('height', ctx.refSizeY)
    .attr('x', viewBox[0]).attr('width', ctx.viewBox[2])
    .attr('style', 'visibility: hidden')
    .html(`<div class="plg-column-tip">${ctx.onColumnTipHtmlRequest(c)}</div>`);

  _.defer(function adjustXGeometry() {
    const tipWidth = tipBox.select('.plg-column-tip').node()
      .getBoundingClientRect().width;
    const idealPos = c * ctx.xStep - tipWidth / 2;

    tipBox
      .attr('x', Math.min(Math.max(idealPos, viewRect[0]), viewRect[2]))
      .attr('width', Math.min(tipWidth, viewBox[2]))
      .attr('style', null);
  });
}

function colHeadTipHide(ctx, c) {
  ctx.colHeadTipShow.cancel();
  d3.select(ctx.svg).selectAll('.plg-column-tip-box').remove();
}

function initMouse(ctx) {
  ctx.filterDefineEnd = () => cc.filterDefineEnd(ctx);

  ctx.filterDragStart = c => filterDragStart(ctx, c);
  ctx.filterTopStart = c => filterTopStart(ctx, c);
  ctx.filterBotStart = c => filterBotStart(ctx, c);

  ctx.filterDragMove = () => filterDragMove(ctx);
  ctx.filterTopMove = () => filterTopMove(ctx);
  ctx.filterBotMove = () => filterBotMove(ctx);

  ctx.colDragFilter = c => colDragFilter(ctx, c);
  ctx.colDragStart = c => colDragStart(ctx, c);
  ctx.colDragMove = c => colDragMove(ctx, c);
  ctx.colDragEnd = c => colDragEnd(ctx, c);

  ctx.colHeadTipShow = _.debounce(c => colHeadTipShow(ctx, c), 200);
  ctx.colHeadTipHide = c => colHeadTipHide(ctx, c);

  d3.select(ctx.svg)
    .on('mousemove.plg', () => updateHover(ctx)) // eslint-disable-line memoryleaks
    .on('mouseleave.plg', () => cc.resetHover(ctx)) // eslint-disable-line memoryleaks
    .call(d3.drag()
      .container(ctx.svg)
      .filter(() => filterDefineFilter(ctx)) // eslint-disable-line memoryleaks
      .on('start.plg', () => filterDefineStart(ctx)) // eslint-disable-line memoryleaks
      .on('drag.plg', () => filterDefineMove(ctx)) // eslint-disable-line memoryleaks
      .on('end.plg', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks
}


// Rendering

function renderDefsGroup(ctx) {
  const { highlights } = ctx;
  const svg = d3.select(ctx.svg);

  const filtersGroup = cc.materialize(svg, 'defs', 'plg-defs', [0]).all;

  cc.materialize(filtersGroup, 'filter', '', highlights).all
    .each(function (hl) {
      const tipColor = hl.colors.tip || 'black';

      const filter = d3.select(this)
        .attr('id', hl.backgroundId);

      cc.materialize(filter, 'feFlood', '', [0]).all
        .attr('flood-color', tipColor)
        .attr('flood-opacity', 0.5);

      cc.materialize(filter, 'feComposite', '', [0]).enter
        .attr('operator', 'over')
        .attr('in', 'SourceGraphic');
    });
}

function renderColumnHeader(ctx, group, c) {
  const column = ctx.columns[c];
  const columnLabel = ctx.columnLabels[c];

  let sel = cc.materialize(group, 'text', 'plg-toolbar-col-text', [0]);
  sel.enter
    .attr('y', topToolbarTextY);

  const txt = sel.all;

  cc.materialize(txt, 'tspan', 'plg-toolbar-icon plg-toolbar-left', [0])
    .enter
    .html('\uf0d9 &thinsp;');                     // caret left

  cc.materialize(txt, 'tspan', 'plg-toolbar-icon plg-toolbar-sort', [c])
    .all
    .html(sortIcons[column.sort] + ' &thinsp;')   // sort
    .on('click.plg', ctx.onSortColumn); // eslint-disable-line memoryleaks

  const labels = cc.materialize(txt, 'tspan', 'plg-toolbar-col-label', [0]).all;
  if (ctx.columnLabels.length) { labels.text(columnLabel); }

  sel = cc.materialize(txt, 'tspan', 'plg-toolbar-icon plg-toolbar-hide', [c]);
  sel.enter
    .html('&thinsp; \uf00d');                     // cross
  sel.all
    .on('click.plg', ctx.onHideColumn); // eslint-disable-line memoryleaks

  cc.materialize(txt, 'tspan', 'plg-toolbar-icon plg-toolbar-right', [0])
    .enter
    .html('&thinsp; \uf0da');                     // caret left
}

function applyColumnLabels(ctx) {
  const { columns, columnLabels } = ctx;
  const truncateText = truncateTextAtWidth(ctx.xStep - 2 * fontSize);

  d3.select(ctx.svg)
    .selectAll('tspan.plg-toolbar-col-label').data(columns)
    .each(function (col, c) {
      this.textContent = col.label;

      truncateText(this);
      columnLabels[c] = this.textContent;
    });
}

function renderTopToolbar(ctx, group) {
  const colDragging = !!ctx.colDrag;
  const columnsMap = colDragging ? ctx.colDrag.columnsMap : ctx.columnsMap;

  group.classed('plg-dragging', colDragging);

  const topToolbarCol = cc.materialize(group, 'g', 'plg-top-toolbar-column', columnsMap)
    .all
    .classed('plg-hovered', dst => dst === ctx.view.hover.c)
    .classed('plg-dragged', (dst, c) => colDragging && c === ctx.colDrag.dstIdx)
    .attr('transform', cc.translator(ctx.refSizeX, columnsMap.length))
    .each(function (dst) { renderColumnHeader(ctx, d3.select(this), dst); })
    .datum((dst, c) => c)
    .call(d3.drag()
      .container(ctx.svg)
      .filter(ctx.colDragFilter)
      .on('start.plg', ctx.colDragStart) // eslint-disable-line memoryleaks
      .on('drag.plg', ctx.colDragMove) // eslint-disable-line memoryleaks
      .on('end.plg', ctx.colDragEnd)); // eslint-disable-line memoryleaks

  if (!ctx.onColumnTipHtmlRequest) { return; }

  topToolbarCol
    .on('mousemove.plg', ctx.colHeadTipShow) // eslint-disable-line memoryleaks
    .on('mouseleave.plg', ctx.colHeadTipHide); // eslint-disable-line memoryleaks
}


function renderAxis(ctx, group) {
  const columns = ctx.columns;
  const colFilters = ctx.view.colFilters;

  cc.materialize(group, 'g', 'plg-column', columns)
    .all
    .attr('transform', cc.translator(ctx.refSizeX, columns.length))
    .each(function (column, c) {
      const columnGroup = d3.select(this);

      let sel = cc.materialize(columnGroup, 'line', 'plg-column-line', [0]);
      sel.enter
        .attr('y1', 0)
        .attr('x1', 0).attr('x2', 0);
      sel.all
        .attr('y2', ctx.refSizeY);


      // Min-Max labels

      const colRange = ctx.colRanges[c];
      const filter = colFilters[column.id];
      const scale = ctx.scales.columns[c];

      let filterType, v0, v1, s0, s1;       // eslint-disable-line  one-var
      if (filter) {
        filterType = filter.type;
        v0 = filter.v0;
        v1 = filter.v1;
        s0 = filter.s0;
        s1 = filter.s1;
      } else {
        filterType = 'none';
        v0 = 0;
        v1 = 1;
        s0 = cc.lowerSampleInValueRange(column.type, scale, v0, v1);
        s1 = cc.upperSampleInValueRange(column.type, scale, v0, v1);
      }

      sel = cc.materialize(columnGroup, 'text', 'plg-column-top-text',
          labelPositionsByFilter[filterType].top, _.constant(0));
      sel.enter
        .attr('x', 0)
        .attr('font-size', fontSize);
      sel.all
        .attr('y', lPos => valueToY(ctx, v1) - lPos)
        .text(cc.sampleToShortText(column, colRange, s1));

      sel = cc.materialize(columnGroup, 'text', 'plg-column-bot-text',
          labelPositionsByFilter[filterType].bottom, _.constant(0));
      sel.enter
        .attr('x', 0)
        .attr('font-size', fontSize);
      sel.all
        .attr('y', lPos => valueToY(ctx, v0) + fontUpShift + lPos)
        .text(cc.sampleToShortText(column, colRange, s0));


      // Column ticks

      const yScale = ctx.yScales[c];
      const ticks = filter
        ? []
        : cc.columnTicks(column, scale, ctx.refSizeY, ctx.ticksDistance);

      if (ticks[0] === s0) { ticks.shift(); }
      if (ticks[ticks.length - 1] === s1) { ticks.pop(); }

      sel = cc.materialize(columnGroup, 'line', 'plg-column-tick-line', ticks);
      sel.enter
        .attr('x1', 0).attr('x2', -4);
      sel.all
        .attr('y1', yScale).attr('y2', yScale);

      sel = cc.materialize(columnGroup, 'text', 'plg-column-tick-text', ticks);
      sel.enter
        .attr('x', -5)
        .attr('dy', '0.25em');
      sel.all
        .attr('y', yScale)
        .text(s => cc.sampleToShortText(column, colRange, s));
    });
}


function splineGeometry(ctx, cStart, cEnd) {
  const xScale = ctx.xScale;
  const yScales = ctx.yScales;

  return function d(line) {
    if (!line) { return ''; }

    let c0 = cStart;

    let y0 = line[ctx.visibleToAll[c0]];
    if (y0 !== null) { y0 = yScales[c0](y0); }

    let move = true;

    let res = '';

    for (let c1 = c0 + 1, y1; c1 < cEnd; c0 = c1, y0 = y1, ++c1) {
      y1 = line[ctx.visibleToAll[c1]];
      if (y1 === null) { move = true; continue; }

      y1 = yScales[c1](y1);
      if (y0 === null) { move = true; continue; }

      if (move) { res += ` M ${xScale(c0)},${y0} C`; move = false; }

      res += ` ${xScale(c0 + splineCP)},${y0} ${xScale(c1 - splineCP)},${y1} ${xScale(c1)},${y1}`;
    }

    return res;
  };
}

function renderSplinesSvg(ctx, group) {
  const filteredLines = ctx.filtered.records;
  const cCount = ctx.columns.length;

  if (cCount < 2) { return; }

  const outGroup = cc.materialize(group, 'g', 'plg-filtered-out', [0]).all;
  const inGroup = cc.materialize(group, 'g', 'plg-filtered-in', [0]).all;

  const filteredOutLines = ctx.showFilteredOutRecords ? filteredLines[1] : [];
  const geometry = splineGeometry(ctx, 0, cCount);

  cc.materialize(outGroup, 'path', '', filteredOutLines)
    .all
    .attr('d', geometry);

  cc.materialize(inGroup, 'path', '', filteredLines[0])
    .all
    .attr('d', geometry)
    .attr('stroke', ctx.colors.record);
}

function drawSplineFactory(ctx, canvas, cStart, cEnd) {
  const xScale = ctx.xScale;
  const yScales = ctx.yScales;

  return function draw(line) {
    let c0 = cStart;

    let y0 = line[ctx.visibleToAll[c0]];
    if (y0 !== null) { y0 = yScales[c0](y0); }

    let move = true;

    canvas.beginPath();

    for (let c1 = c0 + 1, y1; c1 < cEnd; c0 = c1, y0 = y1, ++c1) {
      y1 = line[ctx.visibleToAll[c1]];
      if (y1 === null) { move = true; continue; }

      y1 = yScales[c1](y1);
      if (y0 === null) { move = true; continue; }

      if (move) { canvas.moveTo(xScale(c0), y0); move = false; }

      canvas.bezierCurveTo(
        xScale(c0 + splineCP), y0,
        xScale(c1 - splineCP), y1,
        xScale(c1), y1);
    }

    canvas.stroke();
  };
}

function renderSplinesCanvas(ctx) {
  const canvas = ctx.canvas.getContext('2d');

  canvas.setTransform(1, 0, 0, 1, 0, 0);
  canvas.clearRect(0, 0, ctx.sizeX, ctx.sizeY);

  canvas.translate(-ctx.viewBox[0], -ctx.viewBox[1]);
  canvas.lineWidth = 1;

  const filteredLines = ctx.filtered.records;
  const cCount = ctx.columns.length;

  if (cCount < 2) { return; }

  const drawSpline = drawSplineFactory(ctx, canvas, 0, cCount);

  if (ctx.showFilteredOutRecords) {
    canvas.strokeStyle = 'rgb(230,230,230)';
    _.forEach(filteredLines[1], drawSpline);
  }

  _.forEach(filteredLines[0], function drawColoredSpline(line) {
    canvas.strokeStyle = ctx.colors.record(line);
    drawSpline(line);
  });
}

function hoverText(ctx, line) {
  return `Count: ${line[line.length - 1]}`;
}

function renderHighlight(ctx, group, highlight, mPos) {
  const { lines, colors, noTipsAfter = Number.MAX_VALUE } = highlight;

  if (!lines.length) {
    group.selectAll('.plg-line-hover').remove();
    group.selectAll('.plg-line-hover-txcount').remove();
    group.selectAll('.plg-line-hover-values').remove();

    return;
  }

  const columns = ctx.columns;
  const xScale = ctx.xScale;
  const cCount = ctx.columns.length;


  cc.materialize(group, 'path', 'plg-line-hover', lines)
    .all
    .attr('d', splineGeometry(ctx, 0, cCount))
    .attr('stroke', highlight.colors.highlight || 'black');

  if (lines.length > highlight.noTipsAfter) {
    group.selectAll('.plg-line-hover-txcount').remove();
    group.selectAll('.plg-line-hover-values').remove();

    return;
  }

  const textDisplaceUp = -1.25 * fontSize;
  const textDisplaceDn = 2 * fontSize;
  const refSizeYHalf = ctx.refSizeY / 2;

  // NOTE: Text displacements are inverted for bucket count labels, to avoid
  //       sharing space with column value labels

  const lineIndexes = _(0).range(cCount)
    .map(c => ({ c, l: ctx.visibleToAll[c] }))
    .value();

  cc.materialize(group, 'g', 'plg-line-hover-values', lines)
    .all
    .each(function (line) {
      const g = d3.select(this);

      let sel = cc.materialize(g, 'text', 'plg-line-hover-text', lineIndexes);
      sel.enter
        .attr('font-size', fontSize)
        .attr('filter', highlight.backgroundUrl);
      sel.all
        .attr('x', d => xScale(d.c))
        .attr('y', d => {
          let y = line[d.l];
          if (y === null) { return refSizeYHalf; }

          y = ctx.yScales[d.c](y);
          return y + (y < refSizeYHalf ? textDisplaceDn : textDisplaceUp);
        })
        .text(d => cc.sampleToText(
          ctx.columns[d.c].type, ctx.colRanges[d.c], line[d.l]));
    });

  if (mPos) {
    let sel = cc.materialize(group, 'text', 'plg-line-hover-txcount', lines);
    sel.enter
      .attr('font-size', fontSize)
      .attr('filter', highlight.backgroundUrl);
    sel.all
      .attr('x', mPos[0])
      .attr('y', line => {
        const y = xToLineY(ctx, line, mPos[0]);
        return y + (y < refSizeYHalf ? textDisplaceUp : textDisplaceDn);
      })
      .text(line => hoverText(ctx, line));
  } else {
    group.selectAll('.plg-line-hover-txcount').remove();
  }
}

function renderHighlights(ctx, group, mPos) {
  const { highlights } = ctx;

  cc.materialize(group, 'g', 'plg-highlight', highlights, hl => hl.id)
    .all
    .each(function (hl) {
      const highlight = d3.select(this);
      renderHighlight(ctx, highlight, hl, mPos);
    })
    .order();
}

function renderFilters(ctx, group) {
  const xScale = ctx.xScale;
  const columns = ctx.columns;
  const colFilters = ctx.view.colFilters;


  let sel = cc.materialize(group, 'rect', 'plg-filter-zone', columns);
  sel.enter
    .attr('y', 0)
    .attr('width', filterRangeWidth);
  sel.all
    .attr('x', (col, c) => xScale(c) - filterRangeWidthHalf)
    .attr('height', ctx.refSizeY);

  const colIndices = _(columns.length).range()
    .filter(c => colFilters[columns[c].id])
    .value();

  cc.materialize(group, 'g', 'plg-filter', colIndices)
    .all
    .each(function (c) {
      const filterGroup = d3.select(this);
      const filter = ctx.view.colFilters[columns[c].id];

      const x = xScale(c);
      const y0 = valueToY(ctx, filter.v1);    // Scale inversion
      const y1 = valueToY(ctx, filter.v0);

      renderFilterRange(ctx, filterGroup, filter, c, x, y0, y1);
      renderFilterValue(ctx, filterGroup, filter, c, x, y0);
    });
}

function renderFilterRange(ctx, group, filter, c, x, y0, y1) {
  const items = (filter.type === 'range') ? [c] : [];

  const x0 = x - filterRangeWidthHalf;
  const height = y1 - y0;
  const handlesHeight = Math.min(height, filterRangeHandlesHeight);


  let sel = cc.materialize(group, 'rect', 'plg-filter-range', items);
  sel.enter
    .attr('width', filterRangeWidth);
  sel.all
    .attr('y', y0).attr('x', x0)
    .attr('height', height)
    .call(d3.drag()
      .container(ctx.svg)
      .on('start.plg', ctx.filterDragStart) // eslint-disable-line memoryleaks
      .on('drag.plg', ctx.filterDragMove) // eslint-disable-line memoryleaks
      .on('end.plg', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'plg-filter-range-top', items);
  sel.enter
    .attr('width', filterRangeWidth);
  sel.all
    .attr('y', y0).attr('x', x0)
    .attr('height', handlesHeight)
    .call(d3.drag()
      .container(ctx.svg)
      .on('start.plg', ctx.filterTopStart) // eslint-disable-line memoryleaks
      .on('drag.plg', ctx.filterTopMove) // eslint-disable-line memoryleaks
      .on('end.plg', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'plg-filter-range-bot', items);
  sel.enter
    .attr('width', filterRangeWidth);
  sel.all
    .attr('x', x0)
    .attr('y', Math.max(y1 - filterRangeHandlesHeight, y0))
    .attr('height', handlesHeight)
    .call(d3.drag()
      .container(ctx.svg)
      .on('start.plg', ctx.filterBotStart) // eslint-disable-line memoryleaks
      .on('drag.plg', ctx.filterBotMove) // eslint-disable-line memoryleaks
      .on('end.plg', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'text', 'plg-filter-delete', items);
  sel.enter
    .text('\uf00d');
  sel.all
    .attr('x', x + filterRangeWidthHalf + filterIconsPad)
    .attr('y', y0 + filterIconsPad)
    .on('mouseup.plg', c => filterRemove(ctx, c)); // eslint-disable-line memoryleaks
}

function renderFilterValue(ctx, group, filter, c, x, y) {
  const filters = (filter.type === 'value') ? [filter] : [];

  cc.materialize(group, 'circle', 'plg-filter-value', filters)
    .all
    .attr('cx', x)
    .attr('cy', y)
    .attr('r', flt => flt.highlighted ? 1.5 * filterValueRadius : filterValueRadius)
    .on('mouseover.plg', flt => flt.highlighted = true) // eslint-disable-line memoryleaks
    .on('mouseout.plg', flt => flt.highlighted = false); // eslint-disable-line memoryleaks
}


function cleanUnusedSurface(ctx, state, contentGroup) {
  const { useCanvas } = ctx;
  const { didUseCanvas } = state;

  if (didUseCanvas !== null && useCanvas !== didUseCanvas) {
    if (didUseCanvas) {
      ctx.canvas.getContext('2d').clearRect(0, 0, ctx.sizeX, ctx.sizeY);
    } else {
      contentGroup.selectAll('*').remove();
    }
  }

  state.didUseCanvas = useCanvas;
}

function renderContentGroup(ctx, state) {
  let svg = d3.select(ctx.svg);
  let contentGroup = cc.materialize(svg, 'g', 'plg-content', [0]).all;

  if (ctx.useCanvas) {
    renderSplinesCanvas(ctx);
  } else {
    renderSplinesSvg(ctx, contentGroup);
  }

  cleanUnusedSurface(ctx, state, contentGroup);
}

function renderOverlayGroup(ctx) {
  const svg = d3.select(ctx.svg);
  const overlayGroup = cc.materialize(svg, 'g', 'plg-overlay', [0]).all;

  const topToolbar = cc.materialize(overlayGroup, 'g', 'plg-top-toolbar', [0]).all;
  const axisGroup = cc.materialize(overlayGroup, 'g', 'plg-axis-group', [0]).all;
  const filtersGroup = cc.materialize(overlayGroup, 'g', 'plg-filters-group', [0]).all;
  const hoverGroup = cc.materialize(overlayGroup, 'g', 'plg-hover-group', [0]).all;

  // IMPORTANT: Don't move below, must be done *before* svg changes
  const mPos = ctx.view.hover.event && cc.mousePos(ctx, ctx.view.hover.event);

  renderTopToolbar(ctx, topToolbar);
  renderAxis(ctx, axisGroup);
  renderFilters(ctx, filtersGroup);
  renderHighlights(ctx, hoverGroup, mPos);
}


// State

function buildSizes(ctx) {
  const padding = {
    left: cc.padLeft, right: cc.padRight,
    top: padTop, bottom: padBottom
  };

  const result = cc.buildStripSizes(ctx, padding);

  result.hoverRect = [
    result.contentRect[0] - filterRangeWidthHalf,
    -textLineHeight,
    result.contentRect[2] + filterRangeWidthHalf,
    result.viewRect[3]
  ];

  _.assign(ctx, result);
}

function buildScales(ctx) {
  ctx.xScale = d3.scaleLinear()
    .domain([0, ctx.columns.length - 1])
    .range([0, ctx.refSizeX]);

  ctx.yScales = _.map(ctx.scales.columns, scale => function (val) {
    return valueToY(ctx, scale(val));
  });
}

function buildHighlights(ctx) {
  ctx.highlights = cc.buildHighlights(ctx)
    .map(hl => {
      hl.backgroundId = `${ctx.uuid}-bg-${hl.id}`;
      hl.backgroundUrl = `url(#${hl.backgroundId})`;

      return hl;
    });

  ctx.textBackgroundUrl = ctx.highlights[ctx.highlights.length - 1].backgroundUrl;
}

function initRenderContext(base, ctx) {
  ctx = _.assign({
    viewChanges: {},
    filterDrag: null,
    colDrag: null,
    columnsMap: _.range(0, ctx.columns.length),
    columnLabels: []
  }, base, ctx);

  buildSizes(ctx);
  buildScales(ctx);
  buildHighlights(ctx);

  return ctx;
}


// Exports

export default class ParallelChart {
  constructor(opts) {
    this.baseCtx = opts;
    this.ctx = null;

    this.uuid = `plg-${_.random(0, 1e12)}`;
    this.didUseCanvas = null;
  }

  render(ctx) {
    ctx = initRenderContext(this.baseCtx, ctx);
    if (ctx.columns.length < 2) { return this.reset(); }

    this.ctx = ctx;

    d3.select(ctx.svg)
      .attr('width', ctx.sizeX)
      .attr('height', ctx.sizeY)
      .attr('viewBox', ctx.viewBox.join(' '));

    ctx.canvas.width = ctx.sizeX;
    ctx.canvas.height = ctx.sizeY;
    ctx.canvas.style.width = ctx.sizeX + 'px';
    ctx.canvas.style.height = ctx.sizeY + 'px';

    initMouse(ctx);

    renderDefsGroup(ctx);
    renderContentGroup(ctx, this);
    renderOverlayGroup(ctx);

    // Defer to let text nodes be rendered *before* setting their text content.
    // This is required because text truncation requires that the text node is already
    // rendered to calculate its actual width.
    _.defer(applyColumnLabels, ctx);
  }

  refresh(viewChanges) {
    const { ctx } = this;
    if (!ctx) { return; }

    let refreshContent = false;
    let refreshOverlay = false;

    _.chain(viewChanges).keys().forEach(viewData => {
      switch (viewData) {
        case 'colFilters':
          refreshContent = true;

        case 'hover':
          refreshOverlay = true;
      }
    }).commit();

    if (refreshContent) { renderContentGroup(ctx, this); }
    if (refreshOverlay) { renderOverlayGroup(ctx); }
  }

  reset() {
    const { ctx } = this;
    if (!ctx) { return; }

    if (this.didUseCanvas) {
      ctx.canvas.getContext('2d').clearRect(0, 0, ctx.sizeX, ctx.sizeY);
    }

    d3.select(ctx.svg)
      .on('.plg .drag', null) // eslint-disable-line memoryleaks
      .selectAll('*')
      .remove();

    this.ctx = null;
  }
};

