import moment from 'moment';
import _ from 'lodash';
import angular from 'angular';
import { FilterManagerProvider } from 'ui/filter_manager';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { RegistryFieldFormatsProvider } from 'ui/registry/field_formats';
import 'ui/timefilter';
import { uiModules } from 'ui/modules';
import { getSearchParameters} from 'plugins/scatterplot_vis/helpers/get_search_parameters_helper';
import { ConfigureOptionsProvider } from 'plugins/scatterplot_vis/helpers/configure_options_helper';

const module = uiModules.get('kibana/scatterplot_vis');

module.controller('ScatterPlotVisController', function ($scope, $element, es,
  Private, timefilter, getAppState, createNotifier, savedSearches) {
  const filterManager = Private(FilterManagerProvider);
  const queryFilter = Private(FilterBarQueryFilterProvider);
  const configureOptions = Private(ConfigureOptionsProvider);
  const notify = createNotifier({
    location: 'Scatterplot'
  });

  // Performs significant term bucket
  $scope.mapSTBucket = function (bucket) {
    let colorValue = null;
    if ($scope.options.color) {
      colorValue = $scope.options.color;
    }

    return {
      x: +bucket.bg_count,
      y: +bucket.doc_count,
      _id: bucket[$scope.options.labelFieldName],
      label: bucket[$scope.options.labelFieldName],
      size: +bucket.score,
      color: colorValue
    };
  };

  // Performs any aggregator bucket
  $scope.mapAABucket = function (bucket) {
    let xval = null;
    if ($scope.options.aaXmetric.name === 'count') {
      xval = +bucket.doc_count;
    } else {
      xval = +bucket['1'].value;
    }

    let yval = null;
    if ($scope.options.aaYmetric.name === 'count') {
      yval = +bucket.doc_count;
    } else {
      yval = +bucket['3'].value;
    }

    let colorValue = null;
    if ($scope.options.color) {
      colorValue = $scope.options.color;
    }

    return {
      x: xval,
      y: yval,
      _id: bucket[$scope.options.labelFieldName],
      label: bucket[$scope.options.labelFieldName],
      size: null,
      color: colorValue
    };
  };

  // Performs filter aggreation data
  $scope.mapFABucket = function (bucket) {
    let xval = null;
    let yval = null;
    if ($scope.options.faMetric.name === 'count') {
      xval = +bucket['3'].buckets.x.doc_count;
      yval = +bucket['3'].buckets.y.doc_count;
    } else {
      xval = +bucket['3'].buckets.x['1'].value;
      yval = +bucket['3'].buckets.y['1'].value;
    }

    let colorValue = null;
    if ($scope.options.color) {
      colorValue = $scope.options.color;
    }

    return {
      x: xval,
      y: yval,
      _id: bucket[$scope.options.labelFieldName],
      label: bucket[$scope.options.labelFieldName],
      size: null,
      color: colorValue
    };
  };

  // Performs straight from data search results
  $scope.mapHits = function (hit) {
    function getFieldValue(field) {
      if (_.isUndefined(field) || _.isNull(field)) {
        return null;
      }
      if (_.isArray(field)) {
        return field[0];
      }
      return field;
    }
    function getField(fieldName) {
      let field = null;
      if (hit._source && hit._source[fieldName]) {
        field = hit._source[fieldName];
      } else if (hit.fields) {
        field = hit.fields[fieldName];
      }
      return getFieldValue(field);
    }

    if (!hit._source && !hit.fields) return null;

    let xvalue = null;
    let xFieldObj = getField($scope.options.xFieldName);
    if (!_.isNull(xFieldObj)) {
      if ($scope.options.xField.type === 'date') {
        xvalue = moment(xFieldObj).toDate();
      } else if ($scope.options.xField.type === 'string') {
        xvalue = xFieldObj;
      }
      else {
        xvalue = +xFieldObj;
      }
    }
    if (_.isNull(xvalue)) return null;

    let yvalue = null;
    let yFieldObj = getField($scope.options.yFieldName);

    if (!_.isNull(yFieldObj)) {
      if ($scope.options.yField.type === 'date') {
        yvalue = moment(yFieldObj).toDate();
      } else if ($scope.options.yField.type === 'string') {
        yvalue = yFieldObj;
      } else {
        yvalue = +yFieldObj;
      }
    }
    if (_.isNull(yvalue)) return null;

    let labelValue = null;
    let labelValueObj = getField($scope.options.labelFieldName);

    if ($scope.options.labelFieldName !== '' && labelValueObj) {
      labelValue = labelValueObj;
    }

    let dotSizeValue = null;
    let dotSizeValueObj = getField($scope.options.dotSizeFieldName);

    if ($scope.options.dotSizeFieldName !== '' && dotSizeValueObj) {
      dotSizeValue = +dotSizeValueObj;
    }

    let xJitterValue = null;
    let xJitterValueObj = getField($scope.options.xJitterFieldName);

    if ($scope.options.xJitterFieldName !== '' && xJitterValueObj) {
      if ($scope.options.xJitterField.type === 'date') {
        xJitterValue = moment(xJitterValueObj).toDate();
      } else if ($scope.options.xJitterField.type === 'string') {
        xJitterValue = xJitterValueObj;
      } else {
        xJitterValue = +xJitterValueObj;
      }
    }

    let colorValue = null;
    let colorValueObj = getField($scope.options.colorFieldName);

    if ($scope.options.colorFieldName !== '') {
      if (colorValueObj) {
        if ($scope.options.colorField.type === 'date') {
          colorValue = moment(colorValueObj).toDate();
        } else {
          colorValue = colorValueObj;
        }
      }
    } else if ($scope.options.color) {
      colorValue = $scope.options.color;
    }

    return {
      x: xvalue,
      y: yvalue,
      _id: hit._id,
      label: labelValue,
      size: dotSizeValue,
      color: colorValue,
      xJitter: xJitterValue
    };
  };

  function executeSearch(parameters) {
    return es.search(parameters)
      .then((resp) => {
        let ret = {
          data: [],
          totalHits: resp.hits.total
        };
        if ($scope.options.aggMode === 'Straight data') {
          resp.hits.hits.forEach((hit) => {
            let val = $scope.mapHits(hit);
            if (!_.isNull(val)) ret.data.push(val);
          });
        } else {
          if (resp.aggregations) {
            resp.aggregations['2'].buckets.forEach((bucket) => {
              let val;
              switch ($scope.options.aggMode) {
                case 'Significant term data':
                  val = $scope.mapSTBucket(bucket);
                  break;
                case 'Any aggregator data':
                  val = $scope.mapAABucket(bucket);
                  break;
                case 'Filtered aggregator data':
                  val = $scope.mapFABucket(bucket);
                  break;
              }
              if (!_.isNull(val)) ret.data.push(val);
            });
          }
        }

        return ret;
      });
  };

  $scope.search = function () {
    configureOptions($scope.options);
    const appState = getAppState();
    const appStateQuery = appState.query;
    const appStateFilters = queryFilter.getAppFilters();

    if ($scope.vis.savedSearchId) {
      return savedSearches.get($scope.vis.savedSearchId)
        .then(savedSearch => {
          const searchSource = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
          const savedSearchFilters = searchSource.filter;
          const savedSearchQuery = searchSource.query;
          return executeSearch(
            getSearchParameters($scope, timefilter, appStateFilters, appStateQuery, savedSearchFilters, savedSearchQuery)
          );
        })
        .catch(notify.error);
    }

    return executeSearch(
      getSearchParameters($scope, timefilter, appStateFilters, appStateQuery)
    )
      .catch(notify.error);
  };

  $scope.initParams = function (params) {
    let vps = params;
    let indexFields = $scope.vis.indexPattern.fields;

    // parameters
    $scope.options = {
      xFieldName: vps.xFieldName,
      yFieldName: vps.yFieldName,
      xJitterFieldName: vps.xJitterFieldName,
      labelFieldName: vps.labelFieldName,
      colorFieldName: vps.colorFieldName,
      dotSizeFieldName: vps.dotSizeFieldName,
      stFieldName: vps.stFieldName,
      xField: indexFields.byName[vps.xFieldName],
      yField: indexFields.byName[vps.yFieldName],
      xJitterField: indexFields.byName[vps.xJitterFieldName],
      xJitterScale: vps.xJitterScale,
      labelField: indexFields.byName[vps.labelFieldName],
      color: vps.color,
      colorField: indexFields.byName[vps.colorFieldName],
      dotSizeField: indexFields.byName[vps.dotSizeFieldName],
      dotSize: +vps.dotSize === 0 ? 8 : +vps.dotSize,
      dotSizeScale: vps.dotSizeScale,
      labelEnabled: vps.labelEnabled,
      labelHoverEnabled: vps.labelHoverEnabled,
      sizeParam: vps.sizeParam,
      xAxisLabel: vps.xAxisLabel,
      yAxisLabel: vps.yAxisLabel,
      xAxisScale: vps.xAxisScale,
      yAxisScale: vps.yAxisScale,
      aggMode: vps.aggMode,
      stField: indexFields.byName[vps.stFieldName],
      stSize: vps.stSize,
      shapeOpacity: vps.shapeOpacity,
      faBucket: vps.faBucket,
      faFilterX: vps.faFilterX,
      faFilterY: vps.faFilterY,
      faMetric: vps.faMetric,
      aaBucket: vps.aaBucket,
      aaXmetric: vps.aaXmetric,
      aaYmetric: vps.aaYmetric
    };

    let options = $scope.options;

    let fieldFormats = Private(RegistryFieldFormatsProvider);

    if (options.xField && !options.xField.format) {
      options.xField.format = fieldFormats.getDefaultInstance('string');
    }

    if (options.yField && !options.yField.format) {
      options.yField.format = fieldFormats.getDefaultInstance('string');
    }

    if (!options.faBucket || options.faBucket === '') {
      options.faBucket = {
        name: '',
        fieldName: ''
      };
    }
    options.faBucket.field = indexFields.byName[options.faBucket.fieldName];

    if (!options.faMetric || options.faMetric === '') {
      options.faMetric = {
        name: '',
        fieldName: ''
      };
    }
    options.faMetric.field = indexFields.byName[options.faMetric.fieldName];

    if (!options.aaBucket || options.aaBucket === '') {
      options.aaBucket = {
        name: '',
        fieldName: ''
      };
    }
    options.aaBucket.field = indexFields.byName[options.aaBucket.fieldName];

    if (!options.aaXmetric || options.aaXmetric === '') {
      options.aaXmetric = {
        name: '',
        fieldName: ''
      };
    }
    options.aaXmetric.field = indexFields.byName[options.aaXmetric.fieldName];

    if (!options.aaYmetric || options.aaYmetric === '') {
      options.aaYmetric = {
        name: '',
        fieldName: ''
      };
    }
    options.aaYmetric.field = indexFields.byName[options.aaYmetric.fieldName];

    $scope.options.dataSize = $scope.options.sizeParam ? $scope.options.sizeParam : 0;
  };

  $scope.loadData = function () {
    return $scope.search().then(function (data) {
      $scope.data = data;
    });
  };

  $scope.reloadData = function () {
    $scope.initParams($scope.vis.params);
    $scope.loadData();
  };

  $scope.$watch('vis.params', function () {
    $scope.reloadData();
  });

  $scope.$watch('esResponse', function (resp) {
    if (resp) {
      $scope.reloadData();
    }
  });

  $scope.$on('scatterplot_refresh', function () {
    $scope.reloadData();
  });

  $scope.addFilter = function (fieldName, id) {
    function addRange(range, fieldName, fieldType) {
      let indexFields = $scope.vis.indexPattern.fields;
      if (indexFields.length !== 0 && indexFields.byName) {
        let field = indexFields.byName[fieldName];
        let filter;
        if (fieldType === 'date') {
          timefilter.time.from = range[0];
          timefilter.time.to = range[1];
        }
        else if (field.scripted) {
          if (fieldType === 'string' || fieldType === 'boolean') {
            let filter = { meta: { negate: false, index: $scope.vis.indexPattern.id },
            query: { bool: { should: [] } } };
            let or = [];
            let orValues = [];
            const defs = 'boolean compare(Supplier s, def v)';
            _.forEach(range, value => {
              let term = {
                script: {
                  script: {
                    inline: defs + ' {return s.get() == v;}compare(() -> { ' + field.script + ' }, params.value);',
                    lang: field.lang,
                    params: {
                      value: value
                    }
                  }
                }
              };
              or.push(term);
              orValues.push(value);
            });
            if (or.length > 0) {
              filter.meta.alias = fieldName + ': {' + orValues + '}';
              filter.query.bool.should = or;
              queryFilter.addFilters(filter);
            }
          } else {
            filter = { meta: { negate: false, index: $scope.vis.indexPattern.id }, query: { script: { } } };
            const defs = 'boolean gte(Supplier s, def v) {return s.get() >= v} boolean lte(Supplier s, def v)';
            let script = {
              script: {
                inline: defs + ' {return s.get() <= v}gte(() -> { ' + field.script + ' }, params.gte) ' +
                '&& lte(() -> { ' + field.script + ' }, params.lte)',
                lang: field.lang,
                params: {
                  gte: range[0],
                  lte: range[range.length - 1],
                  value: '>=' + range[0] + ' <=' + range[range.length - 1]
                }
              }
            };
            filter.meta.alias = fieldName + ': [' + script.script.params.gte + ', ' + script.script.params.lte + ']';
            _.merge(filter.query.script, script);
          }
        } else if (fieldType === 'string' || fieldType === 'boolean') {
          let docFieldName = fieldName.replace('.raw', '');
          let filter = { meta: { negate: false, index: $scope.vis.indexPattern.id },
          query: { bool: { should: [] } } };
          let or = [];
          let orValues = [];
          _.forEach(range, value => {
            let term = { term: { } };
            term.term[docFieldName] = value;
            or.push(term);
            orValues.push(value);
          });
          if (or.length > 0) {
            filter.meta.alias = fieldName + ': {' + orValues + '}';
            filter.query.bool.should = or;
            queryFilter.addFilters(filter);
          }
        } else {
          let docFieldName = fieldName.replace('.raw', '');
          filter = { meta: { negate: false, index: $scope.vis.indexPattern.id }, query: { range: { } } };
          let params = {
            gte: range[0],
            lte: range[range.length - 1]
          };
          filter.meta.alias = fieldName + ': [' + params.gte + ', ' + params.lte + ']';
          filter.query.range[docFieldName] = params;
        }
        queryFilter.addFilters(filter);
      }
    }
    if (id && Array.isArray(id)) {
      addRange(id[0], $scope.options.xFieldName, $scope.options.xField.type);
      addRange(id[1], $scope.options.yFieldName, $scope.options.yField.type);
    } else {
      filterManager.add(fieldName, id, '+', $scope.vis.indexPattern.id);
    }
  };

  $scope.$on('scatterplot_select', function (e, id) {
    if ($scope.options.aggMode === 'Straight data') {
      $scope.addFilter('_id', id);
    } else if ($scope.options.aggMode === 'Significant term data') {
      $scope.addFilter($scope.options.stFieldName, id);
    } else if ($scope.options.aggMode === 'Filtered aggregator data') {
      $scope.addFilter($scope.options.faBucket.fieldName, id);
    } else if ($scope.options.aggMode === 'Any aggregator data') {
      $scope.addFilter($scope.options.aaBucket.fieldName, id);
    }
  });

  function onSizeChange() {
    return {
      width: $element.parent().width(),
      height: $element.parent().height()
    };
  }

  let removeWatch = $scope.$watch(onSizeChange, _.debounce(function () {
    $scope.$broadcast('scatterplot:redraw', {});
  }, 250), true);

  $scope.$on('$destroy', function () {
    removeWatch();
  });

  // Re-render if the window is resized
  angular.element(window).bind('resize', function () {
    $scope.$broadcast('scatterplot:redraw', {});
  });

  $scope.initParams($scope.vis.params);
  $scope.loadData();

});
