'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _dump = require('./_dump');

var _dump2 = _interopRequireDefault(_dump);

var _lodash = require('lodash');

var _bluebird = require('bluebird');

var _bluebird2 = _interopRequireDefault(_bluebird);

var _fs = require('fs');

var _mkdirp = require('mkdirp');

var _mkdirp2 = _interopRequireDefault(_mkdirp);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * BackupKibi writes into a file the data and mappings of the kibi index.
 * If the ACL plugin is present, it saves also the ACL index.
 *
 * @param config the investigate.yml config
 * @param backupDir the path to the folder to write to
 */
class BackupKibi {
  constructor(config, backupDir) {
    this._config = config;
    this._backupDir = backupDir;
    this._dump = new _dump2.default(config, backupDir);
  }

  async backup() {

    // Check if the directory exists and if it is empty or not
    const dirInfo = { missing: false, empty: false };
    try {
      const dirContent = (0, _fs.readdirSync)(this._backupDir);
      dirInfo.empty = dirContent.length === 0 ? true : false;
    } catch (error) {
      // If we don't have access throw the error
      if (error.code === 'EACCES') {
        throw error;
      }
      // if the directory does not exist it's ok, elasticdump will create it.
      if (error.code !== 'ENOENT') {
        process.stderr.write(`${error}\n`);
      } else {
        dirInfo.missing = true;
      }
    }

    if (!dirInfo.missing && !dirInfo.empty) {
      throw new Error(`Backup folder [${this._backupDir}] already exists and it's not empty.`);
    }
    if (dirInfo.missing) {
      await _bluebird2.default.fromNode(cb => (0, _mkdirp2.default)(this._backupDir, cb));
    }

    const kibiIndex = (0, _lodash.get)(this._config, 'kibana.index', '.siren');

    await this._dump.fromElasticsearchToFile(kibiIndex, 'data');
    await this._dump.fromElasticsearchToFile(kibiIndex, 'mapping');

    const isAccessControPluginEnabled = (0, _lodash.get)(this._config, 'investigate_access_control.enabled');
    const isACLEnabled = (0, _lodash.get)(this._config, 'investigate_access_control.acl.enabled');
    if (isAccessControPluginEnabled && isACLEnabled) {
      const aclIndex = (0, _lodash.get)(this._config, 'investigate_access_control.acl.index', '.sirenaccess');
      await this._dump.fromElasticsearchToFile(aclIndex, 'data');
      await this._dump.fromElasticsearchToFile(aclIndex, 'mapping');
    }
  }
}
exports.default = BackupKibi;
module.exports = exports['default'];
