'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fs = require('fs');

var _lodash = require('lodash');

var _path = require('path');

/**
 * RestoreKibiPreprocess checks the backup and preprocess it if needed
 * to ensure compatibility with the supported ES version.
 */
class RestoreKibiPreprocess {

  /**
   * @param config the investigate.yml config
   * @param backupDir the folder with the backuped indices
   */
  constructor(config, backupDir) {
    this._config = config;
    this._backupDir = backupDir;
    this.kibiIndex = (0, _lodash.get)(this._config, 'kibana.index', '.siren');
  }

  _removeProps(obj, keys) {
    if (obj instanceof Array) {
      obj.forEach(item => {
        this._removeProps(item, keys);
      });
    } else if (typeof obj === 'object') {
      Object.getOwnPropertyNames(obj).forEach(key => {
        if (keys.indexOf(key) !== -1) {
          delete obj[key];
        } else {
          this._removeProps(obj[key], keys);
        }
      });
    }
  }

  /**
   *  Preprocess the mapping file.
   *  Returns true if the mapping has been processed, false otherwise.
   */
  _preprocessMapping() {
    const mappingFile = (0, _path.join)(this._backupDir, `mapping-${this.kibiIndex}.json`);
    const mappingString = (0, _fs.readFileSync)(mappingFile, 'utf8');

    const mapping = JSON.parse(mappingString);
    const processedMapping = (0, _lodash.cloneDeep)(mapping);

    // Remove all _all occurrences
    this._removeProps(processedMapping, '_all');

    // Remove all include_in_all occurrences
    this._removeProps(processedMapping, 'include_in_all');

    if (!(0, _lodash.isEqual)(mapping, processedMapping)) {
      // Write the processed file
      const processedFileName = `mapping-${this.kibiIndex}-upgraded.json`;
      const processedFile = (0, _path.join)(this._backupDir, processedFileName);
      (0, _fs.writeFileSync)(processedFile, JSON.stringify(processedMapping), 'utf8');

      process.stdout.write('To ensure compatibility with the current ES version the mapping file has been ' + 'preprocessed.\nThe upgraded mapping file has been saved as ' + processedFileName + '.\n\n');
      return true;
    }
    return false;
  }

  /**
   * Preprocess the dump file if needed. Returns the following object:
   * {
   *   mapping: true, // if the mapping file has been upgraded
   *   data: true     // if the data file has been upgaded
   * }
   */
  preprocess() {
    const mappingProcessed = this._preprocessMapping();
    return {
      mapping: mappingProcessed,
      data: false
    };
  }
}
exports.default = RestoreKibiPreprocess;
module.exports = exports['default'];
